/**
 * Copyright (c) 2016 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Shuai Li (CEA LIST) <shuai.li@cea.fr> - Initial API and implementation
 *   Van Cam Pham (CEA LIST) <vancam.pham@cea.fr> - Reverse implementation
 */
package org.eclipse.papyrus.designer.languages.cpp.reverse;

import com.google.common.collect.Iterables;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.cdt.core.dom.ast.IASTArrayModifier;
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTEqualsInitializer;
import org.eclipse.cdt.core.dom.ast.IASTExpression;
import org.eclipse.cdt.core.dom.ast.IASTInitializer;
import org.eclipse.cdt.core.dom.ast.IASTInitializerClause;
import org.eclipse.cdt.core.dom.ast.IASTName;
import org.eclipse.cdt.core.dom.ast.IASTNode;
import org.eclipse.cdt.core.dom.ast.IASTPointerOperator;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTArrayDeclarator;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTConstructorChainInitializer;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTFunctionDefinition;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTReferenceOperator;
import org.eclipse.cdt.core.model.CoreModel;
import org.eclipse.cdt.core.model.CoreModelUtil;
import org.eclipse.cdt.core.model.ICContainer;
import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.core.model.ICModel;
import org.eclipse.cdt.core.model.ICProject;
import org.eclipse.cdt.core.model.IEnumeration;
import org.eclipse.cdt.core.model.IInclude;
import org.eclipse.cdt.core.model.IMethod;
import org.eclipse.cdt.core.model.IMethodDeclaration;
import org.eclipse.cdt.core.model.IParent;
import org.eclipse.cdt.core.model.ISourceReference;
import org.eclipse.cdt.core.model.ISourceRoot;
import org.eclipse.cdt.core.model.IStructure;
import org.eclipse.cdt.core.model.IStructureDeclaration;
import org.eclipse.cdt.core.model.IStructureTemplate;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.cdt.core.model.ITypeDef;
import org.eclipse.cdt.core.model.IVariableDeclaration;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.papyrus.designer.languages.common.base.StdUriConstants;
import org.eclipse.papyrus.designer.languages.common.profile.Codegen.External;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Array;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Const;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Ptr;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Ref;
import org.eclipse.papyrus.designer.languages.cpp.profile.CppProfileResource;
import org.eclipse.papyrus.designer.languages.cpp.reverse.utils.ModelManagement;
import org.eclipse.papyrus.designer.languages.cpp.reverse.utils.RoundtripCppUtils;
import org.eclipse.papyrus.designer.uml.tools.utils.ElementUtils;
import org.eclipse.papyrus.designer.uml.tools.utils.PackageUtil;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.TypedElement;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.Usage;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Utility methods for the reverse
 */
@SuppressWarnings("all")
public class ReverseUtils {
  public static final Logger LOGGER = Logger.getLogger(ReverseCpp2Uml.class.getName());

  public static final String Cpp_LangID = "C++";

  public static final ICElement[] EMPTY_CHILDS = {};

  public static final String TEMPLATE_PARAMETER_SIGNATURE_NAME = "template_paremeter_signature";

  /**
   * Analyze a declaration and apply suitable stereotypes from the C/C++ profile
   * Used for operations and attributes
   * @param declarators a list of declarators
   * @param type the UML type of the element
   * @param typedElement the parameter or attribute that holds a type.
   */
  public static void analyzeDeclaration(final List<IASTDeclarator> declarators, final Type type, final TypedElement typedElement, final String langID) {
    for (final IASTDeclarator declarator : declarators) {
      ReverseUtils.analyzeDeclaration(declarator, type, typedElement, langID);
    }
  }

  /**
   * Analyze a declaration and apply suitable stereotypes from the C/C++ profile
   * Used for operations and attributes
   */
  public static void analyzeDeclaration(final IASTDeclarator declarator, final Type type, final TypedElement typedElement, final String langID) {
    try {
      if ((declarator instanceof ICPPASTArrayDeclarator)) {
        ICPPASTArrayDeclarator arrayDeclarator = ((ICPPASTArrayDeclarator) declarator);
        IASTArrayModifier[] arrays = arrayDeclarator.getArrayModifiers();
        final IASTArrayModifier[] _converted_arrays = (IASTArrayModifier[])arrays;
        int _size = ((List<IASTArrayModifier>)Conversions.doWrapArray(_converted_arrays)).size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          final Array array = StereotypeUtil.<Array>applyApp(typedElement, Array.class);
          array.setDefinition("");
          final IASTArrayModifier[] _converted_arrays_1 = (IASTArrayModifier[])arrays;
          final Consumer<IASTArrayModifier> _function = (IASTArrayModifier it) -> {
            final IASTExpression expr = it.getConstantExpression();
            if ((expr != null)) {
              String _definition = array.getDefinition();
              String _plus = (_definition + "[");
              String _string = expr.toString();
              String _plus_1 = (_plus + _string);
              String _plus_2 = (_plus_1 + "]");
              array.setDefinition(_plus_2);
            }
          };
          ((List<IASTArrayModifier>)Conversions.doWrapArray(_converted_arrays_1)).forEach(_function);
        }
      }
      IASTPointerOperator[] _pointerOperators = declarator.getPointerOperators();
      boolean _tripleNotEquals = (_pointerOperators != null);
      if (_tripleNotEquals) {
        boolean _isApplied = StereotypeUtil.isApplied(typedElement, Ptr.class);
        if (_isApplied) {
          Ptr _stereotypeApplication = UMLUtil.<Ptr>getStereotypeApplication(typedElement, Ptr.class);
          _stereotypeApplication.setDeclaration("");
        }
        IASTPointerOperator[] _pointerOperators_1 = declarator.getPointerOperators();
        for (final IASTPointerOperator pointerOperator : _pointerOperators_1) {
          if ((pointerOperator instanceof ICPPASTReferenceOperator)) {
            final ICPPASTReferenceOperator reference = ((ICPPASTReferenceOperator) pointerOperator);
            final Ref refStereo = StereotypeUtil.<Ref>applyApp(typedElement, Ref.class);
            refStereo.setDeclaration(reference.getSyntax().toString());
          } else {
            if ((pointerOperator instanceof IASTPointerOperator)) {
              final IASTPointerOperator pointer = ((IASTPointerOperator) pointerOperator);
              final Ref ptrStereo = StereotypeUtil.<Ref>applyApp(typedElement, Ref.class);
              if ((ptrStereo != null)) {
                String _declaration = ptrStereo.getDeclaration();
                String _string = pointer.getSyntax().toString();
                String _plus = (_declaration + _string);
                ptrStereo.setDeclaration(_plus);
              } else {
                System.err.println("Hallo???");
              }
            }
          }
        }
        boolean _isApplied_1 = StereotypeUtil.isApplied(typedElement, Ptr.class);
        if (_isApplied_1) {
          final Ptr ptrStereo_1 = UMLUtil.<Ptr>getStereotypeApplication(typedElement, Ptr.class);
          if (((ptrStereo_1.getDeclaration() != null) && ptrStereo_1.getDeclaration().trim().equals("*"))) {
            ptrStereo_1.setDeclaration(null);
          }
        }
        boolean _isApplied_2 = StereotypeUtil.isApplied(typedElement, Ref.class);
        if (_isApplied_2) {
          final Ref refStereo_1 = UMLUtil.<Ref>getStereotypeApplication(typedElement, Ref.class);
          if (((refStereo_1.getDeclaration() != null) && refStereo_1.getDeclaration().trim().equals("&"))) {
            refStereo_1.setDeclaration(null);
          }
        }
        Pattern pattern = Pattern.compile("(\\*)([\\s]*)(const)");
        Matcher matcher = pattern.matcher(declarator.getRawSignature());
        boolean _find = matcher.find();
        if (_find) {
          StereotypeUtil.<Const>applyApp(typedElement, Const.class);
        }
      }
      IASTInitializer initilizer = declarator.getInitializer();
      final Function1<ValueSpecification, Boolean> _function_1 = (ValueSpecification it) -> {
        return Boolean.valueOf(it.getName().equals("defaultValue"));
      };
      ValueSpecification valueExisting = IterableExtensions.<ValueSpecification>head(IterableExtensions.<ValueSpecification>filter(Iterables.<ValueSpecification>filter(typedElement.getOwnedElements(), ValueSpecification.class), _function_1));
      if ((valueExisting != null)) {
        valueExisting.destroy();
      }
      if ((initilizer != null)) {
        ValueSpecification vs = null;
        if ((typedElement instanceof Property)) {
          vs = ((Property) typedElement).createDefaultValue("defaultValue", ((Property)typedElement).getType(), 
            UMLPackage.Literals.OPAQUE_EXPRESSION);
        } else {
          if ((typedElement instanceof Parameter)) {
            vs = ((Parameter) typedElement).createDefaultValue("default", ((Parameter)typedElement).getType(), 
              UMLPackage.Literals.OPAQUE_EXPRESSION);
          }
        }
        if ((vs == null)) {
          return;
        }
        OpaqueExpression oe = ((OpaqueExpression) vs);
        oe.getLanguages().add(langID);
        if ((initilizer instanceof IASTEqualsInitializer)) {
          IASTEqualsInitializer equalsInitialiser = ((IASTEqualsInitializer) initilizer);
          IASTInitializerClause clause = equalsInitialiser.getInitializerClause();
          if ((clause != null)) {
            oe.getBodies().add(clause.getRawSignature());
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public static String getCppTypeName(final String name) {
    String trimName = name.trim();
    trimName = trimName.replace("*", "");
    trimName = trimName.replace("&", "");
    trimName = trimName.replace("[", "");
    trimName = trimName.replace("]", "");
    trimName = trimName.replace("const ", "");
    trimName = trimName.replace(" const", "");
    trimName = trimName.replace("volatile", "");
    trimName = trimName.replace(" volatile", "");
    trimName = trimName.trim();
    return trimName;
  }

  /**
   * TODO might disappear
   * Currently reads (includes forward definitions)
   */
  public static List<ICElement> getAllIStructures(final IParent parent, final boolean allIncludes, final boolean lookNestedTypes, final ICProject project) {
    List<ICElement> ret = new UniqueEList<ICElement>();
    ICElement[] children = ReverseUtils.safeGetChilds(parent);
    for (int i = 0; (i < children.length); i++) {
      {
        ICElement child = children[i];
        boolean _matched = false;
        if ((child instanceof IStructure)) {
          _matched=true;
          ret.add(((IStructure) child));
          if (lookNestedTypes) {
            ret.addAll(ReverseUtils.getAllIStructures(((IStructure) child), allIncludes, lookNestedTypes, project));
          }
        }
        if (!_matched) {
          if ((child instanceof IEnumeration)) {
            _matched=true;
            ret.add(((IEnumeration) child));
          }
        }
        if (!_matched) {
          if ((child instanceof IParent)) {
            _matched=true;
            ret.addAll(ReverseUtils.getAllIStructures(((IParent) child), allIncludes, lookNestedTypes, project));
          }
        }
        if (!_matched) {
          if ((child instanceof IInclude)) {
            _matched=true;
            if (allIncludes) {
              ITranslationUnit unit = ReverseUtils.getTranslationUnitFromInclude(((IInclude) child), project);
              if ((unit != null)) {
                ret.addAll(ReverseUtils.getAllIStructures(unit, allIncludes, lookNestedTypes, project));
              }
            }
          }
        }
        if (!_matched) {
          if ((child instanceof ITypeDef)) {
            _matched=true;
            ret.add(child);
          }
        }
        if (!_matched) {
          if ((child instanceof IStructureDeclaration)) {
            _matched=true;
            ret.add(child);
          }
        }
      }
    }
    return ret;
  }

  /**
   * Obtain the translation unit from an IInclude instance.
   * Check whether examined header file is within project. If yes, translate fileName
   * to a project relative fileName and obtain translation unit
   */
  public static ITranslationUnit getTranslationUnitFromInclude(final IInclude include, final ICProject cproject) {
    try {
      String _fullFileName = include.getFullFileName();
      boolean _tripleNotEquals = (_fullFileName != null);
      if (_tripleNotEquals) {
        ITranslationUnit tu = ReverseUtils.getTranslationUnitFromPath(include.getFullFileName(), include.getCProject());
        if ((tu == null)) {
          final ICModel cModel = CoreModel.getDefault().getCModel();
          ICProject[] _cProjects = cModel.getCProjects();
          for (final ICProject otherCProject : _cProjects) {
            {
              tu = ReverseUtils.getTranslationUnitFromPath(include.getFullFileName(), otherCProject);
              if ((tu != null)) {
                Model _correspondingModel = ReverseUtils.getCorrespondingModel(tu);
                boolean _tripleEquals = (_correspondingModel == null);
                if (_tripleEquals) {
                  final ModelManagement modelManager = ReverseData.current.modelManager;
                  IProject _project = cproject.getProject();
                  String _elementName = otherCProject.getElementName();
                  String _plus = (_elementName + ReverseCpp2Uml.MODEL_POSTFIX);
                  String umlFilePath = modelManager.getPath(_project, 
                    ReverseCpp2Uml.REVERSE_FOLDER, _plus);
                  final boolean reset = false;
                  modelManager.createOrgetModel(otherCProject.getElementName(), umlFilePath, (!reset), reset);
                  final Model model = ReverseUtils.getCorrespondingModel(tu);
                  RoundtripCppUtils.applyProfile(model, CppProfileResource.PROFILE_PATH);
                  RoundtripCppUtils.applyProfile(model, StdUriConstants.UML_STD_PROFILE_PATH);
                  ReverseUtils.setXmlID(model);
                }
                return tu;
              }
            }
          }
        }
        return tu;
      }
      ReverseUtils.LOGGER.log(Level.WARNING, 
        String.format("path of \"%s\" include cannot be resolved", include.getElementName()));
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Obtain the translation unit from an absolute path
   */
  public static ITranslationUnit getTranslationUnitFromPath(final String fullFileName, final ICProject cproject) {
    final IPath fullProjectPath = cproject.getProject().getLocation();
    boolean _contains = fullFileName.contains(fullProjectPath.toString());
    boolean _not = (!_contains);
    if (_not) {
      return null;
    }
    ITranslationUnit ret = null;
    String fileIncludePath = IterableExtensions.<String>lastOrNull(((Iterable<String>)Conversions.doWrapArray(fullFileName.split(fullProjectPath.toString()))));
    IFile file = cproject.getProject().getFile(fileIncludePath);
    if ((file != null)) {
      ITranslationUnit unit = CoreModelUtil.findTranslationUnit(file);
      if ((unit == null)) {
        unit = CoreModel.getDefault().createTranslationUnitFrom(cproject, file.getLocation());
      }
      ret = unit;
    }
    return ret;
  }

  /**
   * Get the translationUnit from an element
   */
  public static ITranslationUnit getTranslationUnitFromElement(final ICElement element) {
    ICElement owner = element;
    while (((owner != null) && (!(owner instanceof ITranslationUnit)))) {
      owner = owner.getParent();
    }
    return ((ITranslationUnit) owner);
  }

  public static void unapplyAllStereotypes(final Element element) {
    final EList<EObject> stereotypeAppList = element.getStereotypeApplications();
    final Consumer<EObject> _function = (EObject it) -> {
      final Stereotype stereotype = UMLUtil.getStereotype(it);
      element.unapplyStereotype(stereotype);
    };
    stereotypeAppList.forEach(_function);
  }

  public static Stereotype applyStereotype(final Element element, final boolean doApply, final Class<? extends EObject> clazz) {
    Stereotype _xifexpression = null;
    if (doApply) {
      _xifexpression = StereotypeUtil.apply(element, clazz);
    }
    return _xifexpression;
  }

  /**
   * Used in the context of parameter type declarations
   */
  public static Type getUMLType(final IASTDeclSpecifier declSpecifier, final ICElement context) {
    Type ret = null;
    ret = ReverseUtils.getUMLType(ASTUtils.getQualifiedName(declSpecifier), context);
    return ret;
  }

  /**
   * Get a UML type from a given context (translation Unit)
   * @param typeName a qualified type name (except for basic types and template parameters)
   * @param context the ICElement that is related with the type lookup.
   *      Used to resolved non-qualified names, e.g. template parameters
   */
  public static Type getUMLType(final String typeName, final ICElement context) {
    try {
      final ITranslationUnit itu = ReverseUtils.getTranslationUnitFromElement(context);
      boolean _isPrimitiveCppType = RoundtripCppUtils.isPrimitiveCppType(typeName);
      if (_isPrimitiveCppType) {
        return RoundtripCppUtils.getPrimitiveType(typeName, ReverseUtils.getCorrespondingModel(itu));
      }
      if ((typeName == null)) {
        return null;
      }
      final Model containerPkg = ReverseUtils.getCorrespondingModel(itu);
      final String strippedTypeName = ReverseUtils.stripTemplate(typeName);
      String _name = containerPkg.getName();
      String _plus = (_name + "::*::");
      final String qName = (_plus + strippedTypeName);
      final NamedElement candidate1 = ElementUtils.getQualifiedElement(containerPkg, qName);
      if ((candidate1 instanceof Type)) {
        return ((Type)candidate1);
      }
      Type ret = null;
      final ICProject project = ReverseData.current.project;
      String[] token = strippedTypeName.split(Namespace.SEPARATOR);
      int _length = token.length;
      int _minus = (_length - 1);
      final String simpleTypeName = (token[_minus]).trim();
      final List<ICElement> nestedStructures = ReverseUtils.getAllIStructures(itu, false, true, project);
      final Function1<ICElement, Boolean> _function = (ICElement it) -> {
        return Boolean.valueOf(it.getElementName().trim().equals(simpleTypeName.trim()));
      };
      final ICElement sameNameType = IterableExtensions.<ICElement>head(IterableExtensions.<ICElement>filter(nestedStructures, _function));
      if ((sameNameType == null)) {
        Iterable<IStructureTemplate> structureTemplates = Iterables.<IStructureTemplate>filter(nestedStructures, IStructureTemplate.class);
        final Function1<IStructureTemplate, Boolean> _function_1 = (IStructureTemplate it) -> {
          final Function1<String, Boolean> _function_2 = (String it_1) -> {
            return Boolean.valueOf(it_1.equals(simpleTypeName));
          };
          int _size = IterableExtensions.size(IterableExtensions.<String>filter(((Iterable<String>)Conversions.doWrapArray(it.getTemplateParameterTypes())), _function_2));
          return Boolean.valueOf((_size > 0));
        };
        final IStructureTemplate match = IterableExtensions.<IStructureTemplate>head(IterableExtensions.<IStructureTemplate>filter(structureTemplates, _function_1));
        if ((match != null)) {
          Type templateClassifier = GetOrCreateP2.getClassifier(match);
          ret = GetOrCreateP1.getOrCreateTemplateParameter(((Classifier) templateClassifier), simpleTypeName, 
            "class");
        }
      }
      if ((ret == null)) {
        boolean _startsWith = typeName.startsWith("ecl");
        if (_startsWith) {
          System.err.println("what happened");
        }
        final org.eclipse.uml2.uml.Package externalPkg = RoundtripCppUtils.getOrcreateExternalPackage(ReverseUtils.getCorrespondingModel(itu), true);
        final String containerQName = ReverseUtils.stripLastQName(typeName);
        Namespace container = null;
        if ((containerQName != null)) {
          container = ReverseUtils.getContainer(externalPkg, containerQName);
        } else {
          container = externalPkg;
        }
        final NamedElement candidate = container.getMember(simpleTypeName);
        if ((candidate instanceof Type)) {
          NamedElement _member = container.getMember(simpleTypeName);
          ret = ((Type) _member);
        } else {
          if ((candidate != null)) {
            System.err.println("A non type member of same name exists, This should not happen");
            return null;
          }
        }
        if ((ret == null)) {
          ReverseUtils.LOGGER.log(Level.WARNING, (typeName + " is not found, it will be created dynamically"));
          if ((!(container instanceof org.eclipse.uml2.uml.Package))) {
            container = container.getNearestPackage();
          }
          ret = ((org.eclipse.uml2.uml.Package) container).createOwnedType(simpleTypeName, UMLPackage.Literals.DATA_TYPE);
          StereotypeUtil.apply(ret, External.class);
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Return the qualified name of the owner of aa variable or method definition.
   * Used during body reverse
   */
  public static Classifier getTypeOfVariableOrMethod(final ICElement element, final ITranslationUnit translationUnit) {
    if (((element instanceof IVariableDeclaration) || (element instanceof IMethod))) {
      final String qName = ASTUtils.getQualifiedName(((ISourceReference) element));
      if ((qName != null)) {
        final Type type = ReverseUtils.getUMLType(ReverseUtils.stripTemplate(qName), translationUnit);
        if ((type instanceof Classifier)) {
          return ((Classifier)type);
        }
      }
    }
    return null;
  }

  /**
   * return true, if the element is active, i.e. not disabled by
   * condititional directives such as #ifdef
   */
  public static boolean isActive(final ICElement element) {
    if ((element instanceof ISourceReference)) {
      return ((ISourceReference)element).isActive();
    }
    return false;
  }

  /**
   * Remove a template signature from a name, i.e. replace MyName<...> with MyName
   */
  public static String stripTemplate(final String name) {
    if ((name != null)) {
      final char LT = '<';
      final char GT = '>';
      final int openIdx = name.indexOf(LT);
      if ((openIdx != (-1))) {
        int count = 1;
        int closeIdx = (openIdx + 1);
        for (; ((count > 0) && (closeIdx < name.length())); closeIdx++) {
          {
            final char ch = name.charAt(closeIdx);
            if ((ch == LT)) {
              count++;
            } else {
              if ((ch == GT)) {
                count--;
              }
            }
          }
        }
        if ((count == 0)) {
          String _substring = name.substring(0, openIdx);
          String _substring_1 = name.substring(closeIdx);
          String _plus = (_substring + _substring_1);
          return ReverseUtils.stripTemplate(_plus);
        }
      }
    }
    return name;
  }

  /**
   * Remove the last segment of a qualified name. Returns null
   * if no separator is found, i.e. name is not qualified
   */
  public static String stripLastQName(final String qName) {
    if ((qName != null)) {
      final int idx = qName.lastIndexOf(Namespace.SEPARATOR);
      if ((idx != (-1))) {
        return qName.substring(0, idx);
      }
    }
    return null;
  }

  /**
   * Get the model corresponding to a translation unit (model map)
   */
  public static Model getCorrespondingModel(final ITranslationUnit unit) {
    Model ret = null;
    for (final Model model : ReverseData.current.models) {
      boolean _contains = unit.getPath().toString().contains(model.getName());
      if (_contains) {
        ret = model;
      }
    }
    return ret;
  }

  /**
   * Get a container (UML) for a given ICElement, it returns either a package
   * or a class. The method will always created packages, if they do not exist yet.
   * Currently, the location of the source folder is used in addition of C++ name-spaces
   * This has the disadvantage that UML namespaces and packages diverge. On the other
   * hand, some projects might use source folders as structuring elements
   */
  public static Namespace getContainer(final ICElement element) {
    return ReverseUtils.getContainer(element, false);
  }

  public static Namespace getContainer(final ICElement element, final boolean forAlias) {
    ITranslationUnit tu = ReverseUtils.getTranslationUnitFromElement(element);
    String qName = null;
    ISourceRoot _sourceFolder = ReverseUtils.getSourceFolder(tu);
    boolean _tripleEquals = (_sourceFolder == null);
    if (_tripleEquals) {
      qName = "external";
    } else {
      qName = ReverseUtils.getSourceFolder(tu).getElementName();
    }
    String qNameSR = null;
    if (forAlias) {
      qNameSR = ASTUtils.getQualifiedName(((ISourceReference) element));
    } else {
      if ((element instanceof ITypeDef)) {
        ICElement _parent = ((ITypeDef)element).getParent();
        qNameSR = ASTUtils.getQualifiedName(((ISourceReference) _parent));
      } else {
        qNameSR = ReverseUtils.stripLastQName(ASTUtils.getQualifiedName(((ISourceReference) element)));
      }
    }
    if ((qNameSR != null)) {
      qName = ((qName + Namespace.SEPARATOR) + qNameSR);
    }
    return ReverseUtils.getContainer(ReverseUtils.getCorrespondingModel(tu), qName);
  }

  /**
   * get a container from a qualified name
   */
  public static Namespace getContainer(final org.eclipse.uml2.uml.Package parent, final String qName) {
    if ((parent == null)) {
      return null;
    }
    Namespace parentPack = parent;
    String[] _split = qName.split(Namespace.SEPARATOR);
    for (final String name : _split) {
      {
        NamedElement _member = parentPack.getMember(name);
        boolean _tripleEquals = (_member == null);
        if (_tripleEquals) {
          if ((parentPack instanceof org.eclipse.uml2.uml.Package)) {
            boolean _equals = name.equals("UmsConfig");
            if (_equals) {
              System.err.println("debug");
            }
            final org.eclipse.uml2.uml.Package pkg = ((org.eclipse.uml2.uml.Package)parentPack).createNestedPackage(name);
            ReverseUtils.setXmlID(pkg);
          } else {
            System.err.println("should not happen");
            return parentPack;
          }
        }
        NamedElement _member_1 = parentPack.getMember(name);
        parentPack = ((Namespace) _member_1);
      }
    }
    return parentPack;
  }

  public static ISourceRoot getSourceFolder(final ICElement parent) {
    ICElement sf = parent;
    while ((sf != null)) {
      {
        if ((sf instanceof ISourceRoot)) {
          return ((ISourceRoot)sf);
        }
        sf = sf.getParent();
      }
    }
    return null;
  }

  public static String getExcludedIncludesMapKey(final ITranslationUnit translationUnit, final Element umlElement) {
    int _hashCode = translationUnit.hashCode();
    String _plus = ("" + Integer.valueOf(_hashCode));
    int _hashCode_1 = umlElement.hashCode();
    return (_plus + Integer.valueOf(_hashCode_1));
  }

  public static List<Map.Entry<ICElement, EObject>> getElementsSameName(final String name) {
    final Function1<Map.Entry<ICElement, EObject>, Boolean> _function = (Map.Entry<ICElement, EObject> it) -> {
      return Boolean.valueOf(it.getKey().getElementName().equals(name));
    };
    return IterableExtensions.<Map.Entry<ICElement, EObject>>toList(IterableExtensions.<Map.Entry<ICElement, EObject>>filter(ReverseData.current.map.entrySet(), _function));
  }

  public static ICElement lookTypeInContainer(final ICContainer container, final String typeName) {
    ICElement ret = null;
    ICElement[] _safeGetChilds = ReverseUtils.safeGetChilds(container);
    for (final ICElement child : _safeGetChilds) {
      {
        if ((child instanceof ITranslationUnit)) {
          List<ICElement> nesteds = ReverseUtils.getAllIStructures(((IParent)child), false, true, ReverseData.current.project);
          final Function1<ICElement, Boolean> _function = (ICElement it) -> {
            return Boolean.valueOf((((it instanceof IStructure) || (it instanceof IEnumeration)) || (it instanceof ITypeDef)));
          };
          final Function1<ICElement, Boolean> _function_1 = (ICElement it) -> {
            return Boolean.valueOf(it.getElementName().equals(typeName));
          };
          ret = IterableExtensions.<ICElement>head(IterableExtensions.<ICElement>filter(IterableExtensions.<ICElement>filter(nesteds, _function), _function_1));
        } else {
          if ((child instanceof ICContainer)) {
            ret = ReverseUtils.lookTypeInContainer(((ICContainer)child), typeName);
          }
        }
        if ((ret != null)) {
          return ret;
        }
      }
    }
    return ret;
  }

  public static Type lookForATypeByName(final String typeName) {
    try {
      Type ret = null;
      ICElement cType = null;
      for (final ICContainer container : ReverseData.current.containers) {
        if ((cType == null)) {
          cType = ReverseUtils.lookTypeInContainer(container, typeName);
        }
      }
      if ((cType != null)) {
        ret = GetOrCreateP2.getClassifier(cType);
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public static boolean isSimpleAssociation(final Property prop) {
    if (((StereotypeUtil.isApplied(prop, Ptr.class) || StereotypeUtil.isApplied(prop, Ref.class)) && (!StereotypeUtil.isApplied(prop, Array.class)))) {
      return true;
    }
    return false;
  }

  public static boolean isAggregation(final Property prop) {
    if (((StereotypeUtil.isApplied(prop, Ptr.class) || StereotypeUtil.isApplied(prop, Ref.class)) && StereotypeUtil.isApplied(prop, Array.class))) {
      return true;
    }
    return false;
  }

  public static boolean isComposition(final Property prop) {
    if (((!StereotypeUtil.isApplied(prop, Ptr.class)) && (!StereotypeUtil.isApplied(prop, Ref.class)))) {
      return true;
    }
    return false;
  }

  public static String getMemberInit(final IMethodDeclaration method) {
    return ReverseUtils.getMemberInit(ASTUtils.findEnclosingNode(method));
  }

  public static String getMemberInit(final IASTNode methodNode) {
    String ret = "";
    if ((methodNode instanceof ICPPASTFunctionDefinition)) {
      final ICPPASTConstructorChainInitializer[] inits = ((ICPPASTFunctionDefinition)methodNode).getMemberInitializers();
      if ((inits != null)) {
        StringConcatenation _builder = new StringConcatenation();
        {
          boolean _hasElements = false;
          for(final ICPPASTConstructorChainInitializer i : inits) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "");
            }
            IASTName _memberInitializerId = i.getMemberInitializerId();
            _builder.append(_memberInitializerId);
            String _rawSignature = i.getInitializer().getRawSignature();
            _builder.append(_rawSignature);
          }
        }
        ret = _builder.toString();
      }
    }
    return ret.toString();
  }

  /**
   * Write an XML id based on its name (without the top-level package name).
   * This enables a pragmatical re-use of XML-IDs (e.g. in a Papyrus diagram) when the project
   * is re-generated. OF course, a good synchronization would avoid re-creating UML
   * elements (and the associated IDs)
   * @param ne a named element
   */
  public static void setXmlID(final NamedElement ne) {
    final org.eclipse.uml2.uml.Package root = PackageUtil.getRootPackage(ne);
    String qName = null;
    if ((ne instanceof Usage)) {
      String _qualifiedName = ((Usage)ne).getSuppliers().get(0).getQualifiedName();
      String _plus = ("U_" + _qualifiedName);
      qName = _plus;
    } else {
      qName = ne.getQualifiedName();
    }
    if ((qName != null)) {
      String _replaceAll = qName.replaceFirst(root.getName(), "").replaceAll(NamedElement.SEPARATOR, "_").replaceAll("#", "_").replaceAll(" ", "_").replaceAll("<", "_").replaceAll(">", "_").replaceAll("&", "_");
      String _plus_1 = ("ID" + _replaceAll);
      ReverseUtils.setXmlID(ne, _plus_1);
    }
  }

  /**
   * Set the XML ID of an eObject
   * 
   * @param eObject
   * @param uniqueID
   */
  public static void setXmlID(final EObject eObject, final String id) {
    final Resource eObjectRes = eObject.eResource();
    if ((eObjectRes instanceof XMLResource)) {
      final XMLResource xmlResource = ((XMLResource) eObjectRes);
      String uniqueID = id;
      int counter = 0;
      while ((xmlResource.getEObject(uniqueID) != null)) {
        {
          uniqueID = (id + Integer.valueOf(counter));
          counter++;
        }
      }
      xmlResource.setID(eObject, uniqueID);
    }
  }

  /**
   * It seems that the call to getChildren can cause an stack-overflow in some cases.
   * Use this function to capture it (in this case, an empty array is returned)
   */
  public static ICElement[] safeGetChilds(final IParent parent) {
    try {
      ICElement[] children = null;
      try {
        children = parent.getChildren();
      } catch (final Throwable _t) {
        if (_t instanceof StackOverflowError) {
          final StackOverflowError e = (StackOverflowError)_t;
          ReverseUtils.LOGGER.log(Level.WARNING, 
            String.format("Exception <%s> during fetching of children of parent <%s>", e.getMessage(), parent.toString()));
          children = ReverseUtils.EMPTY_CHILDS;
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      return children;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
