/*****************************************************************************
 * Copyright (c) 2023, 2025 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Issue GL-9
 *****************************************************************************/

package org.eclipse.papyrus.model2doc.markup.emf.template2structure.internal.mapping;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.model2doc.core.logger.IModel2DocLogger;
import org.eclipse.papyrus.model2doc.emf.documentstructure.BodyPart;
import org.eclipse.papyrus.model2doc.emf.documentstructure.InsertedGeneratedFile;
import org.eclipse.papyrus.model2doc.emf.template2structure.mapping.IMappingService;
import org.eclipse.papyrus.model2doc.markup.emf.documentstructuretemplate.EAttributeInMarkupToFile;
import org.eclipse.papyrus.model2doc.markup.emf.documentstructuretemplate.MarkupEMFDocumentStructureTemplatePackage;
import org.eclipse.papyrus.model2doc.markup.emf.template2structure.mapping.AbstractMarkdownToHtmlFileMapper;

/**
 * This mapper convert the value of EAttributes written in markdown on in html into an html file.
 */
public class EAttributeInMarkupToHtmlFileMapper extends AbstractMarkdownToHtmlFileMapper<EAttributeInMarkupToFile> {

	/**
	 * Constructor.
	 *
	 * @param inputEClass
	 * @param outputClass
	 */
	public EAttributeInMarkupToHtmlFileMapper() {
		super(MarkupEMFDocumentStructureTemplatePackage.eINSTANCE.getEAttributeInMarkupToFile(), BodyPart.class);
	}

	/**
	 * @see org.eclipse.papyrus.model2doc.emf.template2structure.mapping.AbstractTemplateToStructureMapper#doMap(org.eclipse.papyrus.model2doc.emf.template2structure.mapping.IMappingService, org.eclipse.emf.ecore.EObject, org.eclipse.emf.ecore.EObject,
	 *      java.lang.Class, IModel2DocLogger)
	 *
	 * @param <T>
	 * @param mappingService
	 * @param documentTemplateElement
	 * @param semanticModelElement
	 * @param expectedReturnedClass
	 * @param logger
	 * @return
	 */
	@Override
	protected <T> List<T> doMap(final IMappingService mappingService, final EAttributeInMarkupToFile documentTemplateElement, final EObject semanticModelElement, final Class<T> expectedReturnedClass, final IModel2DocLogger logger) {
		if (false == documentTemplateElement.generateBranch(semanticModelElement)) {
			return Collections.emptyList();
		}
		if (false == documentTemplateElement.isGenerate() || false == semanticModelElement instanceof EObject) {
			// This element can't have children, so if isGenerate() returns false, we go out!
			return Collections.emptyList();
		}

		final Collection<Object> eAttributeValues = documentTemplateElement.getEAttributeValues(semanticModelElement);
		final List<T> generatedElements = new ArrayList<>();

		final String elementID = getXMIID(semanticModelElement);
		final String eattributeName = documentTemplateElement.getEAttribute().getName();

		int index = 0;
		final Iterator<Object> eAttributeValuesIter = eAttributeValues.iterator();
		while (eAttributeValuesIter.hasNext()) {
			final InsertedGeneratedFile generatedFile = STRUCTURE_EFACTORY.createInsertedGeneratedFile();
			Object value = eAttributeValuesIter.next();
			if (value == null) {
				value = EMPTY_STRING;
			}
			final StringBuilder fileNameBuilder = new StringBuilder();
			fileNameBuilder.append(elementID).append(UNDESCORE).append(eattributeName);
			if (eAttributeValues.size() > 1) {
				fileNameBuilder.append(UNDESCORE);
				fileNameBuilder.append(index++);
			}

			generatedFile.setFilePath(createHTMLFile(documentTemplateElement, value.toString(), fileNameBuilder.toString(), logger));
			generatedElements.add(expectedReturnedClass.cast(generatedFile));
		}

		return buildMapperResult(documentTemplateElement, semanticModelElement, expectedReturnedClass, generatedElements);
	}
}
