/*****************************************************************************
 * Copyright (c) 2023 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *
 *****************************************************************************/

package org.eclipse.papyrus.model2doc.emf.template2structure.internal.generator;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.papyrus.model2doc.emf.template2structure.generator.ITemplate2StructureGenerator;
import org.eclipse.papyrus.model2doc.emf.template2structure.mapping.AbstractTemplateToStructureMapper;

/**
 * This class allows to describe the {@link ITemplate2StructureGenerator} with their inheritance and mappers
 */
public final class Template2StructureGeneratorDescription {

	/**
	 * the generator described by this class
	 */
	private ITemplate2StructureGenerator generator;

	/**
	 * the mappers to use with this generator
	 */
	private List<AbstractTemplateToStructureMapper<?>> mappers = new ArrayList<>();

	/**
	 * the id of the inherited generator
	 */
	private String inheritedFromGeneratorId;

	/**
	 * the inherited generator
	 */
	private Template2StructureGeneratorDescription inheritedGenerator;

	/**
	 *
	 * Constructor.
	 *
	 * @param generator
	 *            the generator represented by this class
	 * @param inheritedFromGeneratorId
	 *            the id of the inherited generator (can be null)
	 *
	 *
	 */
	public Template2StructureGeneratorDescription(final ITemplate2StructureGenerator generator, final String inheritedFromGeneratorId) {
		Assert.isNotNull(generator);
		this.generator = generator;
		this.inheritedFromGeneratorId = inheritedFromGeneratorId;
	}

	/**
	 *
	 * @return
	 *         the generator represented by this class
	 */
	public final ITemplate2StructureGenerator getGenerator() {
		return this.generator;
	}

	/**
	 *
	 * @return
	 *         the id of the inherited generator
	 */
	public final String getInheritsFromGeneratorId() {
		return this.inheritedFromGeneratorId;
	}

	/**
	 *
	 * @return
	 *         the list of available mappers for this generator
	 */
	public List<AbstractTemplateToStructureMapper<?>> getMappers() {
		final List<AbstractTemplateToStructureMapper<?>> mappers = new LinkedList<>();
		// 1. we start with the mappers registered for the represented generator
		mappers.addAll(this.mappers);

		// 2. we continue with the mappers inherited from the parent generator.
		// this implementation allows to override easily the mappers declared on the inherited generator
		if (this.inheritedGenerator != null) {
			mappers.addAll(this.inheritedGenerator.getMappers());
		}
		return mappers;
	}

	/**
	 * Register a mapper for this generator
	 *
	 * @param newMapper
	 *            a new mapper
	 */
	public final void registerMapper(final AbstractTemplateToStructureMapper<?> newMapper) {
		this.mappers.add(newMapper);
	}

	/**
	 *
	 * @return
	 *         the id of the represented generator
	 */
	public final String getGeneratorId() {
		return this.generator.getGeneratorId();
	}

	/**
	 *
	 * @param inheritedGenerator
	 */
	public final void setInheritedGenerator(final Template2StructureGeneratorDescription inheritedGenerator) {
		this.inheritedGenerator = inheritedGenerator;
	}

	/**
	 *
	 * @return
	 *         the inherited generator
	 */
	public final Template2StructureGeneratorDescription getInheritedGenerator() {
		return this.inheritedGenerator;
	}
}
