/*****************************************************************************
 * Copyright (c) 2020 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * Contributors:
 *  Matteo MORELLI      matteo.morelli@cea.fr - Bug #566899
 *
 *****************************************************************************/

package org.eclipse.papyrus.robotics.ros2.codegen.python.skillrealization

import java.util.Map
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillDefinition
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillOperationalState
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillSemantic
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SkillUtils
import org.eclipse.uml2.uml.Interface

import static extension org.eclipse.papyrus.robotics.ros2.codegen.common.utils.RosHelpers.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SkillUtils.*
import org.eclipse.papyrus.designer.languages.common.base.file.IPFileSystemAccess

/**
 * Create the C++ code of a skill realization executed
 * on the sequencing layer (uses a config&coord interface to SW components)
 */
class CreateSkillRealizationPythonCode {

	private static def msgAbortMultipleOperationalStatesIsUnsopported() {
		return String.format(
				"abort transformation, code-generation of skill FSMs with many operational states is not supported"
				)
	}

	private static def msgAbortSystemHasNotCompIf(String ifName) {
		return String.format(
				"abort transformation, there are no components in the system with the required coordination interface (%s)",
					ifName
				)
	}

/*
	static def createCppSkillFSMManager(SkillDefinition definition, SkillSemantic semantics) '''
	'''
*/

	static def createCppSimplestSkillWithSWComponentInteraction(SkillDefinition skill, SkillOperationalState ops, String actionName) '''
		// Generated by Papyrus4Robotics
		//
		
		«FOR tp : skill.uniqueSkillParameterTypes»
			#include "«tp.ROS2TypeIncludePath»"
		«ENDFOR»
		#include "«ops.coordinationIfIncludePath»"
		#include "nav2_behavior_tree/bt_action_node.hpp"
		
		class «skill.name»Action : public nav2_behavior_tree::BtActionNode<«ops.coordinationIfQn»>
		{
		public:
		  «skill.name»Action(
		      const std::string& name,
		      const std::string & action_name,
		      const BT::NodeConfiguration& conf)
		  : nav2_behavior_tree::BtActionNode<«ops.coordinationIfQn»>(name, action_name, conf)
		    {
		    }
		
		  «IF !skill.ins.isNullOrEmpty»
		  	void on_tick() override
		  	{
		  	  «FOR param : skill.ins»
		  	  	«param.type.externalName» «param.name»;
		  	  	getInput("«param.name»", «param.name»);
		  	  	«FOR prop : param.type.ownedAttributes»
		  	  		goal_.«prop.name» = «param.name».«prop.name»;
		  	  	«ENDFOR»
		  	«ENDFOR»
		  	}
		  «ENDIF»
		
		  «IF !skill.outs.isNullOrEmpty»
		  	BT::NodeStatus on_success() override
		  	{
		  		«FOR param : skill.outs»
		  			«param.type.externalName» «param.name»;
		  			«FOR prop : param.type.ownedAttributes»
		  				«param.name».«prop.name» = result_.«prop.name»;
		  			«ENDFOR»
		  			setOutput("«param.name»", «param.name»);
		  		«ENDFOR»
		  		return BT::NodeStatus::SUCCESS;
		  	}
		  «ENDIF»
		
		  «createProvidedPortsMethod(skill)»
		};
		
		#include "behaviortree_cpp_v3/bt_factory.h"
		BT_REGISTER_NODES(factory)
		{
		  BT::NodeBuilder builder =
		    [](const std::string & name, const BT::NodeConfiguration & config)
		    {
		      return std::make_unique<«skill.name»Action>(name, "«actionName»", config);
		    };
		
		  factory.registerBuilder<«skill.name»Action>("«skill.name»", builder);
		}
	'''

	static def createCppSimplestSkillNoInteraction(SkillDefinition skill) '''
		// Generated by Papyrus4Robotics
		// ...
		
		#include <string>
		#include "rclcpp/rclcpp.hpp"
		«FOR tp : skill.uniqueSkillParameterTypes»
			#include "«tp.ROS2TypeIncludePath»"
		«ENDFOR»
		#include "behaviortree_cpp_v3/action_node.h"
		
		class «skill.name»Action : public BT::SyncActionNode
		{
		
		public:
		  // Any TreeNode with ports must have a constructor with this signature
		  «skill.name»Action(
		    const std::string& name,
		    const BT::NodeConfiguration& config)
		  : SyncActionNode(name, config)
		  {
		  }
		
		  «createProvidedPortsMethod(skill)»
		
		  BT::NodeStatus tick() override
		  {
			«IF !skill.ins.isNullOrEmpty»
				// Read from input ports
				//
				«FOR param : skill.ins»
					«param.type.externalName» «param.name»;
					getInput("«param.name»", «param.name»);
				«ENDFOR»

			/* ----------------------------------------------
			 * USER CODE HERE USING READINGS FROM INPUT PORTS
			 * ----------------------------------------------
			 */
			«ENDIF»

			«IF skill.ins.isNullOrEmpty && skill.outs.isNullOrEmpty»
			/* -------------------
			 * SOME USER CODE HERE
			 * -------------------
			 */
			«ENDIF»

			«IF !skill.outs.isNullOrEmpty»
			// Share through the blackboard
			//
			«FOR param : skill.outs»
				«param.type.externalName» «param.name»;
				/* -----------------------------------------
				 * USER CODE HERE TO INITIALIZE THE VARIABLE
				 * -----------------------------------------
				 */
				setOutput("«param.name»", «param.name»);

			«ENDFOR»
			«ENDIF»
		  	return BT::NodeStatus::SUCCESS;
		  }
		
		};
		
		#include "behaviortree_cpp_v3/bt_factory.h"
		
		// This function must be implemented in the .cpp file to create
		// a plugin that can be loaded at run-time
		BT_REGISTER_NODES(factory)
		{
		    factory.registerNodeType<«skill.name»Action>("«skill.name»");
		}
	'''

	static def createProvidedPortsMethod(SkillDefinition skill) '''
		«IF !skill.ins.isNullOrEmpty || !skill.outs.isNullOrEmpty»
			// «skill.name» has in/out parameters => must provide a providedPorts method
			static BT::PortsList providedPorts()
			{
			  return{
			    «FOR param : skill.ins SEPARATOR ','»
			    	BT::InputPort<«param.type.externalName»>("«param.name»")
			    «ENDFOR»
			    «FOR param : skill.outs SEPARATOR ','»
			    	BT::OutputPort<«param.type.externalName»>("«param.name»")
			    	«ENDFOR»
			  	};
				}
			«ENDIF»
	'''

	static def genCode(IPFileSystemAccess fileAccess, Map<SkillDefinition, SkillSemantic> skdefToSemanticsMap, Map<Interface, String> serviceToNameMap) {
		for (entry : skdefToSemanticsMap.entrySet) {
			val definition  = entry.key
			val semantics   = entry.value
			val skillRealizationFileName = definition.realizationFileName+".cpp"
			// Generate a skill manager to manage complex realization semantics with more than 1 operational state
			if (semantics.operational.size > 1) {
				//fileAccess.generateFile(skillRealizationFileName, createCppSkillFSMManager(definition, semantics).toString)
				throw new TransformationException(
						msgAbortMultipleOperationalStatesIsUnsopported()
					)
			}
			else {
			// here it is guaranteed that semantics.operational.size == 1
				val ops = semantics.firstOpState
				if (SkillUtils.doesConfigAndCoordOfComponents(ops)) {
					if (!serviceToNameMap.containsKey(SkillUtils.getCompInterface(ops)))
						throw new
							TransformationException(
								msgAbortSystemHasNotCompIf(
									SkillUtils.getCompInterface(ops).name
								)
							);
					val actionName = serviceToNameMap.get(SkillUtils.getCompInterface(ops))
					fileAccess.generateFile(skillRealizationFileName, createCppSimplestSkillWithSWComponentInteraction(definition, ops, actionName).toString)
				}
				else
					fileAccess.generateFile(skillRealizationFileName, createCppSimplestSkillNoInteraction(definition).toString)
			}
		}
	}
}
