/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.cpp.component;

import java.util.ArrayList;
import org.eclipse.core.resources.IProject;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.common.base.file.IPFileSystemAccess;
import org.eclipse.papyrus.designer.languages.common.profile.Codegen.TemplateBinding;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Include;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.designer.transformation.core.transformations.ExecuteTransformationChain;
import org.eclipse.papyrus.designer.transformation.core.transformations.TransformationContext;
import org.eclipse.papyrus.designer.uml.tools.utils.ElementUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.ActivityUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.ApplyProfiles;
import org.eclipse.papyrus.robotics.codegen.common.utils.ComponentUtils;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.core.utils.ParameterUtils;
import org.eclipse.papyrus.robotics.core.utils.PortUtils;
import org.eclipse.papyrus.robotics.profile.robotics.commpattern.CommunicationPattern;
import org.eclipse.papyrus.robotics.profile.robotics.components.Activity;
import org.eclipse.papyrus.robotics.profile.robotics.components.ComponentDefinition;
import org.eclipse.papyrus.robotics.profile.robotics.functions.Function;
import org.eclipse.papyrus.robotics.ros2.codegen.common.component.AbstractCompTransformations;
import org.eclipse.papyrus.robotics.ros2.codegen.common.component.ComponentTransformationUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.message.CreateMsgPackage;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.MessageUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.RosHelpers;
import org.eclipse.papyrus.robotics.ros2.codegen.cpp.utils.ProjectTools;
import org.eclipse.papyrus.robotics.ros2.codegen.cpp.utils.RosCppTypes;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.PackageableElement;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;

@SuppressWarnings("all")
public class ComponentTransformations extends AbstractCompTransformations {
  public ComponentTransformations(final IPFileSystemAccess fileAccess, final IProject genProject) {
    super(fileAccess, genProject);
  }

  public static void createPubsSubsAttrs(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      {
        final CommunicationPattern cp = InteractionUtils.getCommunicationPattern(port);
        if ((InteractionUtils.isPush(cp) || InteractionUtils.isPubSub(cp))) {
          Property attribute = null;
          int _size = port.getProvideds().size();
          boolean _greaterThan = (_size > 0);
          if (_greaterThan) {
            final Type rosPublisher = RosCppTypes.getType(port, "ros2Library::rclcpp_lifecycle::LifecyclePublisher");
            StringConcatenation _builder = new StringConcatenation();
            String _varName = ElementUtils.varName(port);
            _builder.append(_varName);
            _builder.append("_pub_");
            attribute = component.createOwnedAttribute(_builder.toString(), rosPublisher);
          } else {
            int _size_1 = port.getRequireds().size();
            boolean _greaterThan_1 = (_size_1 > 0);
            if (_greaterThan_1) {
              final Type rosSubscriber = RosCppTypes.getType(port, "ros2Library::rclcpp::Subscription");
              StringConcatenation _builder_1 = new StringConcatenation();
              String _varName_1 = ElementUtils.varName(port);
              _builder_1.append(_varName_1);
              _builder_1.append("_sub_");
              attribute = component.createOwnedAttribute(_builder_1.toString(), rosSubscriber);
            }
          }
          ApplyProfiles.applyCommonProfile(attribute);
          final TemplateBinding template = StereotypeUtil.<TemplateBinding>applyApp(attribute, TemplateBinding.class);
          template.getActuals().add(InteractionUtils.getCommObject(port));
          RosHelpers.useSharedPtr(attribute);
        }
      }
    }
  }

  public static void createSendAttrs(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      boolean _isSend = InteractionUtils.isSend(InteractionUtils.getCommunicationPattern(port));
      if (_isSend) {
        Property attribute = null;
        int _size = port.getProvideds().size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          final Type rosSubscriber = RosCppTypes.getType(port, "ros2Library::rclcpp::Subscription");
          StringConcatenation _builder = new StringConcatenation();
          String _varName = ElementUtils.varName(port);
          _builder.append(_varName);
          _builder.append("_recv_");
          attribute = component.createOwnedAttribute(_builder.toString(), rosSubscriber);
        } else {
          int _size_1 = port.getRequireds().size();
          boolean _greaterThan_1 = (_size_1 > 0);
          if (_greaterThan_1) {
            final Type rosPublisher = RosCppTypes.getType(port, "ros2Library::rclcpp_lifecycle::LifecyclePublisher");
            StringConcatenation _builder_1 = new StringConcatenation();
            String _varName_1 = ElementUtils.varName(port);
            _builder_1.append(_varName_1);
            _builder_1.append("_send_");
            attribute = component.createOwnedAttribute(_builder_1.toString(), rosPublisher);
          }
        }
        ApplyProfiles.applyCommonProfile(attribute);
        final TemplateBinding template = StereotypeUtil.<TemplateBinding>applyApp(attribute, TemplateBinding.class);
        template.getActuals().add(InteractionUtils.getCommObject(port));
        RosHelpers.useSharedPtr(attribute);
      }
    }
  }

  public static void createServiceAttrs(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      boolean _isQuery = InteractionUtils.isQuery(InteractionUtils.getCommunicationPattern(port));
      if (_isQuery) {
        Property attribute = null;
        int _size = port.getProvideds().size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          final Type rosService = RosCppTypes.getType(port, "ros2Library::rclcpp::Service");
          StringConcatenation _builder = new StringConcatenation();
          String _varName = ElementUtils.varName(port);
          _builder.append(_varName);
          _builder.append("_srv_");
          attribute = component.createOwnedAttribute(_builder.toString(), rosService);
        } else {
          int _size_1 = port.getRequireds().size();
          boolean _greaterThan_1 = (_size_1 > 0);
          if (_greaterThan_1) {
            final Type rosClient = RosCppTypes.getType(port, "ros2Library::rclcpp::Client");
            StringConcatenation _builder_1 = new StringConcatenation();
            String _varName_1 = ElementUtils.varName(port);
            _builder_1.append(_varName_1);
            _builder_1.append("_client_");
            attribute = component.createOwnedAttribute(_builder_1.toString(), rosClient);
          }
        }
        TemplateBinding template = StereotypeUtil.<TemplateBinding>applyApp(attribute, TemplateBinding.class);
        if ((template == null)) {
          ApplyProfiles.applyCommonProfile(attribute);
          template = StereotypeUtil.<TemplateBinding>applyApp(attribute, TemplateBinding.class);
        }
        template.getActuals().add(MessageUtils.getServiceType(port));
        RosHelpers.useSharedPtr(attribute);
      }
    }
  }

  /**
   * Add action attributes
   */
  public static void createActionAttrs(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      boolean _isAction = InteractionUtils.isAction(InteractionUtils.getCommunicationPattern(port));
      if (_isAction) {
        Property attribute = null;
        int _size = port.getProvideds().size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          final Type rosService = RosCppTypes.getType(port, "ros2Library::rclcpp_action::Server");
          StringConcatenation _builder = new StringConcatenation();
          String _varName = ElementUtils.varName(port);
          _builder.append(_varName);
          _builder.append("_actsrv_");
          attribute = component.createOwnedAttribute(_builder.toString(), rosService);
        } else {
          int _size_1 = port.getRequireds().size();
          boolean _greaterThan_1 = (_size_1 > 0);
          if (_greaterThan_1) {
            final Type rosClient = RosCppTypes.getType(port, "ros2Library::rclcpp_action::Client");
            StringConcatenation _builder_1 = new StringConcatenation();
            String _varName_1 = ElementUtils.varName(port);
            _builder_1.append(_varName_1);
            _builder_1.append("_actcli_");
            attribute = component.createOwnedAttribute(_builder_1.toString(), rosClient);
          }
        }
        TemplateBinding template = StereotypeUtil.<TemplateBinding>applyApp(attribute, TemplateBinding.class);
        if ((template == null)) {
          ApplyProfiles.applyCommonProfile(attribute);
          template = StereotypeUtil.<TemplateBinding>applyApp(attribute, TemplateBinding.class);
        }
        template.getActuals().add(MessageUtils.getServiceType(port));
        RosHelpers.useSharedPtr(attribute);
      }
    }
  }

  /**
   * Remove functions that are not referenced by activities (this can happen after
   * deletion of a function from an activity)
   */
  public static void removeUnrefFunctions(final org.eclipse.uml2.uml.Class component) {
    final ComponentDefinition cd = UMLUtil.<ComponentDefinition>getStereotypeApplication(component, ComponentDefinition.class);
    final ArrayList<Function> fctList = new ArrayList<Function>();
    EList<Activity> _activities = cd.getActivities();
    for (final Activity activity : _activities) {
      fctList.addAll(activity.getFunctions());
    }
    PackageableElement[] _clone = ((PackageableElement[])Conversions.unwrapArray(component.getNearestPackage().getPackagedElements(), PackageableElement.class)).clone();
    for (final PackageableElement pe : _clone) {
      if ((pe instanceof OpaqueBehavior)) {
        final Function fct = UMLUtil.<Function>getStereotypeApplication(pe, Function.class);
        if (((fct != null) && (!fctList.contains(fct)))) {
          ((OpaqueBehavior)pe).destroy();
        }
      }
    }
  }

  @Override
  public void componentTrafo(final org.eclipse.uml2.uml.Class component, final CreateMsgPackage msgPkgCreator) {
    try {
      MessageUtils.createMessagesOrServices(msgPkgCreator, component);
      if ((this.genProject == null)) {
        throw new TransformationException(ExecuteTransformationChain.USER_CANCEL);
      }
      ComponentTransformationUtils.liftFunctions(component);
      Constructor.createConstructor(component);
      Include include = StereotypeUtil.<Include>applyApp(component, Include.class);
      String _header = include.getHeader();
      String _plus = (_header + "#include <rclcpp/rclcpp.hpp>\n");
      include.setHeader(_plus);
      boolean _hasActions = MessageUtils.hasActions(component);
      if (_hasActions) {
        String _header_1 = include.getHeader();
        String _plus_1 = (_header_1 + "#include <rclcpp_action/rclcpp_action.hpp>\n");
        include.setHeader(_plus_1);
      }
      boolean _isRegistered = ComponentUtils.isRegistered(component);
      if (_isRegistered) {
        String _body = include.getBody();
        CharSequence _registerComponent = CreateMain.registerComponent(component);
        String _plus_2 = (_body + _registerComponent);
        include.setBody(_plus_2);
      }
      CreateMain.createMain(component);
      boolean _hasExternalCode = ActivityUtils.hasExternalCode(component);
      if (_hasExternalCode) {
        new Ros2CodeSkeleton(component).createSkeleton();
      }
      ComponentTransformationUtils.removeActivities(component);
      ComponentTransformations.createPubsSubsAttrs(component);
      ComponentTransformations.createSendAttrs(component);
      ComponentTransformations.createServiceAttrs(component);
      ComponentTransformations.createActionAttrs(component);
      int _size = ParameterUtils.getAllParameters(component).size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        ParameterTransformations.declareParameters(component);
        ParameterTransformations.initParameters(component);
      }
      ParameterTransformations.moveParameters(component);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  @Override
  public void componentCodegen(final org.eclipse.uml2.uml.Class component, final CreateMsgPackage msgPkgCreator) {
    final RoboticsCppCreator codeGen = new RoboticsCppCreator(this.genProject, this.fileAccess, "src-skel/", "src/");
    ComponentTransformationUtils.removeTemplateSig(component);
    ComponentTransformationUtils.removePorts(component);
    ComponentTransformations.removeUnrefFunctions(component);
    TransformationContext.current.project = this.genProject;
    ProjectTools.genCode(codeGen, component);
  }
}
