/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.common.component;

import java.util.ArrayList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.core.utils.PortUtils;
import org.eclipse.papyrus.robotics.profile.robotics.components.Activity;
import org.eclipse.papyrus.robotics.profile.robotics.components.ComponentDefinition;
import org.eclipse.papyrus.robotics.profile.robotics.functions.Function;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.PackageableElement;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtext.xbase.lib.Conversions;

@SuppressWarnings("all")
public class ComponentTransformationUtils {
  /**
   * Move functions in passed activity to component (node) level
   */
  public static void liftFunctions(final org.eclipse.uml2.uml.Class component) {
    final ComponentDefinition cd = UMLUtil.<ComponentDefinition>getStereotypeApplication(component, ComponentDefinition.class);
    EList<Activity> _activities = cd.getActivities();
    for (final Activity activity : _activities) {
      ComponentTransformationUtils.liftFunctions(activity, component);
    }
  }

  /**
   * Remove activities from node
   */
  public static void removeActivities(final org.eclipse.uml2.uml.Class component) {
    final ComponentDefinition cd = UMLUtil.<ComponentDefinition>getStereotypeApplication(component, ComponentDefinition.class);
    Activity[] _clone = ((Activity[])Conversions.unwrapArray(cd.getActivities(), Activity.class)).clone();
    for (final Activity activity : _clone) {
      {
        component.getAttribute(null, activity.getBase_Class()).destroy();
        activity.getBase_Class().destroy();
      }
    }
  }

  /**
   * The service definition is used as parameter type for the callback parameters with a suitable ptr
   * declaration that sub-selects one of the contained types. In case of a service definition of query
   * for instance, ROS uses <service def name>::Request::SharedPtr
   * The C++ code generator follows the template binding of the service definition and includes the
   * referenced types. Therefore, we need to remove this binding to avoid this code is generated. Adding
   * a no-code-gen tag to these would not a solution, as the referenced type could actually be used in
   * another message definition.
   */
  public static void removeTemplateSig(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      if (((InteractionUtils.getServiceDefinition(port) != null) && (InteractionUtils.getTemplateBinding(InteractionUtils.getServiceDefinition(port)) != null))) {
        InteractionUtils.getTemplateBinding(InteractionUtils.getServiceDefinition(port)).destroy();
      }
    }
  }

  /**
   * Move functions in passed activity to component level
   */
  public static void liftFunctions(final Activity activity, final org.eclipse.uml2.uml.Class component) {
    EList<Property> _attributes = activity.getBase_Class().getAttributes();
    for (final Property fct : _attributes) {
      Type _type = fct.getType();
      if ((_type instanceof Behavior)) {
        Type _type_1 = fct.getType();
        final Behavior fctType = ((Behavior) _type_1);
        Function fctSt = UMLUtil.<Function>getStereotypeApplication(fctType, Function.class);
        if ((fctSt != null)) {
          Function _copy = EcoreUtil.<Function>copy(fctSt);
          final Function copy = ((Function) _copy);
          component.getOwnedBehaviors().add(fctType);
          fctSt = StereotypeUtil.<Function>applyApp(fctType, Function.class);
          EList<EStructuralFeature> _eStructuralFeatures = fctSt.eClass().getEStructuralFeatures();
          for (final EStructuralFeature feature : _eStructuralFeatures) {
            boolean _isChangeable = feature.isChangeable();
            if (_isChangeable) {
              fctSt.eSet(feature, copy.eGet(feature));
            }
          }
        } else {
          component.getOwnedBehaviors().add(fctType);
        }
        final Operation specification = component.createOwnedOperation(fctType.getName(), null, null);
        fctType.setSpecification(specification);
      }
    }
  }

  /**
   * Remove functions that are not referenced by activities (this can happen after
   * deletion of a function from an activity)
   */
  public static void removeUnrefFunctions(final org.eclipse.uml2.uml.Class component) {
    final ComponentDefinition cd = UMLUtil.<ComponentDefinition>getStereotypeApplication(component, ComponentDefinition.class);
    final ArrayList<Function> fctList = new ArrayList<Function>();
    EList<Activity> _activities = cd.getActivities();
    for (final Activity activity : _activities) {
      fctList.addAll(activity.getFunctions());
    }
    PackageableElement[] _clone = ((PackageableElement[])Conversions.unwrapArray(component.getNearestPackage().getPackagedElements(), PackageableElement.class)).clone();
    for (final PackageableElement pe : _clone) {
      if ((pe instanceof OpaqueBehavior)) {
        final Function fct = UMLUtil.<Function>getStereotypeApplication(pe, Function.class);
        if (((fct != null) && (!fctList.contains(fct)))) {
          ((OpaqueBehavior)pe).destroy();
        }
      }
    }
  }

  /**
   * Remove the port, as well as its type (component service,
   * normally a nested classifier)
   */
  public static void removePorts(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      {
        port.getType().destroy();
        port.destroy();
      }
    }
  }
}
