/*****************************************************************************
 * Copyright (c) 2011 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   CEA LIST - Initial API and implementation
 *   Nicolas FAUVERGUE (ALL4TEC) nicolas.fauvergue@all4tec.net - Bug 496905
 *
 *****************************************************************************/
package org.eclipse.papyrus.sysml.diagram.common.parser;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.papyrus.infra.emf.utils.EMFHelper;
import org.eclipse.papyrus.sysml.diagram.common.preferences.ILabelPreferenceConstants;
import org.eclipse.papyrus.sysml.diagram.common.utils.SysMLMultiplicityElementUtil;
import org.eclipse.papyrus.sysml.portandflows.FlowDirection;
import org.eclipse.papyrus.sysml.portandflows.FlowPort;
import org.eclipse.papyrus.sysml.portandflows.FlowSpecification;
import org.eclipse.papyrus.sysml.portandflows.PortandflowsPackage;
import org.eclipse.papyrus.uml.diagram.common.parser.PropertyLabelParser;
import org.eclipse.papyrus.uml.internationalization.utils.utils.UMLLabelInternationalization;
import org.eclipse.papyrus.uml.tools.utils.ICustomAppearance;
import org.eclipse.papyrus.uml.tools.utils.ValueSpecificationUtil;
import org.eclipse.uml2.uml.InstanceValue;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.VisibilityKind;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * Semantic Parser for {@link FlowPort}
 */
public class FlowPortLabelParser extends PropertyLabelParser {

	/** The String format for displaying a FlowProperty with direction */
	protected static final String DIRECTION_FORMAT = "%s %s";

	/** The String format for displaying a FlowProperty with direction */
	protected static final String CONJUGATED_FORMAT = "~%s";

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPrintString(IAdaptable element, int flags) {

		Collection<String> maskValues = getMaskValues(element);

		if (maskValues.isEmpty()) {
			return MaskedLabel;
		}

		String result = "";
		EObject eObject = EMFHelper.getEObject(element);

		if ((eObject != null) && (eObject instanceof Port)) {

			Property property = (Property) eObject;

			FlowPort flowPort = UMLUtil.getStereotypeApplication(property, FlowPort.class);
			if (flowPort != null) {

				// manage direction only if the FlowPort is type and type is not a FlowSpecification
				if (maskValues.contains(ICustomAppearance.DISP_DIRECTION)) {
					String direction;
					switch (flowPort.getDirection().getValue()) {
					case FlowDirection.IN_VALUE:
						direction = "in";
						break;
					case FlowDirection.OUT_VALUE:
						direction = "out";
						break;
					case FlowDirection.INOUT_VALUE:
						direction = "inout";
						break;
					default:
						direction = "inout";
						break;
					}

					// manage direction only if the FlowPort is not a FlowSpecification
					if ((property.getType() == null) || ((property.getType() != null) && (UMLUtil.getStereotypeApplication(property.getType(), FlowSpecification.class) == null))) {
						result = String.format(DIRECTION_FORMAT, direction, result);
					}
				}
			}

			// manage visibility
			if (maskValues.contains(ICustomAppearance.DISP_VISIBILITY)) {
				String visibility;
				switch (property.getVisibility().getValue()) {
				case VisibilityKind.PACKAGE:
					visibility = "~";
					break;
				case VisibilityKind.PUBLIC:
					visibility = "+";
					break;
				case VisibilityKind.PROTECTED:
					visibility = "#";
					break;
				case VisibilityKind.PRIVATE:
					visibility = "-";
					break;
				default:
					visibility = "+";
					break;
				}
				result = String.format(VISIBILITY_FORMAT, visibility, result);
			}

			// manage derived modifier
			if ((maskValues.contains(ICustomAppearance.DISP_DERIVE)) && (property.isDerived())) {
				result = String.format(DERIVED_FORMAT, result);
			}

			// manage name
			if ((maskValues.contains(ICustomAppearance.DISP_NAME)) && (property.isSetName())) {
				String name = UMLLabelInternationalization.getInstance().getLabel(property);
				result = String.format(NAME_FORMAT, result, name);
			}

			// manage type and conjugated property
			if ((maskValues.contains(ICustomAppearance.DISP_TYPE))) {

				String type = "<Undefined>";
				if (property.getType() != null) {
					type = UMLLabelInternationalization.getInstance().getLabel(property.getType());
				}

				// If type is undefined only show "<Undefined>" when explicitly asked.
				if ((maskValues.contains(ILabelPreferenceConstants.DISP_UNDEFINED_TYPE)) || (!"<Undefined>".equals(type))) {
					if ((flowPort != null) && (flowPort.isConjugated())) {
						type = String.format(CONJUGATED_FORMAT, type);
					}
					result = String.format(TYPE_FORMAT, result, type);
				}
			}

			// manage multiplicity
			if ((maskValues.contains(ICustomAppearance.DISP_MULTIPLICITY))) {

				// TODO : add a case for default with multiplicity not set.
				String multiplicity = SysMLMultiplicityElementUtil.formatMultiplicity(property, maskValues);
				if (multiplicity != null && !multiplicity.isEmpty()) {
					result += " " + multiplicity;
				}
			}

			// manage default value
			if ((maskValues.contains(ICustomAppearance.DISP_DEFAULT_VALUE)) && ((property.getDefaultValue() != null))) {
				ValueSpecification valueSpecification = property.getDefaultValue();
				if (valueSpecification instanceof InstanceValue) {
					result = String.format(DEFAULT_VALUE_FORMAT, result, ValueSpecificationUtil.getSpecificationValue(valueSpecification, true));
				}
			}

			// manage modifier
			if (maskValues.contains(ICustomAppearance.DISP_MODIFIERS)) {
				StringBuffer sb = new StringBuffer();
				if (property.isReadOnly()) {
					sb.append(sb.length() == 0 ? "readOnly" : ", readOnly");
				}
				if (property.isOrdered()) {
					sb.append(sb.length() == 0 ? "ordered" : ", ordered");
				}
				if (property.isUnique()) {
					sb.append(sb.length() == 0 ? "unique" : ", unique");
				}
				if (property.isDerivedUnion()) {
					sb.append(sb.length() == 0 ? "union" : ", union");
				}
				EList<Property> redefinedProperties = property.getRedefinedProperties();
				if (redefinedProperties != null && !redefinedProperties.isEmpty()) {
					for (Property p : redefinedProperties) {
						sb.append(sb.length() == 0 ? UMLLabelInternationalization.getInstance().getLabel(p) : ", redefines " + UMLLabelInternationalization.getInstance().getLabel(p));
					}
				}
				if (sb.length() != 0) {
					result = String.format(MODIFIER_FORMAT, result, sb.toString());
				}
			}
		}

		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isAffectingEvent(Object event, int flags) {

		if (event instanceof Notification) {
			Object feature = ((Notification) event).getFeature();
			if (feature instanceof EStructuralFeature) {
				return PortandflowsPackage.eINSTANCE.getFlowPort_Direction().equals(feature) || PortandflowsPackage.eINSTANCE.getFlowPort_IsConjugated().equals(feature) || super.isAffectingEvent(event, flags);
			}
		}

		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<EObject> getSemanticElementsBeingParsed(EObject element) {
		List<EObject> semanticElementsBeingParsed = super.getSemanticElementsBeingParsed(element);

		if ((element != null) && (element instanceof Port)) {
			Port semElement = (Port) element;

			FlowPort flowPort = UMLUtil.getStereotypeApplication(semElement, FlowPort.class);
			if (flowPort != null) {
				semanticElementsBeingParsed.add(flowPort);
			}
		}
		return semanticElementsBeingParsed;
	}

	@Override
	public Map<String, String> getMasks() {
		Map<String, String> masks = new HashMap<String, String>();
		masks.put(ICustomAppearance.DISP_DIRECTION, "Direction");
		masks.put(ICustomAppearance.DISP_VISIBILITY, "Visibility");
		masks.put(ICustomAppearance.DISP_DERIVE, "Is Derived");
		masks.put(ICustomAppearance.DISP_NAME, "Name");
		masks.put(ICustomAppearance.DISP_TYPE, "Type");
		masks.put(ILabelPreferenceConstants.DISP_UNDEFINED_TYPE, "Show <Undefined> type");
		masks.put(ICustomAppearance.DISP_MULTIPLICITY, "Multiplicity");
		masks.put(ILabelPreferenceConstants.DISP_DEFAULT_MULTIPLICITY, "Show default multiplicity");
		masks.put(ICustomAppearance.DISP_DEFAULT_VALUE, "Default Value");
		masks.put(ICustomAppearance.DISP_MODIFIERS, "Modifiers");
		return masks;
	}

	@Override
	public Collection<String> getDefaultValue(IAdaptable element) {
		return Arrays.asList(ICustomAppearance.DISP_DIRECTION, ICustomAppearance.DISP_NAME, ICustomAppearance.DISP_TYPE, ILabelPreferenceConstants.DISP_UNDEFINED_TYPE);
	}
}
