/**
 * Copyright (c) 2022 CEA LIST.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.papyrus.designer.transformation.library.xtend;

import java.util.List;
import java.util.Objects;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Typedef;
import org.eclipse.papyrus.designer.transformation.extensions.ITextTemplate;
import org.eclipse.papyrus.designer.uml.tools.utils.ElementUtils;
import org.eclipse.papyrus.designer.uml.tools.utils.OperationUtils;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.PrimitiveType;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class CppUtils implements ITextTemplate {
  /**
   * create the C++ signature for an operation (including parenthesis)
   */
  public static CharSequence cppSignature(final Operation operation) {
    StringConcatenation _builder = new StringConcatenation();
    String _name = operation.getName();
    _builder.append(_name);
    _builder.append("(");
    {
      EList<Parameter> _ownedParameters = operation.getOwnedParameters();
      boolean _hasElements = false;
      for(final Parameter parameter : _ownedParameters) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(", ", "");
        }
        _builder.newLineIfNotEmpty();
        CharSequence _cppParameter = CppUtils.cppParameter(parameter);
        _builder.append(_cppParameter);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  /**
   * make a C++ call, pass all parameters except the return parameter
   */
  public static CharSequence cppCall(final Operation operation) {
    StringConcatenation _builder = new StringConcatenation();
    String _name = operation.getName();
    _builder.append(_name);
    _builder.append("(");
    {
      EList<Parameter> _parametersNonRet = OperationUtils.parametersNonRet(operation);
      boolean _hasElements = false;
      for(final Parameter parameter : _parametersNonRet) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(", ", "");
        }
        _builder.newLineIfNotEmpty();
        String _name_1 = parameter.getName();
        _builder.append(_name_1);
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t");
      }
    }
    _builder.append(")");
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  /**
   * @param type
   *            a type
   * @return return the definition of a typedef, if the type has been defined via
   *         the stereotype CppType of the Cpp profile
   */
  public static String dereferenceTypedef(final Type type) {
    if ((type instanceof PrimitiveType)) {
      final Typedef cppType = UMLUtil.<Typedef>getStereotypeApplication(type, Typedef.class);
      if ((cppType != null)) {
        cppType.getDefinition();
      }
    }
    return type.getQualifiedName();
  }

  /**
   * make a C++ call, pass all parameters except the return parameter, prefix with "return",
   * if there is a return type in the operations declaration
   */
  public static CharSequence returnCppCall(final Operation operation) {
    StringConcatenation _builder = new StringConcatenation();
    {
      Type _type = operation.getType();
      boolean _tripleNotEquals = (_type != null);
      if (_tripleNotEquals) {
        _builder.append("return ");
      }
    }
    CharSequence _cppCall = CppUtils.cppCall(operation);
    _builder.append(_cppCall);
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public static CharSequence cppParameter(final Parameter parameter) {
    StringConcatenation _builder = new StringConcatenation();
    Type _type = parameter.getType();
    _builder.append(_type);
    {
      ParameterDirectionKind _direction = parameter.getDirection();
      boolean _equals = Objects.equals(_direction, Integer.valueOf(ParameterDirectionKind.OUT));
      if (_equals) {
        _builder.append("_out");
      }
    }
    _builder.append(" ");
    String _name = parameter.getName();
    _builder.append(_name);
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public static CharSequence cppType(final Type type) {
    StringConcatenation _builder = new StringConcatenation();
    {
      String _qualifiedName = type.getQualifiedName();
      boolean _equals = Objects.equals(_qualifiedName, "UMLPrimitiveTypes::Boolean");
      if (_equals) {
        _builder.append("bool");
        _builder.newLine();
      } else {
        String _qualifiedName_1 = type.getQualifiedName();
        boolean _equals_1 = Objects.equals(_qualifiedName_1, "UMLPrimitiveTypes::Integer");
        if (_equals_1) {
          _builder.append("int");
          _builder.newLine();
        } else {
          String _dereferenceTypedef = CppUtils.dereferenceTypedef(type);
          _builder.append(_dereferenceTypedef);
          _builder.newLineIfNotEmpty();
        }
      }
    }
    return _builder;
  }

  public static CharSequence cppRetType(final Operation operation) {
    StringConcatenation _builder = new StringConcatenation();
    {
      Type _type = operation.getType();
      boolean _tripleEquals = (_type == null);
      if (_tripleEquals) {
        _builder.append("void");
        _builder.newLine();
      } else {
        CharSequence _cppType = CppUtils.cppType(operation.getType());
        _builder.append(_cppType);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  /**
   * Open a set of C++ namespaces associated with the packages of of the passed named element
   * TODO: use indentTab? => requires making this script recursive
   * Need to include referenced types (assuming a naming convention?
   */
  public static CharSequence openNamespace(final NamedElement namedElement) {
    StringConcatenation _builder = new StringConcatenation();
    {
      List<Namespace> _reverse = ListExtensions.<Namespace>reverse(ElementUtils.usedNamespaces(namedElement));
      for(final Namespace ns : _reverse) {
        _builder.append("namespace ");
        String _name = ns.getName();
        _builder.append(_name);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  /**
   * Close a set of C++ namespaces associated with the packages of of the passed named element
   */
  public static CharSequence closeNamespace(final NamedElement namedElement) {
    StringConcatenation _builder = new StringConcatenation();
    {
      EList<Namespace> _usedNamespaces = ElementUtils.usedNamespaces(namedElement);
      for(final Namespace ns : _usedNamespaces) {
        _builder.append("}; // of namespace [ns.name/]");
        _builder.newLine();
      }
    }
    return _builder;
  }
}
