/**
 * Copyright (c) 2022, 2023 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Mohamed Harkat - Initial API and implementation
 *   Ansgar Radermacher - Integration and bug fixes
 */
package org.eclipse.papyrus.designer.languages.python.codegen.gen;

import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.python.codegen.transformation.PythonCodeGenUtils;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.xtend2.lib.StringConcatenation;

@SuppressWarnings("all")
public class PyEnumeration {
  /**
   * A simple code for enumeration generation. It assigns for each literal a number starting from zero.
   */
  public static CharSequence genEnumeration(final Enumeration enumeration, final boolean useRelativeImports) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("### Enumeration ###");
    _builder.newLine();
    _builder.append("from enum import Enum");
    _builder.newLine();
    CharSequence _writeImports = PythonCodeGenUtils.writeImports(enumeration, useRelativeImports);
    _builder.append(_writeImports);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.newLine();
    int i = 0;
    _builder.newLineIfNotEmpty();
    _builder.append("class ");
    String _name = enumeration.getName();
    _builder.append(_name);
    _builder.append("(");
    {
      boolean _isEmpty = enumeration.getGenerals().isEmpty();
      if (_isEmpty) {
        _builder.append("Enum");
      } else {
        {
          EList<Classifier> _generals = enumeration.getGenerals();
          boolean _hasElements = false;
          for(final Classifier superEnumeration : _generals) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "");
            }
            String _name_1 = superEnumeration.getName();
            _builder.append(_name_1);
          }
        }
      }
    }
    _builder.append("):");
    _builder.newLineIfNotEmpty();
    {
      boolean _isEmpty_1 = enumeration.getOwnedLiterals().isEmpty();
      if (_isEmpty_1) {
        _builder.append("\t");
        _builder.append("pass");
        _builder.newLine();
      }
    }
    _builder.newLine();
    {
      EList<EnumerationLiteral> _ownedLiterals = enumeration.getOwnedLiterals();
      for(final EnumerationLiteral literal : _ownedLiterals) {
        {
          ValueSpecification _specification = literal.getSpecification();
          boolean _tripleNotEquals = (_specification != null);
          if (_tripleNotEquals) {
            _builder.append("\t");
            String _name_2 = literal.getName();
            _builder.append(_name_2, "\t");
            _builder.append(" = ");
            String _string = Integer.valueOf(literal.getSpecification().integerValue()).toString();
            _builder.append(_string, "\t");
            _builder.newLineIfNotEmpty();
          } else {
            _builder.append("\t");
            while (PyEnumeration.checkValueParentEnum(enumeration, i)) {
              i++;
            }
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            String _name_3 = literal.getName();
            _builder.append(_name_3, "\t");
            _builder.append(" = ");
            int _plusPlus = i++;
            _builder.append(_plusPlus, "\t");
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    return _builder;
  }

  /**
   * A method that assigns a value to the literal. If the value of the literal is supplied by the user, then return
   * that value, else assign a different value to the literal. This method needs to check all the assigned values
   * to generate a new one.
   */
  public static int assignLiteralValue() {
    return 0;
  }

  public static boolean checkValueParentEnum(final Enumeration enumeration, final int value) {
    EList<Classifier> _generals = enumeration.getGenerals();
    for (final Classifier general : _generals) {
      if ((general instanceof Enumeration)) {
        EList<EnumerationLiteral> _ownedLiterals = ((Enumeration)general).getOwnedLiterals();
        for (final EnumerationLiteral literal : _ownedLiterals) {
          if (((literal.getSpecification() != null) && (value == literal.getSpecification().integerValue()))) {
            return true;
          }
        }
      }
    }
    return false;
  }
}
