/*****************************************************************************
 * Copyright (c) 2013 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 *
 *****************************************************************************/

package org.eclipse.papyrus.designer.components.validation.constraints;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.validation.AbstractModelConstraint;
import org.eclipse.emf.validation.IValidationContext;
import org.eclipse.papyrus.uml.tools.utils.ConnectorUtil;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;

/**
 * Check whether a port of a part remains without connection. Whereas it is typically problematic
 * for ports with a required interface, we also raise an info for those with a provided interface
 *
 */
abstract public class UnconnectedPorts extends AbstractModelConstraint
{
	/**
	 * Called by one of the sub-classes (does not override a superclass method)
	 *
	 * @param ctx
	 *            the validation context
	 * @param required
	 *            if true, examine required ports, if false provided ports
	 * @return the validation status
	 */
	public IStatus validatePorts(IValidationContext ctx, boolean required)
	{
		String portsStr = ""; //$NON-NLS-1$

		Property attribute = (Property) ctx.getTarget();
		Class owner = attribute.getClass_();
		if (owner != null) {
			if (attribute.getType() instanceof Class) {
				Class class_ = (Class) attribute.getType();
				for (Port port : class_.getOwnedPorts()) {
					if ((port.getRequireds().size() > 0 && required) ||
							(port.getProvideds().size() > 0 && !required)) {
						{
							boolean found = false;
							for (Connector connector : owner.getOwnedConnectors()) {
								if (ConnectorUtil.connectsPort(connector, port)) {
									found = true;
									break;
								}
							}
							if (!found) {
								if (portsStr.length() != 0) {
									portsStr += ", "; //$NON-NLS-1$
								}
								portsStr += port.getName();
							}
						}
					}
				}
			}
		}
		if (portsStr.length() > 0) {
			return ctx.createFailureStatus(String.format(
					"The port(s) '%s' of part '%s' remains unconnected within composite class '%s'", //$NON-NLS-1$
					portsStr, attribute.getName(), owner.getQualifiedName()));
		}
		else {
			return ctx.createSuccessStatus();
		}
	}
}
