/*******************************************************************************
 * Copyright (c) 2006 - 2016 CEA LIST.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Shuai Li (CEA LIST) <shuai.li@cea.fr> - initial API and implementation
 *******************************************************************************/
 
package org.eclipse.papyrus.designer.languages.java.codegen.xtend

import java.util.ArrayList
import java.util.List
import org.eclipse.emf.common.util.UniqueEList
import org.eclipse.papyrus.designer.languages.common.base.GenUtils
import org.eclipse.papyrus.designer.languages.java.codegen.utils.JavaGenUtils
import org.eclipse.papyrus.designer.languages.java.profile.PapyrusJava.ExternLibrary
import org.eclipse.papyrus.designer.languages.java.profile.PapyrusJava.External
import org.eclipse.papyrus.designer.languages.java.profile.PapyrusJava.Template
import org.eclipse.uml2.uml.Classifier
import org.eclipse.uml2.uml.Element
import org.eclipse.uml2.uml.Enumeration
import org.eclipse.uml2.uml.Interface
import org.eclipse.uml2.uml.NamedElement
import org.eclipse.uml2.uml.Namespace
import org.eclipse.uml2.uml.Package
import org.eclipse.uml2.uml.UMLFactory
import org.eclipse.uml2.uml.util.UMLUtil
import org.eclipse.uml2.uml.TemplateParameterSubstitution

class JavaClassImportDeclaration {
	
	static def javaClassImports(NamedElement ne, Namespace ns) {
		var List<String> result = new ArrayList<String>()
		if (GenUtils.hasStereotypeTree(ne, ExternLibrary)) {
			result = GenUtils.getApplicationTree(ne, ExternLibrary).imports
		} else {
			result.add(JavaClassImportDeclaration.importName(ne, ns))
		}
		return result
	}
	
	/**
	 * @param ne the element to import
	 * @param ns the element in which we import ne
	 */
	static def importName(NamedElement ne, Namespace ns) {
		if (GenUtils.hasStereotypeTree(ne, Template)) {
			return UMLUtil.getStereotypeApplication(ne, Template).declaration
		} else {
			if (GenUtils.hasStereotypeTree(ne, External)) {
				return UMLUtil.getStereotypeApplication(ne, External).name
			} else {
				// ne is owned by a template parameter substitution and it is not stereotyped external. This is an invalid model.
				if (ne.owner instanceof TemplateParameterSubstitution) {
					return ""
				}
				
				// Never import ne if its short name is the same as the name of an inner class directly owned by the current ns
				// We will use the full name of ne later on
				for (element : ns.ownedElements) {
					if (element instanceof Enumeration || element instanceof Interface || element.eClass.equals(UMLFactory.eINSTANCE.getUMLPackage().getClass_())) {
						if ((element as Classifier).name.equals(ne.name)) {
							return ""
						}
					}
				}
				
				// Find if there is an import of ns, with the same class name and only import it if no so
				var classifier = ns
				if (!(ns.owner instanceof Package)) {
					classifier = getPackagedClassifier(ns);
					if (classifier == null) {
						return ""
					}
				}
				
				var importsOfClassifier = JavaGenUtils.imports.get(classifier);
				
				if (importsOfClassifier == null) {
					importsOfClassifier = new UniqueEList<String>()
					JavaGenUtils.imports.put(classifier, importsOfClassifier)
				} else {
					for (import : importsOfClassifier) {
						val String[] parts = import.split("\\.")
						if (parts.length > 0) {
							val String lastPart = parts.get(parts.length - 1)
							if (ne.name.equals(lastPart)) {
								return "" // Empty imports are not generated by JavaImportUtil.importDirective
							}
						}
					}
				}
				
				val qName = GenUtils.getFullName(ne, ".", false)
				importsOfClassifier.add(qName)
				return qName
			}
		}
	}
	
	private static def Classifier getPackagedClassifier(Element ns) {
		if (ns instanceof Classifier) {
			return (ns as Classifier)
		} else {
			if (ns.owner == null) {
				return null
			}
			return getPackagedClassifier(ns.owner)
		}
	}
}