/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.utils;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.Block;
import org.eclipse.jdt.core.dom.CastExpression;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.EnhancedForStatement;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.FieldDeclaration;
import org.eclipse.jdt.core.dom.ForStatement;
import org.eclipse.jdt.core.dom.ImportDeclaration;
import org.eclipse.jdt.core.dom.MemberValuePair;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.MethodInvocation;
import org.eclipse.jdt.core.dom.Name;
import org.eclipse.jdt.core.dom.NameQualifiedType;
import org.eclipse.jdt.core.dom.NormalAnnotation;
import org.eclipse.jdt.core.dom.QualifiedName;
import org.eclipse.jdt.core.dom.QualifiedType;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.SimpleType;
import org.eclipse.jdt.core.dom.SingleVariableDeclaration;
import org.eclipse.jdt.core.dom.Statement;
import org.eclipse.jdt.core.dom.Type;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jdt.core.dom.VariableDeclarationExpression;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;
import org.eclipse.jdt.core.dom.VariableDeclarationStatement;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class RenamerVisitor extends ASTVisitor {
  private RenamingRuleManager rulesManager;

  private Map<ImportDeclaration, Name> newImportsNames;

  private Map<SimpleType, Name> newSimpleTypesNames;

  private Map<MethodInvocation, SimpleName> newMethodNames;

  private Map<MethodDeclaration, SimpleName> newMethodDeclNames;

  private List<EClass> allClasses;

  public RenamerVisitor(final RenamingRuleManager rulesManager, final List<EClass> allClasses) {
    this.rulesManager = rulesManager;
    this.newImportsNames = CollectionLiterals.<ImportDeclaration, Name>newHashMap();
    this.newSimpleTypesNames = CollectionLiterals.<SimpleType, Name>newHashMap();
    this.newMethodNames = CollectionLiterals.<MethodInvocation, SimpleName>newHashMap();
    this.newMethodDeclNames = CollectionLiterals.<MethodDeclaration, SimpleName>newHashMap();
    this.allClasses = allClasses;
  }

  @Override
  public boolean visit(final ImportDeclaration node) {
    final String nodeName = node.getName().toString();
    String newNodeName = null;
    boolean _endsWith = nodeName.endsWith(".*");
    if (_endsWith) {
      int _length = nodeName.length();
      int _minus = (_length - 2);
      final String importedPackage = nodeName.substring(0, _minus);
      String _directRenaming = this.getDirectRenaming(importedPackage);
      String _plus = (_directRenaming + ".*");
      newNodeName = _plus;
    } else {
      final Function1<Pair<String, String>, Boolean> _function = (Pair<String, String> it) -> {
        String _key = it.getKey();
        return Boolean.valueOf(Objects.equal(_key, nodeName));
      };
      final Pair<String, String> rule = IterableExtensions.<Pair<String, String>>findFirst(this.rulesManager.getAllClassRules(), _function);
      if ((rule != null)) {
        newNodeName = rule.getValue();
      } else {
        final String packageName = this.getQualifierPart(nodeName);
        final String newPackageName = this.getDirectRenaming(packageName);
        if ((newPackageName != null)) {
          final String className = this.getLastPart(nodeName);
          newNodeName = ((newPackageName + ".") + className);
        }
      }
    }
    if ((newNodeName != null)) {
      final Name newName = this.toName(node.getAST(), newNodeName);
      this.newImportsNames.put(node, newName);
    }
    return super.visit(node);
  }

  @Override
  public boolean visit(final SimpleType node) {
    final Name typeName = node.getName();
    final Pair<String, String> rule = this.getClassRule(typeName);
    if ((rule != null)) {
      if ((typeName instanceof SimpleName)) {
        final String toName = this.getLastPart(rule.getValue());
        final SimpleName newName = node.getAST().newSimpleName(toName);
        this.newSimpleTypesNames.put(node, newName);
      } else {
        if ((typeName instanceof QualifiedName)) {
          final Name newName_1 = this.toName(node.getAST(), rule.getValue());
          this.newSimpleTypesNames.put(node, newName_1);
        }
      }
    } else {
      if ((typeName instanceof QualifiedName)) {
        final String packagePart = ((QualifiedName)typeName).toString().substring(0, ((QualifiedName)typeName).toString().lastIndexOf("."));
        String _string = ((QualifiedName)typeName).toString();
        int _lastIndexOf = ((QualifiedName)typeName).toString().lastIndexOf(".");
        int _plus = (_lastIndexOf + 1);
        final String simpleName = _string.substring(_plus);
        String _renaming = this.getRenaming(packagePart);
        String _plus_1 = (_renaming + ".");
        final String renamedType = (_plus_1 + simpleName);
        final Function1<EClass, Boolean> _function = (EClass it) -> {
          return Boolean.valueOf(renamedType.endsWith(this.getFqn(it)));
        };
        final EClass candidate = IterableExtensions.<EClass>findFirst(this.allClasses, _function);
        if ((candidate != null)) {
          final Name newName_2 = this.toName(node.getAST(), renamedType);
          this.newSimpleTypesNames.put(node, newName_2);
        }
      }
    }
    return super.visit(node);
  }

  @Override
  public boolean visit(final MethodInvocation node) {
    final SimpleName methodName = node.getName();
    final Function1<Pair<String, String>, Boolean> _function = (Pair<String, String> it) -> {
      return Boolean.valueOf((Objects.equal(("get" + StringExtensions.toFirstUpper(this.getLastPart(it.getKey()))), methodName.toString()) || 
        Objects.equal(("set" + StringExtensions.toFirstUpper(this.getLastPart(it.getKey()))), methodName.toString())));
    };
    final Iterable<Pair<String, String>> candidateRules = IterableExtensions.<Pair<String, String>>filter(this.rulesManager.getAllPropertyRules(), _function);
    final Expression exp = node.getExpression();
    if ((exp instanceof Name)) {
      final Name invokerName = ((Name)exp);
      final Set<Pair<String, String>> aspectEntry = this.rulesManager.getRulesForAspect(invokerName.toString());
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(aspectEntry);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        final Function1<Pair<String, String>, Boolean> _function_1 = (Pair<String, String> rule) -> {
          String _lastPart = this.getLastPart(rule.getKey());
          String _string = methodName.toString();
          return Boolean.valueOf(Objects.equal(_lastPart, _string));
        };
        final Pair<String, String> rule = IterableExtensions.<Pair<String, String>>findFirst(aspectEntry, _function_1);
        if ((rule != null)) {
          final String newName = this.getLastPart(rule.getValue());
          this.newMethodNames.put(node, node.getAST().newSimpleName(newName));
        }
      } else {
        final String typeName = this.getType(invokerName);
        final Function1<Pair<String, String>, Boolean> _function_2 = (Pair<String, String> rule_1) -> {
          return Boolean.valueOf((Objects.equal(this.getQualifierPart(rule_1.getKey()), typeName) || 
            Objects.equal(this.getLastPart(this.getQualifierPart(rule_1.getKey())), typeName)));
        };
        final Iterable<Pair<String, String>> targetedRules = IterableExtensions.<Pair<String, String>>filter(candidateRules, _function_2);
        final Function1<Pair<String, String>, Boolean> _function_3 = (Pair<String, String> rule_1) -> {
          final String clazz = this.getQualifierPart(rule_1.getKey());
          ASTNode _root = methodName.getRoot();
          return Boolean.valueOf(this.isImported(((CompilationUnit) _root), clazz));
        };
        final Pair<String, String> importedRule = IterableExtensions.<Pair<String, String>>findFirst(targetedRules, _function_3);
        if ((importedRule != null)) {
          final String prefix = methodName.toString().substring(0, 3);
          String _firstUpper = StringExtensions.toFirstUpper(this.getLastPart(importedRule.getValue()));
          final String newName_1 = (prefix + _firstUpper);
          this.newMethodNames.put(node, node.getAST().newSimpleName(newName_1));
        }
      }
    } else {
      if ((exp instanceof CastExpression)) {
      } else {
      }
    }
    return super.visit(node);
  }

  @Override
  public boolean visit(final MethodDeclaration node) {
    final String methodName = node.getName().toString();
    final Function1<Pair<String, String>, Boolean> _function = (Pair<String, String> rule) -> {
      String _lastPart = this.getLastPart(rule.getKey());
      return Boolean.valueOf(Objects.equal(_lastPart, methodName));
    };
    final Iterable<Pair<String, String>> candidateRules = IterableExtensions.<Pair<String, String>>filter(this.rulesManager.getAllPropertyRules(), _function);
    final ASTNode container = node.getParent();
    if ((container instanceof TypeDeclaration)) {
      final TypeDeclaration clazz = ((TypeDeclaration)container);
      final Function1<NormalAnnotation, Boolean> _function_1 = (NormalAnnotation it) -> {
        String _string = it.getTypeName().toString();
        return Boolean.valueOf(Objects.equal(_string, "Aspect"));
      };
      final NormalAnnotation aspectAnnot = IterableExtensions.<NormalAnnotation>findFirst(Iterables.<NormalAnnotation>filter(clazz.modifiers(), NormalAnnotation.class), _function_1);
      if ((aspectAnnot != null)) {
        Object _get = aspectAnnot.values().get(0);
        final String aspectedClazz = this.getQualifierPart(((MemberValuePair) _get).getValue().toString());
        final Function1<Pair<String, String>, Boolean> _function_2 = (Pair<String, String> rule) -> {
          boolean _xblockexpression = false;
          {
            final String candidateClazz = this.getQualifierPart(rule.getKey());
            _xblockexpression = (Objects.equal(this.getLastPart(candidateClazz), aspectedClazz) && 
              this.isImported(((CompilationUnit) node.getRoot()), candidateClazz));
          }
          return Boolean.valueOf(_xblockexpression);
        };
        final Pair<String, String> rule = IterableExtensions.<Pair<String, String>>findFirst(candidateRules, _function_2);
        if ((rule != null)) {
          final String newName = this.getLastPart(rule.getValue());
          this.newMethodDeclNames.put(node, node.getAST().newSimpleName(newName));
        }
      }
    }
    return super.visit(node);
  }

  /**
   * Update the AST after all visits
   */
  @Override
  public void postVisit(final ASTNode node) {
    if ((node instanceof CompilationUnit)) {
      final Consumer<Map.Entry<ImportDeclaration, Name>> _function = (Map.Entry<ImportDeclaration, Name> entry) -> {
        ImportDeclaration _key = entry.getKey();
        _key.setName(entry.getValue());
      };
      this.newImportsNames.entrySet().forEach(_function);
      final Consumer<Map.Entry<SimpleType, Name>> _function_1 = (Map.Entry<SimpleType, Name> entry) -> {
        SimpleType _key = entry.getKey();
        _key.setName(entry.getValue());
      };
      this.newSimpleTypesNames.entrySet().forEach(_function_1);
      final Consumer<Map.Entry<MethodInvocation, SimpleName>> _function_2 = (Map.Entry<MethodInvocation, SimpleName> entry) -> {
        MethodInvocation _key = entry.getKey();
        _key.setName(entry.getValue());
      };
      this.newMethodNames.entrySet().forEach(_function_2);
      final Consumer<Map.Entry<MethodDeclaration, SimpleName>> _function_3 = (Map.Entry<MethodDeclaration, SimpleName> entry) -> {
        MethodDeclaration _key = entry.getKey();
        _key.setName(entry.getValue());
      };
      this.newMethodDeclNames.entrySet().forEach(_function_3);
    }
    super.postVisit(node);
  }

  /**
   * Return the type in the declaration of {@link variable}.
   * Return null if can't find any declaration
   */
  public String getType(final Name variable) {
    return this.getType(variable, variable.getParent());
  }

  /**
   * Search if {@link container} (or its parents) defines {@link variable} and return
   * the corresponding type.
   * 
   * Return null if not found
   */
  public String getType(final Name variable, final ASTNode container) {
    if ((container == null)) {
      return null;
    } else {
      if ((container instanceof Expression)) {
        return this.getType(variable, ((Expression)container).getParent());
      } else {
        if ((container instanceof Statement)) {
          if ((container instanceof Block)) {
            final Block block = ((Block)container);
            final Function1<VariableDeclarationStatement, Boolean> _function = (VariableDeclarationStatement varDecl) -> {
              final Function1<Object, Boolean> _function_1 = (Object va) -> {
                String _string = ((VariableDeclarationFragment) va).getName().toString();
                String _string_1 = variable.toString();
                return Boolean.valueOf(Objects.equal(_string, _string_1));
              };
              return Boolean.valueOf(IterableExtensions.<Object>exists(varDecl.fragments(), _function_1));
            };
            final VariableDeclarationStatement varDef = IterableExtensions.<VariableDeclarationStatement>findFirst(Iterables.<VariableDeclarationStatement>filter(block.statements(), VariableDeclarationStatement.class), _function);
            if (((varDef != null) && (varDef.getStartPosition() < variable.getStartPosition()))) {
              if ((varDef != null)) {
                final Type type = varDef.getType();
                if ((type instanceof NameQualifiedType)) {
                  return ((NameQualifiedType)type).getName().toString();
                } else {
                  if ((type instanceof QualifiedType)) {
                    return ((QualifiedType)type).getName().toString();
                  } else {
                    if ((type instanceof SimpleType)) {
                      return ((SimpleType)type).getName().toString();
                    }
                  }
                }
              }
            }
          } else {
            if ((container instanceof EnhancedForStatement)) {
              final EnhancedForStatement forLoop = ((EnhancedForStatement)container);
              SimpleName _name = forLoop.getParameter().getName();
              boolean _equals = Objects.equal(_name, variable);
              if (_equals) {
                final Type type_1 = forLoop.getParameter().getType();
                if ((type_1 instanceof NameQualifiedType)) {
                  return ((NameQualifiedType)type_1).getName().toString();
                } else {
                  if ((type_1 instanceof QualifiedType)) {
                    return ((QualifiedType)type_1).getName().toString();
                  } else {
                    if ((type_1 instanceof SimpleType)) {
                      return ((SimpleType)type_1).getName().toString();
                    }
                  }
                }
              }
            } else {
              if ((container instanceof ForStatement)) {
                final ForStatement forLoop_1 = ((ForStatement)container);
                final Function1<VariableDeclarationExpression, Boolean> _function_1 = (VariableDeclarationExpression varDecl) -> {
                  final Function1<Object, Boolean> _function_2 = (Object va) -> {
                    String _string = ((VariableDeclarationFragment) va).getName().toString();
                    String _string_1 = variable.toString();
                    return Boolean.valueOf(Objects.equal(_string, _string_1));
                  };
                  return Boolean.valueOf(IterableExtensions.<Object>exists(varDecl.fragments(), _function_2));
                };
                final VariableDeclarationExpression initDecl = IterableExtensions.<VariableDeclarationExpression>findFirst(Iterables.<VariableDeclarationExpression>filter(forLoop_1.initializers(), VariableDeclarationExpression.class), _function_1);
                if ((initDecl != null)) {
                  final Type type_2 = initDecl.getType();
                  if ((type_2 instanceof NameQualifiedType)) {
                    return ((NameQualifiedType)type_2).getName().toString();
                  } else {
                    if ((type_2 instanceof QualifiedType)) {
                      return ((QualifiedType)type_2).getName().toString();
                    } else {
                      if ((type_2 instanceof SimpleType)) {
                        return ((SimpleType)type_2).getName().toString();
                      }
                    }
                  }
                }
              }
            }
          }
          return this.getType(variable, ((Statement)container).getParent());
        } else {
          if ((container instanceof MethodDeclaration)) {
            final MethodDeclaration method = ((MethodDeclaration)container);
            final Function1<Object, Boolean> _function_2 = (Object paramDecl) -> {
              String _string = ((SingleVariableDeclaration) paramDecl).getName().toString();
              String _string_1 = variable.toString();
              return Boolean.valueOf(Objects.equal(_string, _string_1));
            };
            final Object param = IterableExtensions.<Object>findFirst(method.parameters(), _function_2);
            if ((param != null)) {
              final Type type_3 = ((SingleVariableDeclaration) param).getType();
              if ((type_3 instanceof NameQualifiedType)) {
                return ((NameQualifiedType)type_3).getName().toString();
              } else {
                if ((type_3 instanceof QualifiedType)) {
                  return ((QualifiedType)type_3).getName().toString();
                } else {
                  if ((type_3 instanceof SimpleType)) {
                    return ((SimpleType)type_3).getName().toString();
                  }
                }
              }
            }
            return this.getType(variable, ((MethodDeclaration)container).getParent());
          } else {
            if ((container instanceof TypeDeclaration)) {
              final TypeDeclaration typeDef = ((TypeDeclaration)container);
              final Function1<FieldDeclaration, Boolean> _function_3 = (FieldDeclaration fieldDecl) -> {
                boolean _xblockexpression = false;
                {
                  final List fragments = fieldDecl.fragments();
                  final Function1<Object, Boolean> _function_4 = (Object field) -> {
                    String _string = ((VariableDeclarationFragment) field).getName().toString();
                    String _string_1 = variable.toString();
                    return Boolean.valueOf(Objects.equal(_string, _string_1));
                  };
                  _xblockexpression = IterableExtensions.<Object>exists(fragments, _function_4);
                }
                return Boolean.valueOf(_xblockexpression);
              };
              final FieldDeclaration fieldDef = IterableExtensions.<FieldDeclaration>findFirst(((Iterable<FieldDeclaration>)Conversions.doWrapArray(typeDef.getFields())), _function_3);
              if ((fieldDef != null)) {
                final Type type_4 = fieldDef.getType();
                if ((type_4 instanceof NameQualifiedType)) {
                  return ((NameQualifiedType)type_4).getName().toString();
                } else {
                  if ((type_4 instanceof QualifiedType)) {
                    return ((QualifiedType)type_4).getName().toString();
                  } else {
                    if ((type_4 instanceof SimpleType)) {
                      return ((SimpleType)type_4).getName().toString();
                    }
                  }
                }
              }
              return this.getType(variable, ((TypeDeclaration)container).getParent());
            }
          }
        }
      }
    }
    return this.getType(variable, container.getParent());
  }

  /**
   * Return the substring after the last '.'
   */
  public String getLastPart(final String qualifiedName) {
    int _lastIndexOf = qualifiedName.lastIndexOf(".");
    int _plus = (_lastIndexOf + 1);
    return qualifiedName.substring(_plus);
  }

  /**
   * Return the substring before the last '.'
   */
  public String getQualifierPart(final String qualifiedName) {
    return qualifiedName.substring(0, qualifiedName.lastIndexOf("."));
  }

  /**
   * Return a ClassRule for {@link type}.
   * Return null if {@link type} is not renamed.
   */
  public Pair<String, String> getClassRule(final Name type) {
    Pair<String, String> res = null;
    final String typeName = type.toString();
    final Pair<String, String> rule = this.rulesManager.getClassRule(typeName);
    if ((rule != null)) {
      return rule;
    } else {
      ASTNode _root = type.getRoot();
      final Function1<Object, Name> _function = (Object it) -> {
        return ((ImportDeclaration) it).getName();
      };
      final List<Name> importDecl = ListExtensions.<Object, Name>map(((CompilationUnit) _root).imports(), _function);
      final Function1<Pair<String, String>, Boolean> _function_1 = (Pair<String, String> it) -> {
        String _lastPart = this.getLastPart(it.getKey());
        return Boolean.valueOf(Objects.equal(_lastPart, typeName));
      };
      final Iterable<Pair<String, String>> candidatesRule = IterableExtensions.<Pair<String, String>>filter(this.rulesManager.getAllClassRules(), _function_1);
      Pair<String, String> _findFirst = null;
      if (candidatesRule!=null) {
        final Function1<Pair<String, String>, Boolean> _function_2 = (Pair<String, String> candidateRule) -> {
          boolean _xifexpression = false;
          final Function1<Name, Boolean> _function_3 = (Name it) -> {
            String _fullyQualifiedName = it.getFullyQualifiedName();
            String _key = candidateRule.getKey();
            return Boolean.valueOf(Objects.equal(_fullyQualifiedName, _key));
          };
          boolean _exists = IterableExtensions.<Name>exists(importDecl, _function_3);
          if (_exists) {
            _xifexpression = true;
          } else {
            String _qualifierPart = this.getQualifierPart(candidateRule.getKey());
            final String candidatePackage = (_qualifierPart + ".*");
            final Function1<Name, Boolean> _function_4 = (Name it) -> {
              String _fullyQualifiedName = it.getFullyQualifiedName();
              return Boolean.valueOf(Objects.equal(_fullyQualifiedName, candidatePackage));
            };
            return Boolean.valueOf(IterableExtensions.<Name>exists(importDecl, _function_4));
          }
          return Boolean.valueOf(_xifexpression);
        };
        _findFirst=IterableExtensions.<Pair<String, String>>findFirst(candidatesRule, _function_2);
      }
      res = _findFirst;
    }
    return res;
  }

  /**
   * Parse {@link qualifiedName} and produce a SimpleName
   * or a QualifiedName
   */
  public Name toName(final AST ast, final String qualifiedName) {
    final Function1<String, SimpleName> _function = (String it) -> {
      return ast.newSimpleName(it);
    };
    final List<SimpleName> simpleNames = ListExtensions.<String, SimpleName>map(((List<String>)Conversions.doWrapArray(qualifiedName.split("\\."))), _function);
    Name currentName = simpleNames.get(0);
    for (int i = 1; (i < simpleNames.size()); i++) {
      currentName = ast.newQualifiedName(currentName, simpleNames.get(i));
    }
    return currentName;
  }

  /**
   * Return true if the qualified class {@link candidateImport} is imported
   */
  public boolean isImported(final CompilationUnit root, final String candidateImport) {
    final Function1<Object, Name> _function = (Object it) -> {
      return ((ImportDeclaration) it).getName();
    };
    final List<Name> importDecl = ListExtensions.<Object, Name>map(root.imports(), _function);
    final Function1<Name, Boolean> _function_1 = (Name it) -> {
      String _fullyQualifiedName = it.getFullyQualifiedName();
      return Boolean.valueOf(Objects.equal(_fullyQualifiedName, candidateImport));
    };
    boolean _exists = IterableExtensions.<Name>exists(importDecl, _function_1);
    if (_exists) {
      return true;
    } else {
      final String candidatePackage = this.getQualifierPart(candidateImport);
      final Function1<Name, Boolean> _function_2 = (Name it) -> {
        String _fullyQualifiedName = it.getFullyQualifiedName();
        return Boolean.valueOf(Objects.equal(_fullyQualifiedName, (candidatePackage + ".*")));
      };
      boolean _exists_1 = IterableExtensions.<Name>exists(importDecl, _function_2);
      if (_exists_1) {
        return true;
      }
    }
    return false;
  }

  /**
   * Return true if {@link subpack} is a subpackage of {@link pack}
   */
  public boolean isSubpackage(final String pack, final String subpack) {
    return (((subpack.length() > pack.length()) && 
      subpack.startsWith(pack)) && 
      Objects.equal(Character.valueOf(subpack.charAt(pack.length())).toString(), "."));
  }

  /**
   * Return the renamed package name if their exist a rule for {@link packageName}
   * will NOT apply for a super package.
   * 
   * Return null if not renamed.
   */
  public String getDirectRenaming(final String packageName) {
    final Function1<Pair<String, String>, Boolean> _function = (Pair<String, String> it) -> {
      String _key = it.getKey();
      return Boolean.valueOf(Objects.equal(_key, packageName));
    };
    final Pair<String, String> rule = IterableExtensions.<Pair<String, String>>findFirst(this.rulesManager.getAllPackageRules(), _function);
    if ((rule != null)) {
      return rule.getValue();
    }
    return null;
  }

  /**
   * Return the renamed package name if their exist a rule for {@link packageName}
   * or for a super package.
   * 
   * Return null if not renamed.
   */
  public String getRenaming(final String packageName) {
    final Function1<Pair<String, String>, Boolean> _function = (Pair<String, String> it) -> {
      String _key = it.getKey();
      return Boolean.valueOf(Objects.equal(_key, packageName));
    };
    final Pair<String, String> rule = IterableExtensions.<Pair<String, String>>findFirst(this.rulesManager.getAllPackageRules(), _function);
    if ((rule != null)) {
      return rule.getValue();
    } else {
      final Function1<Pair<String, String>, Boolean> _function_1 = (Pair<String, String> it) -> {
        return Boolean.valueOf(this.isSubpackage(it.getKey(), packageName));
      };
      final Function1<Pair<String, String>, Integer> _function_2 = (Pair<String, String> it) -> {
        return Integer.valueOf(it.getKey().length());
      };
      final Pair<String, String> longestRule = IterableExtensions.<Pair<String, String>>last(IterableExtensions.<Pair<String, String>, Integer>sortBy(IterableExtensions.<Pair<String, String>>filter(this.rulesManager.getAllPackageRules(), _function_1), _function_2));
      if ((longestRule != null)) {
        final String prefix = longestRule.getValue();
        final String sufix = packageName.substring(longestRule.getKey().length());
        return (prefix + sufix);
      }
    }
    return null;
  }

  /**
   * Return the fully qualified name of clazz
   */
  public String getFqn(final EClass clazz) {
    final List<String> parts = CollectionLiterals.<String>newArrayList();
    parts.add(clazz.getName());
    EPackage current = clazz.getEPackage();
    while ((current != null)) {
      {
        parts.add(current.getName());
        current = current.getESuperPackage();
      }
    }
    return IterableExtensions.join(ListExtensions.<String>reverse(parts), ".");
  }
}
