package org.bouncycastle.pqc.legacy.crypto.test;

import java.security.SecureRandom;
import java.util.Random;

import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.digests.SHA256Digest;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.pqc.crypto.util.PrivateKeyFactory;
import org.bouncycastle.pqc.legacy.crypto.mceliece.McElieceCipher;
import org.bouncycastle.pqc.legacy.crypto.mceliece.McElieceKeyGenerationParameters;
import org.bouncycastle.pqc.legacy.crypto.mceliece.McElieceKeyPairGenerator;
import org.bouncycastle.pqc.legacy.crypto.mceliece.McElieceKobaraImaiCipher;
import org.bouncycastle.pqc.legacy.crypto.mceliece.McElieceParameters;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.encoders.Base64;
import org.bouncycastle.util.test.SimpleTest;

public class McElieceCipherTest
    extends SimpleTest
{

    SecureRandom keyRandom = new SecureRandom();

    public String getName()
    {
        return "McEliecePKCS";
    }

    public void performTest()
        throws Exception
    {
        leadingZeroTest();

        int numPassesKPG = 1;
        int numPassesEncDec = 10;
        Random rand = new Random();
        byte[] mBytes;
        for (int j = 0; j < numPassesKPG; j++)
        {

            McElieceParameters params = new McElieceParameters();
            McElieceKeyPairGenerator mcElieceKeyGen = new McElieceKeyPairGenerator();
            McElieceKeyGenerationParameters genParam = new McElieceKeyGenerationParameters(keyRandom, params);

            mcElieceKeyGen.init(genParam);
            AsymmetricCipherKeyPair pair = mcElieceKeyGen.generateKeyPair();

            ParametersWithRandom param = new ParametersWithRandom(pair.getPublic(), keyRandom);
            Digest msgDigest = new SHA256Digest();
            McElieceCipher mcEliecePKCSDigestCipher = new McElieceCipher();


            for (int k = 1; k <= numPassesEncDec; k++)
            {
                System.out.println("############### test: " + k);
                // initialize for encryption
                mcEliecePKCSDigestCipher.init(true, param);

                // generate random message
                int mLength = (rand.nextInt() & 0x1f) + 1;
                mBytes = new byte[mLength];
                rand.nextBytes(mBytes);

                // encrypt
                msgDigest.update(mBytes, 0, mBytes.length);
                byte[] hash = new byte[msgDigest.getDigestSize()];

                msgDigest.doFinal(hash, 0);

                byte[] enc = mcEliecePKCSDigestCipher.messageEncrypt(hash);

                // initialize for decryption
                mcEliecePKCSDigestCipher.init(false, pair.getPrivate());
                byte[] constructedmessage = mcEliecePKCSDigestCipher.messageDecrypt(enc);

                boolean verified = true;
                for (int i = 0; i < hash.length; i++)
                {
                    verified = verified && hash[i] == constructedmessage[i];
                }

                if (!verified)
                {
                    fail("en/decryption fails");
                }
                else
                {
                    System.out.println("test okay");
                    System.out.println();
                }

            }
        }

    }

    public void leadingZeroTest()
        throws Exception
    {
        /*
        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
        JPRvWBk1htKrDwB/tcjmdg02SxHNKyHwZqJ5EWxoFBpcCuthOyezRfBOKTAd7Py0pEYeq1POQqyWuhoIH7bLtdWK3WtUootS31fY2RGJ6h14Lyw80GmmawHFWbXwxirn3SkuWlF9DEs8fLa6SToRvR6JLJXYrM9I1U120z63xhRt0NyvYG/n/0H7IGpTD2gHjoK+AxpcyIV0pgRV69du6XX754P3JhH3WtUhDLo0pyeGI07di3aXxmLWoz8iAU1N3enwxDksZxT51hPkZERlcZ4l8p2B3rjE+JnQkaa0gWkE7WP+aoxcax6OXN00MgbX9yteJy/cRKdw+A==
        5Btx1aKTMktU9pwKG9U1KZp9OTag90msKyZhaqzWnXrDwJpPfbWZHt7VGOvbkKxQ8O6VdqDqyRMcAJPARZzjbYCTwug5jsUsEL2FQfaEpSw3bV8NYHcVPe9nz5a92cOjkSku9+GhTtAv1X/4UkIjgTdYeLEHoNmAmhfh3wrB9yWbOm7NujYa2T84w+9CxOBTfuPnKlay4O6vMRSpZGNG+BNMAkeDVGcQKqEofk80yDqW1GAbRY/kD7tqh0IXo6ARlGlumy/Zgmbb+1J4zo4obtB5s4JdrWo2d3JMK9vviQz9R2jOQme65prfp8pAM9OG3zNwqkFO8YdKjNMHUOZ+vEpwbzdzDhqOJprSWfAWL+PHk8xHevMKrCaqHLRp5Xk+rSkGAEaa9GuoNmQkO9Cx5qfvfpfzB+Md2YjmpeofOzL7QQRvE6yPRPANJys0pi7gGd2DHiw=

         */
        PrivateKeyInfo cca2PrivKey = PrivateKeyInfo.getInstance(
            Base64.decode(
                "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"));
        byte[] plainText = Base64.decode(
            "JPRvWBk1htKrDwB/tcjmdg02SxHNKyHwZqJ5EWxoFBpcCuthOyezRfBOKTAd7Py0pEYeq1POQqyWuhoIH7bLtdWK3WtUootS31fY2RGJ6h14Lyw80GmmawHFWbXwxirn3SkuWlF9DEs8fLa6SToRvR6JLJXYrM9I1U120z63xhRt0NyvYG/n/0H7IGpTD2gHjoK+AxpcyIV0pgRV69du6XX754P3JhH3WtUhDLo0pyeGI07di3aXxmLWoz8iAU1N3enwxDksZxT51hPkZERlcZ4l8p2B3rjE+JnQkaa0gWkE7WP+aoxcax6OXN00MgbX9yteJy/cRKdw+A==");
        byte[] cipherText = Base64.decode(
            "5Btx1aKTMktU9pwKG9U1KZp9OTag90msKyZhaqzWnXrDwJpPfbWZHt7VGOvbkKxQ8O6VdqDqyRMcAJPARZzjbYCTwug5jsUsEL2FQfaEpSw3bV8NYHcVPe9nz5a92cOjkSku9+GhTtAv1X/4UkIjgTdYeLEHoNmAmhfh3wrB9yWbOm7NujYa2T84w+9CxOBTfuPnKlay4O6vMRSpZGNG+BNMAkeDVGcQKqEofk80yDqW1GAbRY/kD7tqh0IXo6ARlGlumy/Zgmbb+1J4zo4obtB5s4JdrWo2d3JMK9vviQz9R2jOQme65prfp8pAM9OG3zNwqkFO8YdKjNMHUOZ+vEpwbzdzDhqOJprSWfAWL+PHk8xHevMKrCaqHLRp5Xk+rSkGAEaa9GuoNmQkO9Cx5qfvfpfzB+Md2YjmpeofOzL7QQRvE6yPRPANJys0pi7gGd2DHiw=");

        McElieceKobaraImaiCipher cipher = new McElieceKobaraImaiCipher();

        cipher.init(false, PrivateKeyFactory.createKey(cca2PrivKey));

        isTrue(Arrays.areEqual(plainText, cipher.messageDecrypt(cipherText)));
    }

    public static void main(
        String[] args)
    {
        runTest(new McElieceCipherTest());
    }

}
