// SPDX-FileCopyrightText: Copyright (c) 2008-2013, NVIDIA Corporation. All rights reserved.
// SPDX-License-Identifier: Apache-2.0

/*! \file unique_by_key.h
 *  \brief Sequential implementations of unique_by_key algorithms.
 */

#pragma once

#include <thrust/detail/config.h>

#if defined(_CCCL_IMPLICIT_SYSTEM_HEADER_GCC)
#  pragma GCC system_header
#elif defined(_CCCL_IMPLICIT_SYSTEM_HEADER_CLANG)
#  pragma clang system_header
#elif defined(_CCCL_IMPLICIT_SYSTEM_HEADER_MSVC)
#  pragma system_header
#endif // no system header
#include <thrust/iterator/iterator_traits.h>
#include <thrust/system/detail/sequential/execution_policy.h>

#include <cuda/std/__utility/pair.h>

THRUST_NAMESPACE_BEGIN
namespace system::detail::sequential
{
_CCCL_EXEC_CHECK_DISABLE
template <typename DerivedPolicy,
          typename InputIterator1,
          typename InputIterator2,
          typename OutputIterator1,
          typename OutputIterator2,
          typename BinaryPredicate>
_CCCL_HOST_DEVICE ::cuda::std::pair<OutputIterator1, OutputIterator2> unique_by_key_copy(
  sequential::execution_policy<DerivedPolicy>&,
  InputIterator1 keys_first,
  InputIterator1 keys_last,
  InputIterator2 values_first,
  OutputIterator1 keys_output,
  OutputIterator2 values_output,
  BinaryPredicate binary_pred)
{
  using InputKeyType    = thrust::detail::it_value_t<InputIterator1>;
  using OutputValueType = thrust::detail::it_value_t<OutputIterator2>;

  if (keys_first != keys_last)
  {
    InputKeyType temp_key      = *keys_first;
    OutputValueType temp_value = *values_first;

    for (++keys_first, ++values_first; keys_first != keys_last; ++keys_first, (void) ++values_first)
    {
      InputKeyType key      = *keys_first;
      OutputValueType value = *values_first;

      if (!binary_pred(temp_key, key))
      {
        *keys_output   = temp_key;
        *values_output = temp_value;

        ++keys_output;
        ++values_output;

        temp_key   = key;
        temp_value = value;
      }
    }

    *keys_output   = temp_key;
    *values_output = temp_value;

    ++keys_output;
    ++values_output;
  }

  return ::cuda::std::make_pair(keys_output, values_output);
} // end unique_by_key_copy()

template <typename DerivedPolicy, typename ForwardIterator1, typename ForwardIterator2, typename BinaryPredicate>
_CCCL_HOST_DEVICE ::cuda::std::pair<ForwardIterator1, ForwardIterator2> unique_by_key(
  sequential::execution_policy<DerivedPolicy>& exec,
  ForwardIterator1 keys_first,
  ForwardIterator1 keys_last,
  ForwardIterator2 values_first,
  BinaryPredicate binary_pred)
{
  // sequential unique_by_key_copy() permits in-situ operation
  return sequential::unique_by_key_copy(
    exec, keys_first, keys_last, values_first, keys_first, values_first, binary_pred);
} // end unique_by_key()
} // namespace system::detail::sequential
THRUST_NAMESPACE_END
