/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent.rewriter;

import java.io.File;
import java.io.InputStream;
import java.io.Reader;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.portals.applications.webcontent.rewriter.html.SwingParserAdaptor;
import org.apache.portals.applications.webcontent.rewriter.rules.Ruleset;
import org.apache.portals.applications.webcontent.rewriter.xml.SaxParserAdaptor;
import org.exolab.castor.mapping.Mapping;
import org.exolab.castor.xml.Unmarshaller;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**
 * RewriterServiceImpl
 * 
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor </a>
 * @version $Id: MappingRewriterController.java 833059 2009-11-05 15:38:39Z woonsan $
 *          Exp $
 */
public class MappingRewriterController implements RewriterController
{
    protected final static Logger log = LoggerFactory.getLogger(MappingRewriterController.class);
    final static String CONFIG_MAPPING_FILE = "mapping";
    final static String CONFIG_BASIC_REWRITER = "basic.class";
    final static String CONFIG_RULESET_REWRITER = "ruleset.class";
    final static String CONFIG_ADAPTOR_HTML = "adaptor.html";
    final static String CONFIG_ADAPTOR_XML = "adaptor.xml";

    // configuration parameters
    protected String mappingFile = null;

    /** the Castor mapping file name */
    protected Mapping mapper = null;

    /** Collection of rulesets in the system */
    protected Map rulesets = Collections.synchronizedMap(new HashMap());

    /** configured basic rewriter class */
    protected Class basicRewriterClass = BasicRewriter.class;
    
    /** configured ruleset rewriter class */
    protected Class rulesetRewriterClass = RulesetRewriterImpl.class;

    /** Adaptors */
    protected Map<String, Class> parserAdaptorMimeTypeClassMap;
    
    /** Basic rewriter instance properties */
    protected Map<String, String []> basicRewriterProps;
    
    /** Ruleset rewriter instance properties */
    protected Map<String, String []> rulesetRewriterProps;
    
    /** Parser adaptor rewriter instance properties map */
    protected Map<String, Map<String, String []>> parserAdaptorMimeTypePropsMap;
    
    public MappingRewriterController( String mappingFile ) throws RewriterException
    {
        this.mappingFile = mappingFile;
        
        this.parserAdaptorMimeTypeClassMap = new HashMap<String, Class>();
        this.parserAdaptorMimeTypeClassMap.put("text/html", SwingParserAdaptor.class);
        this.parserAdaptorMimeTypeClassMap.put("text/xml", SaxParserAdaptor.class);
        
        if (this.mappingFile != null)
        {
            loadMapping();
        }
    }

    public MappingRewriterController( String mappingFile, Class basicRewriterClass, Class ruleBasedRewriterClass, Map<String, Class> adaptorMimeTypeClassMap )
            throws RewriterException
    {
        this.mappingFile = mappingFile;
        
        this.parserAdaptorMimeTypeClassMap = new HashMap<String, Class>();
        this.parserAdaptorMimeTypeClassMap.put("text/html", SwingParserAdaptor.class);
        this.parserAdaptorMimeTypeClassMap.put("text/xml", SaxParserAdaptor.class);
        
        if (basicRewriterClass != null)
        {
            this.basicRewriterClass = basicRewriterClass;
        }
        
        if (ruleBasedRewriterClass != null)
        {
            this.rulesetRewriterClass = ruleBasedRewriterClass;
        }
        
        if (adaptorMimeTypeClassMap != null)
        {
            this.parserAdaptorMimeTypeClassMap.putAll(adaptorMimeTypeClassMap);
        }
        
        if (this.mappingFile != null)
        {
            loadMapping();
        }
    }
    
    public MappingRewriterController( String mappingFile, List rewriterClasses, List adaptorClasses )
            throws RewriterException
    {
        this.mappingFile = mappingFile;
        
        this.parserAdaptorMimeTypeClassMap = new HashMap<String, Class>();
        this.parserAdaptorMimeTypeClassMap.put("text/html", SwingParserAdaptor.class);
        this.parserAdaptorMimeTypeClassMap.put("text/xml", SaxParserAdaptor.class);
        
        if (rewriterClasses.size() > 0)
        {
            this.basicRewriterClass = (Class) rewriterClasses.get(0);
            
            if (rewriterClasses.size() > 1)
            {
                this.rulesetRewriterClass = (Class) rewriterClasses.get(1);
            }
        }
        
        if (adaptorClasses.size() > 0)
        {
            this.parserAdaptorMimeTypeClassMap.put("text/html", (Class) adaptorClasses.get(0));
            
            if (adaptorClasses.size() > 1)
            {
                this.parserAdaptorMimeTypeClassMap.put("text/xmll", (Class) adaptorClasses.get(1));
            }
        }

        if (this.mappingFile != null)
        {
            loadMapping();
        }
    }
    
    public MappingRewriterController( String mappingFile, String basicRewriterClassName, String rulesetRewriterClassName, 
                    String adaptorHtmlClassName, String adaptorXmlClassName )
    throws RewriterException
    {
        this(mappingFile, toClassList(basicRewriterClassName,rulesetRewriterClassName), toClassList(adaptorHtmlClassName,adaptorXmlClassName));
    }

    protected static List toClassList(String classNameA, String classNameB)
    {
        try
        {
            List list = new ArrayList(2);
            if ( classNameA != null )
            {
                list.add(Class.forName(classNameA));
            }
            if ( classNameB != null )
            {
                list.add(Class.forName(classNameB));
            }
            return list;
        } 
        catch (ClassNotFoundException e)
        {
            throw new RuntimeException(e);
        }
    }    

    public Rewriter createRewriter() throws InstantiationException, IllegalAccessException
    {
        Rewriter rewriter = (Rewriter) basicRewriterClass.newInstance();
        
        if (basicRewriterProps != null)
        {
            try
            {
                for (Map.Entry<String, String []> entry : basicRewriterProps.entrySet())
                {
                    BeanUtils.setProperty(rewriter, entry.getKey(), entry.getValue());
                }
            }
            catch (InvocationTargetException e)
            {
                throw new RuntimeException(e);
            }
        }
        
        return rewriter;
    }

    public RulesetRewriter createRewriter( Ruleset ruleset ) throws RewriterException
    {
        try
        {
            RulesetRewriter rewriter = (RulesetRewriter) rulesetRewriterClass.newInstance();
            rewriter.setRuleset(ruleset);
            
            if (rulesetRewriterProps != null)
            {
                for (Map.Entry<String, String []> entry : rulesetRewriterProps.entrySet())
                {
                    BeanUtils.setProperty(rewriter, entry.getKey(), entry.getValue());
                }
            }
            
            return rewriter;
        }
        catch (Exception e)
        {
            log.error("Error creating rewriter class", e);
        }
        
        return null;
    }

    public ParserAdaptor createParserAdaptor( String mimeType ) throws RewriterException
    {
        try
        {
            ParserAdaptor parserAdaptor = null;
            Class parserAdaptorClass = parserAdaptorMimeTypeClassMap.get(mimeType);
            
            if (parserAdaptorClass != null)
            {
                parserAdaptor = (ParserAdaptor) parserAdaptorClass.newInstance();
            
                if (parserAdaptorMimeTypePropsMap != null)
                {
                    Map<String, String []> parserAdaptorProps = parserAdaptorMimeTypePropsMap.get(mimeType);
                    
                    if (parserAdaptorProps != null)
                    {
                        for (Map.Entry<String, String []> entry : parserAdaptorProps.entrySet())
                        {
                            BeanUtils.setProperty(parserAdaptor, entry.getKey(), entry.getValue());
                        }
                    }
                }
            }
            
            return parserAdaptor;
        }
        catch (Exception e)
        {
            log.error("Error creating rewriter class", e);
        }
        
        return null;
    }

    /**
     * Load the mapping file for ruleset configuration
     *  
     */
    protected void loadMapping() throws RewriterException
    {
        if (this.mappingFile == null)
        {
            throw new RewriterException("The mapping file is not set.");
        }
        
        Reader reader = getReader(this.mappingFile);
        
        try
        {
            if (reader == null && this.mappingFile != null)
            {
                File file = new File(this.mappingFile);
                
                if (!file.isFile())
                {
                    throw new RewriterException("The mapping file is not available: " + this.mappingFile);
                }
                
                this.mapper = new Mapping();
                this.mapper.loadMapping(file.toURL());
            }
            else
            {
                InputSource is = new InputSource(reader);
                is.setSystemId(this.mappingFile);
                this.mapper.loadMapping(is);
            }
        }
        catch (Exception e)
        {
            String msg = "RewriterService: Error in castor mapping creation";
            log.error(msg, e);
            throw new RewriterException(msg, e);
        }
    }
    
    public Ruleset lookupRuleset( String id )
    {
        return (Ruleset) rulesets.get(id);
    }

    public Ruleset loadRuleset( Reader reader )
    {
        Ruleset ruleset = null;
        try
        {
            DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = dbfactory.newDocumentBuilder();

            InputSource source = new InputSource(reader);

            Document doc = builder.parse(source);

            Unmarshaller unmarshaller = new Unmarshaller(this.mapper);

            ruleset = (Ruleset) unmarshaller.unmarshal(doc);
            ruleset.sync();
            rulesets.put(ruleset.getId(), ruleset);

        }
        catch (Throwable t)
        {
            log.error("ForwardService: Could not unmarshal: " + reader, t);
        }

        return ruleset;
    }
    
    public Ruleset loadRuleset( InputStream input )
    {
        Ruleset ruleset = null;
        try
        {
            DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = dbfactory.newDocumentBuilder();

            InputSource source = new InputSource(input);

            Document doc = builder.parse(source);

            Unmarshaller unmarshaller = new Unmarshaller(this.mapper);

            ruleset = (Ruleset) unmarshaller.unmarshal(doc);
            ruleset.sync();
            rulesets.put(ruleset.getId(), ruleset);

        }
        catch (Throwable t)
        {
            log.error("ForwardService: Could not unmarshal: " + input, t);
        }

        return ruleset;
    }
    
    public Map<String, String []> getBasicRewriterProps()
    {
        return basicRewriterProps;
    }
    
    public void setBasicRewriterProps(Map<String, String []> props)
    {
        basicRewriterProps = props;
    }
    
    public Map<String, String []> getRulesetRewriterProps()
    {
        return rulesetRewriterProps;
    }
    
    public void setRulesetRewriterProps(Map<String, String []> props)
    {
        rulesetRewriterProps = props;
    }
    
    public Map<String, Map<String, String []>> getParserAdaptorMimeTypePropsMap()
    {
        return parserAdaptorMimeTypePropsMap;
    }
    
    public void setParserAdaptorMimeTypePropsMap(Map<String, Map<String, String []>> propsMap)
    {
        parserAdaptorMimeTypePropsMap = propsMap;
    }
    
    protected Reader getReader(String resourcePath) throws RewriterException
    {
        return null;
    }

}