/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.sql.engine.exec.exp.agg;

import java.math.BigDecimal;
import java.util.Comparator;
import java.util.List;
import java.util.function.IntFunction;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.calcite.DataContext;
import org.apache.calcite.avatica.util.ByteString;
import org.apache.calcite.rel.core.AggregateCall;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rel.type.RelDataTypeFactory;
import org.apache.calcite.rex.RexLiteral;
import org.apache.calcite.rex.RexNode;
import org.apache.calcite.sql.type.SqlTypeName;
import org.apache.calcite.util.ImmutableBitSet;
import org.apache.ignite.internal.sql.engine.exec.exp.IgniteSqlFunctions;
import org.apache.ignite.internal.sql.engine.exec.exp.agg.Accumulator;
import org.apache.ignite.internal.sql.engine.exec.exp.agg.AccumulatorsState;
import org.apache.ignite.internal.sql.engine.exec.exp.agg.MutableDouble;
import org.apache.ignite.internal.sql.engine.exec.exp.agg.MutableLong;
import org.apache.ignite.internal.sql.engine.type.IgniteTypeFactory;
import org.apache.ignite.internal.sql.engine.util.Commons;
import org.apache.ignite.internal.sql.engine.util.IgniteMath;
import org.apache.ignite.internal.sql.engine.util.Primitives;
import org.apache.ignite.internal.sql.engine.util.RexUtils;
import org.apache.ignite.internal.util.ArrayUtils;
import org.apache.ignite.lang.ErrorGroups;
import org.apache.ignite.sql.SqlException;
import org.jetbrains.annotations.Nullable;

public class Accumulators {
    private final IgniteTypeFactory typeFactory;

    public Accumulators(IgniteTypeFactory typeFactory) {
        this.typeFactory = typeFactory;
    }

    public Supplier<Accumulator> accumulatorFactory(DataContext context, AggregateCall call, RelDataType inputType) {
        return this.accumulatorFunctionFactory(context, call, inputType);
    }

    private Supplier<Accumulator> accumulatorFunctionFactory(DataContext context, AggregateCall call, RelDataType inputType) {
        switch (call.getAggregation().getName()) {
            case "COUNT": {
                return LongCount.FACTORY;
            }
            case "AVG": {
                return this.avgFactory(call);
            }
            case "SUM": {
                return this.sumFactory(call);
            }
            case "$SUM0": {
                return this.sumEmptyIsZeroFactory(call);
            }
            case "MIN": 
            case "EVERY": {
                return this.minMaxFactory(true, call);
            }
            case "MAX": 
            case "SOME": {
                return this.minMaxFactory(false, call);
            }
            case "SINGLE_VALUE": {
                return this.singleValueFactory(call);
            }
            case "ANY_VALUE": {
                return this.anyValueFactory(call);
            }
            case "LITERAL_AGG": {
                assert (call.rexList.size() == 1) : "Incorrect number of pre-operands for LiteralAgg: " + String.valueOf(call) + ", input: " + String.valueOf(inputType);
                RexNode lit = (RexNode)call.rexList.get(0);
                assert (lit instanceof RexLiteral) : "Non-literal argument for LiteralAgg: " + String.valueOf(call) + ", argument: " + String.valueOf(lit);
                return LiteralVal.newAccumulator(context, (RexLiteral)lit);
            }
            case "GROUPING": {
                return this.groupingFactory(call);
            }
        }
        throw new AssertionError((Object)call.getAggregation().getName());
    }

    private Supplier<Accumulator> avgFactory(AggregateCall call) {
        switch (call.type.getSqlTypeName()) {
            case TINYINT: 
            case SMALLINT: 
            case INTEGER: 
            case BIGINT: {
                return () -> DecimalAvg.FACTORY.apply(0);
            }
            case DECIMAL: {
                return () -> DecimalAvg.FACTORY.apply(call.type.getScale());
            }
        }
        if (call.type.getSqlTypeName() == SqlTypeName.ANY) {
            throw Accumulators.unsupportedAggregateFunction(call);
        }
        return DoubleAvg.FACTORY;
    }

    private Supplier<Accumulator> sumFactory(AggregateCall call) {
        switch (call.type.getSqlTypeName()) {
            case BIGINT: 
            case DECIMAL: {
                return () -> new Sum(new DecimalSumEmptyIsZero(call.type.getScale()));
            }
            case DOUBLE: 
            case REAL: 
            case FLOAT: {
                return () -> new Sum(new DoubleSumEmptyIsZero());
            }
        }
        if (call.type.getSqlTypeName() == SqlTypeName.ANY) {
            throw Accumulators.unsupportedAggregateFunction(call);
        }
        return () -> new Sum(new LongSumEmptyIsZero());
    }

    private Supplier<Accumulator> sumEmptyIsZeroFactory(AggregateCall call) {
        switch (call.type.getSqlTypeName()) {
            case BIGINT: {
                return LongSumEmptyIsZero.FACTORY;
            }
            case DECIMAL: {
                return () -> DecimalSumEmptyIsZero.FACTORY.apply(call.type.getScale());
            }
            case DOUBLE: 
            case REAL: 
            case FLOAT: {
                return DoubleSumEmptyIsZero.FACTORY;
            }
        }
        if (call.type.getSqlTypeName() == SqlTypeName.ANY) {
            throw Accumulators.unsupportedAggregateFunction(call);
        }
        return LongSumEmptyIsZero.FACTORY;
    }

    private Supplier<Accumulator> minMaxFactory(boolean min, AggregateCall call) {
        RelDataType type = call.getType();
        SqlTypeName typeName = type.getSqlTypeName();
        switch (typeName) {
            case CHAR: 
            case VARCHAR: {
                return min ? VarCharMinMax.MIN_FACTORY : VarCharMinMax.MAX_FACTORY;
            }
            case BINARY: 
            case VARBINARY: {
                return min ? VarBinaryMinMax.MIN_FACTORY : VarBinaryMinMax.MAX_FACTORY;
            }
        }
        if (type.getSqlTypeName() == SqlTypeName.ANY) {
            throw Accumulators.unsupportedAggregateFunction(call);
        }
        return MinMaxAccumulator.newAccumulator(min, (RelDataTypeFactory)this.typeFactory, type);
    }

    private Supplier<Accumulator> singleValueFactory(AggregateCall call) {
        RelDataType type = call.getType();
        if (type.getSqlTypeName() == SqlTypeName.ANY) {
            throw Accumulators.unsupportedAggregateFunction(call);
        }
        return SingleVal.newAccumulator(type);
    }

    private Supplier<Accumulator> anyValueFactory(AggregateCall call) {
        RelDataType type = call.getType();
        if (type.getSqlTypeName() == SqlTypeName.ANY) {
            throw Accumulators.unsupportedAggregateFunction(call);
        }
        return AnyVal.newAccumulator(type);
    }

    private Supplier<Accumulator> groupingFactory(AggregateCall call) {
        RelDataType type = call.getType();
        if (type.getSqlTypeName() == SqlTypeName.ANY) {
            throw Accumulators.unsupportedAggregateFunction(call);
        }
        return GroupingAccumulator.newAccumulator(call.getArgList());
    }

    private static AssertionError unsupportedAggregateFunction(AggregateCall call) {
        String functionName = call.getAggregation().getName();
        SqlTypeName typeName = call.getType().getSqlTypeName();
        return new AssertionError((Object)(functionName + " is not supported for " + String.valueOf(typeName)));
    }

    public static class LongCount
    implements Accumulator {
        public static final Supplier<Accumulator> FACTORY = LongCount::new;

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            assert (ArrayUtils.nullOrEmpty((Object[])args) || args.length == 1);
            if (ArrayUtils.nullOrEmpty((Object[])args) || args[0] != null) {
                MutableLong cnt = (MutableLong)state.get();
                if (cnt == null) {
                    cnt = new MutableLong();
                    state.set(cnt);
                }
                cnt.add(1L);
            }
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            if (state.get() == null) {
                result.set(0L);
            } else {
                MutableLong cnt = (MutableLong)state.get();
                result.set(cnt.longValue());
            }
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return List.of(typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.ANY), false));
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return typeFactory.createSqlType(SqlTypeName.BIGINT);
        }
    }

    public static class LiteralVal
    implements Accumulator {
        private final RelDataType type;
        @Nullable
        private final Object value;

        private LiteralVal(RelDataType type, @Nullable Object value) {
            this.type = type;
            this.value = value;
        }

        public static Supplier<Accumulator> newAccumulator(DataContext context, RexLiteral literal) {
            Class javaClass = (Class)Commons.typeFactory().getJavaClass(literal.getType());
            if (javaClass.isPrimitive()) {
                javaClass = Primitives.wrap(javaClass);
            }
            Object value = RexUtils.literalValue(context, literal, javaClass);
            return () -> new LiteralVal(literal.getType(), value);
        }

        @Override
        public void add(AccumulatorsState state, Object[] args) {
            assert (args.length == 1) : args.length;
            state.set(this.value);
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            result.set(this.value);
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return List.of(typeFactory.createTypeWithNullability(this.type, true));
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return this.type;
        }
    }

    public static class DoubleAvg
    implements Accumulator {
        public static final Supplier<Accumulator> FACTORY = DoubleAvg::new;

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            Double in = (Double)args[0];
            if (in == null) {
                return;
            }
            DoubleAvgState avgState = (DoubleAvgState)state.get();
            if (avgState == null) {
                avgState = new DoubleAvgState();
                state.set(avgState);
            }
            avgState.sum += in.doubleValue();
            ++avgState.cnt;
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            DoubleAvgState avgState = (DoubleAvgState)state.get();
            if (avgState == null) {
                result.set(null);
            } else if (avgState.cnt > 0L) {
                result.set(avgState.sum / (double)avgState.cnt);
            } else {
                result.set(null);
            }
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return List.of(typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.DOUBLE), true));
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.DOUBLE), true);
        }

        public static class DoubleAvgState {
            private double sum;
            private long cnt;
        }
    }

    public static class LongSumEmptyIsZero
    implements Accumulator {
        public static final Supplier<Accumulator> FACTORY = LongSumEmptyIsZero::new;

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            Long in = (Long)args[0];
            if (in == null) {
                return;
            }
            MutableLong sum = (MutableLong)state.get();
            if (sum == null) {
                sum = new MutableLong();
                state.set(sum);
            }
            sum.add(in);
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            if (!state.hasValue()) {
                result.set(0L);
            } else {
                MutableLong sum = (MutableLong)state.get();
                result.set(sum.longValue());
            }
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return List.of(typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.BIGINT), true));
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.BIGINT), true);
        }
    }

    public static class DoubleSumEmptyIsZero
    implements Accumulator {
        public static final Supplier<Accumulator> FACTORY = DoubleSumEmptyIsZero::new;

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            Double in = (Double)args[0];
            if (in == null) {
                return;
            }
            MutableDouble sum = (MutableDouble)state.get();
            if (sum == null) {
                sum = new MutableDouble();
                state.set(sum);
            }
            sum.add(in);
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            if (!state.hasValue()) {
                result.set(0.0);
            } else {
                MutableDouble sum = (MutableDouble)state.get();
                result.set(sum.doubleValue());
            }
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return List.of(typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.DOUBLE), true));
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.DOUBLE), true);
        }
    }

    public static class VarCharMinMax
    implements Accumulator {
        public static final Supplier<Accumulator> MIN_FACTORY = () -> new VarCharMinMax(true);
        public static final Supplier<Accumulator> MAX_FACTORY = () -> new VarCharMinMax(false);
        private final boolean min;

        VarCharMinMax(boolean min) {
            this.min = min;
        }

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            CharSequence in = (CharSequence)args[0];
            if (in == null) {
                return;
            }
            CharSequence val = (CharSequence)state.get();
            val = val == null ? in : (this.min ? (CharSeqComparator.INSTANCE.compare(val, in) < 0 ? val : in) : (CharSeqComparator.INSTANCE.compare(val, in) < 0 ? in : val));
            state.set(val);
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            result.set(state.get());
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return List.of(typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.VARCHAR), true));
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.VARCHAR), true);
        }

        private static class CharSeqComparator
        implements Comparator<CharSequence> {
            private static final CharSeqComparator INSTANCE = new CharSeqComparator();

            private CharSeqComparator() {
            }

            @Override
            public int compare(CharSequence s1, CharSequence s2) {
                int len = Math.min(s1.length(), s2.length());
                for (int i = 0; i < len; ++i) {
                    int cmp = Character.compare(s1.charAt(i), s2.charAt(i));
                    if (cmp == 0) continue;
                    return cmp;
                }
                return Integer.compare(s1.length(), s2.length());
            }
        }
    }

    public static class VarBinaryMinMax
    implements Accumulator {
        public static final Supplier<Accumulator> MIN_FACTORY = () -> new VarBinaryMinMax(true);
        public static final Supplier<Accumulator> MAX_FACTORY = () -> new VarBinaryMinMax(false);
        private final boolean min;

        VarBinaryMinMax(boolean min) {
            this.min = min;
        }

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            ByteString in = (ByteString)args[0];
            if (in == null) {
                return;
            }
            ByteString val = (ByteString)state.get();
            val = val == null ? in : (this.min ? (val.compareTo(in) < 0 ? val : in) : (val.compareTo(in) < 0 ? in : val));
            state.set(val);
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            result.set(state.get());
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return ArrayUtils.asList((Object[])new RelDataType[]{typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.VARBINARY), true)});
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.VARBINARY), true);
        }
    }

    public static final class MinMaxAccumulator
    implements Accumulator {
        private final boolean min;
        private final List<RelDataType> arguments;
        private final RelDataType returnType;

        private MinMaxAccumulator(boolean min, RelDataTypeFactory typeFactory, RelDataType relDataType) {
            RelDataType nullableType = typeFactory.createTypeWithNullability(relDataType, true);
            this.min = min;
            this.arguments = List.of(nullableType);
            this.returnType = nullableType;
        }

        public static Supplier<Accumulator> newAccumulator(boolean min, RelDataTypeFactory typeFactory, RelDataType type) {
            return () -> new MinMaxAccumulator(min, typeFactory, type);
        }

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            Comparable in = (Comparable)args[0];
            if (in == null) {
                return;
            }
            Comparable current = (Comparable)state.get();
            if (current == null) {
                state.set(in);
            } else {
                state.set(this.doApply(in, current));
            }
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            result.set(state.get());
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return this.arguments;
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return this.returnType;
        }

        private Comparable doApply(Comparable in, Comparable val) {
            if (val == null) {
                val = in;
            } else {
                int cmp = val.compareTo(in);
                val = this.min ? (cmp > 0 ? in : val) : (cmp < 0 ? in : val);
            }
            return val;
        }
    }

    public static class SingleVal
    implements Accumulator {
        private final RelDataType type;

        public static Supplier<Accumulator> newAccumulator(RelDataType type) {
            return () -> new SingleVal(type);
        }

        private SingleVal(RelDataType type) {
            this.type = type;
        }

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            assert (args.length == 1);
            if (state.hasValue()) {
                throw new SqlException(ErrorGroups.Sql.RUNTIME_ERR, "Subquery returned more than 1 value.");
            }
            state.set(args[0]);
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            result.set(state.get());
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return List.of(typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.ANY), true));
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return this.type;
        }
    }

    public static class AnyVal
    implements Accumulator {
        private final RelDataType type;

        private AnyVal(RelDataType type) {
            this.type = type;
        }

        public static Supplier<Accumulator> newAccumulator(RelDataType type) {
            return () -> new AnyVal(type);
        }

        @Override
        public void add(AccumulatorsState state, Object[] args) {
            assert (args.length == 1) : args.length;
            Object current = state.get();
            if (current == null) {
                state.set(args[0]);
            }
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            result.set(state.get());
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return List.of(typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.ANY), true));
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return this.type;
        }
    }

    public static class GroupingAccumulator
    implements Accumulator {
        private final List<Integer> argList;

        public static Supplier<Accumulator> newAccumulator(List<Integer> argList) {
            return () -> new GroupingAccumulator(argList);
        }

        private GroupingAccumulator(List<Integer> argList) {
            assert (!argList.isEmpty()) : "GROUPING function must have at least one argument.";
            assert (argList.size() < 64) : "GROUPING function with more than 63 arguments is not supported.";
            this.argList = argList;
        }

        @Override
        public void add(AccumulatorsState state, Object[] args) {
            assert (false);
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            ImmutableBitSet groupKey = (ImmutableBitSet)state.get();
            assert (groupKey != null);
            long res = 0L;
            long bit = 1L << this.argList.size() - 1;
            for (Integer col : this.argList) {
                res += groupKey.get(col.intValue()) ? bit : 0L;
                bit >>= 1;
            }
            result.set(res);
            state.set(null);
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return this.argList.stream().map(i -> typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.ANY), true)).collect(Collectors.toList());
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return typeFactory.createSqlType(SqlTypeName.BIGINT);
        }
    }

    public static class DecimalSumEmptyIsZero
    implements Accumulator {
        public static final IntFunction<Accumulator> FACTORY = DecimalSumEmptyIsZero::new;
        private final int precision = Short.MAX_VALUE;
        private final int scale;

        private DecimalSumEmptyIsZero(int scale) {
            this.scale = scale;
        }

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            BigDecimal in = (BigDecimal)args[0];
            if (in == null) {
                return;
            }
            BigDecimal sum = (BigDecimal)state.get();
            if (sum == null) {
                state.set(in);
            } else {
                state.set(sum.add(in));
            }
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            if (!state.hasValue()) {
                result.set(BigDecimal.ZERO);
            } else {
                BigDecimal value = (BigDecimal)state.get();
                result.set(IgniteSqlFunctions.toBigDecimal(value, this.precision, this.scale));
            }
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return List.of(typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.DECIMAL), true));
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.DECIMAL, this.precision, this.scale), false);
        }
    }

    public static class Sum
    implements Accumulator {
        private final Accumulator acc;

        public Sum(Accumulator acc) {
            this.acc = acc;
        }

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            if (args[0] == null) {
                return;
            }
            this.acc.add(state, args);
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            if (!state.hasValue()) {
                result.set(null);
            } else {
                this.acc.end(state, result);
            }
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return this.acc.argumentTypes(typeFactory);
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return this.acc.returnType(typeFactory);
        }
    }

    public static class DecimalAvg
    implements Accumulator {
        public static final IntFunction<Accumulator> FACTORY = DecimalAvg::new;
        private final int precision;
        private final int scale;

        DecimalAvg(int scale) {
            this.precision = -1;
            this.scale = scale;
        }

        @Override
        public void add(AccumulatorsState state, Object ... args) {
            BigDecimal in = (BigDecimal)args[0];
            if (in == null) {
                return;
            }
            DecimalAvgState sumState = (DecimalAvgState)state.get();
            if (sumState == null) {
                sumState = new DecimalAvgState();
                state.set(sumState);
            }
            sumState.sum = sumState.sum.add(in);
            sumState.cnt = sumState.cnt.add(BigDecimal.ONE);
        }

        @Override
        public void end(AccumulatorsState state, AccumulatorsState result) {
            DecimalAvgState sumState = (DecimalAvgState)state.get();
            if (sumState == null) {
                result.set(null);
            } else if (sumState.cnt.compareTo(BigDecimal.ZERO) == 0) {
                result.set(null);
            } else {
                result.set(IgniteMath.decimalDivide(sumState.sum, sumState.cnt, this.precision, this.scale));
            }
        }

        @Override
        public List<RelDataType> argumentTypes(IgniteTypeFactory typeFactory) {
            return List.of(typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.DECIMAL), true));
        }

        @Override
        public RelDataType returnType(IgniteTypeFactory typeFactory) {
            return typeFactory.createTypeWithNullability(typeFactory.createSqlType(SqlTypeName.DECIMAL, this.precision, this.scale), true);
        }

        public static class DecimalAvgState {
            private BigDecimal sum = BigDecimal.ZERO;
            private BigDecimal cnt = BigDecimal.ZERO;
        }
    }
}

