/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.network.scalecube;

import io.scalecube.cluster.Cluster;
import io.scalecube.cluster.Member;
import io.scalecube.cluster.membership.MembershipEvent;
import io.scalecube.cluster.metadata.MetadataCodec;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.Collections;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import org.apache.ignite.internal.logger.IgniteLogger;
import org.apache.ignite.internal.logger.Loggers;
import org.apache.ignite.internal.network.AbstractTopologyService;
import org.apache.ignite.internal.network.ClusterNodeImpl;
import org.apache.ignite.internal.network.TopologyEventHandler;
import org.apache.ignite.network.ClusterNode;
import org.apache.ignite.network.NetworkAddress;
import org.apache.ignite.network.NodeMetadata;
import org.jetbrains.annotations.Nullable;

final class ScaleCubeTopologyService
extends AbstractTopologyService {
    private static final IgniteLogger LOG = Loggers.forClass(ScaleCubeTopologyService.class);
    private static final MetadataCodec METADATA_CODEC = MetadataCodec.INSTANCE;
    private volatile Cluster cluster;
    private final ConcurrentMap<NetworkAddress, ClusterNode> members = new ConcurrentHashMap<NetworkAddress, ClusterNode>();
    private final ConcurrentMap<String, ClusterNode> consistentIdToMemberMap = new ConcurrentHashMap<String, ClusterNode>();
    private final ConcurrentMap<UUID, ClusterNode> idToMemberMap = new ConcurrentHashMap<UUID, ClusterNode>();

    ScaleCubeTopologyService() {
    }

    void setCluster(Cluster cluster) {
        this.cluster = cluster;
    }

    void onMembershipEvent(MembershipEvent event) {
        NodeMetadata metadata = ScaleCubeTopologyService.deserializeMetadata(event.newMetadata());
        ClusterNode member = ScaleCubeTopologyService.fromMember(event.member(), metadata);
        if (event.isAdded()) {
            this.members.put(member.address(), member);
            this.consistentIdToMemberMap.put(member.name(), member);
            this.idToMemberMap.put(member.id(), member);
            LOG.info("Node joined [node={}]", new Object[]{member});
            this.fireAppearedEvent(member);
        } else if (event.isUpdated()) {
            this.members.put(member.address(), member);
            this.consistentIdToMemberMap.put(member.name(), member);
            this.idToMemberMap.put(member.id(), member);
        } else if (event.isRemoved() || event.isLeaving()) {
            this.members.compute(member.address(), (addr, node) -> {
                if (node == null || node.id().equals(member.id())) {
                    LOG.info("Node left [member={}, eventType={}]", new Object[]{member, event.type()});
                    return null;
                }
                LOG.info("Node left (noop as it has already reappeared) [member={}, eventType={}]", new Object[]{member, event.type()});
                return node;
            });
            this.consistentIdToMemberMap.compute(member.name(), (consId, node) -> {
                if (node == null || node.id().equals(member.id())) {
                    return null;
                }
                return node;
            });
            this.idToMemberMap.remove(member.id());
            this.fireDisappearedEvent(member);
        }
        if (LOG.isInfoEnabled()) {
            LOG.info("Topology snapshot [nodes={}]", new Object[]{this.members.values().stream().map(ClusterNode::name).collect(Collectors.toList())});
        }
    }

    void updateLocalMetadata(@Nullable NodeMetadata metadata) {
        ClusterNode node = ScaleCubeTopologyService.fromMember(this.cluster.member(), metadata);
        this.members.put(node.address(), node);
        this.consistentIdToMemberMap.put(node.name(), node);
        this.idToMemberMap.put(node.id(), node);
    }

    private void fireAppearedEvent(ClusterNode member) {
        for (TopologyEventHandler handler : this.getEventHandlers()) {
            handler.onAppeared(member);
        }
    }

    private void fireDisappearedEvent(ClusterNode member) {
        for (TopologyEventHandler handler : this.getEventHandlers()) {
            handler.onDisappeared(member);
        }
    }

    public ClusterNode localMember() {
        Member localMember = this.cluster.member();
        NodeMetadata nodeMetadata = this.cluster.metadata().orElse(null);
        assert (localMember != null) : "Cluster has not been started";
        return ScaleCubeTopologyService.fromMember(localMember, nodeMetadata);
    }

    public Collection<ClusterNode> allMembers() {
        return Collections.unmodifiableCollection(this.members.values());
    }

    public ClusterNode getByAddress(NetworkAddress addr) {
        return (ClusterNode)this.members.get(addr);
    }

    public ClusterNode getByConsistentId(String consistentId) {
        return (ClusterNode)this.consistentIdToMemberMap.get(consistentId);
    }

    @Nullable
    public ClusterNode getById(UUID id) {
        return (ClusterNode)this.idToMemberMap.get(id);
    }

    private static ClusterNode fromMember(Member member, @Nullable NodeMetadata nodeMetadata) {
        NetworkAddress addr = new NetworkAddress(member.address().host(), member.address().port());
        return new ClusterNodeImpl(UUID.fromString(member.id()), member.alias(), addr, nodeMetadata);
    }

    @Nullable
    private static NodeMetadata deserializeMetadata(@Nullable ByteBuffer buffer) {
        if (buffer == null) {
            return null;
        }
        try {
            return (NodeMetadata)METADATA_CODEC.deserialize(buffer);
        }
        catch (Exception e) {
            LOG.warn("Couldn't deserialize metadata: {}", (Throwable)e);
            return null;
        }
    }
}

