/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.grails.forge.feature.migration;

import jakarta.inject.Singleton;
import org.grails.forge.application.generator.GeneratorContext;
import org.grails.forge.build.dependencies.Dependency;
import org.grails.forge.feature.migration.templates.dbMigrationGradle;
import org.grails.forge.template.RockerWritable;
import org.grails.forge.template.URLTemplate;
import org.grails.forge.util.VersionInfo;

@Singleton
public class DatabaseMigrationPlugin implements MigrationFeature {

    @Override
    public String getName() {
        return "database-migration";
    }

    @Override
    public String getTitle() {
        return "Database Migration";
    }

    @Override
    public String getDescription() {
        return "Add support for Liquibase database migrations. The Database Migration Plugin " +
                "helps you manage database schema changes while developing Grails applications.";
    }

    public String getThirdPartyDocumentation() {
        return "https://docs.liquibase.com/home.html";
    }

    @Override
    public String getDocumentation() {
        return "https://docs.grails.org/" + VersionInfo.getDocumentationVersion() + "/grails-data/hibernate5/manual/index.html#databaseMigration";
    }

    @Override
    public void apply(GeneratorContext generatorContext) {
        final String srcDirPath = getSrcDirPath();
        generatorContext.addBuildscriptDependency(Dependency.builder()
                .groupId("org.apache.grails")
                .artifactId("grails-data-hibernate5-dbmigration")
                .buildSrc()
                .extension(new RockerWritable(dbMigrationGradle.template(srcDirPath))));
        generatorContext.addDependency(Dependency.builder()
                .groupId("org.apache.grails")
                .artifactId("grails-data-hibernate5-dbmigration")
                .implementation());

        final ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        generatorContext.addTemplate(srcDirPath, new URLTemplate(srcDirPath + "/.gitkeep", classLoader.getResource(".gitkeep")));
    }

    private String getSrcDirPath() {
        return "grails-app/migrations";
    }
}
