/*
 * Decompiled with CFR 0.152.
 */
package net.sf.saxon.str;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import net.sf.saxon.str.Slice16;
import net.sf.saxon.str.Slice24;
import net.sf.saxon.str.Slice8;
import net.sf.saxon.str.StringTool;
import net.sf.saxon.str.Twine16;
import net.sf.saxon.str.Twine24;
import net.sf.saxon.str.Twine8;
import net.sf.saxon.str.UnicodeBuilder;
import net.sf.saxon.str.UnicodeString;

public final class LargeTextBuffer {
    private static final int BITS = 16;
    private static final int SEGLEN = 65536;
    private static final int MASK = 65535;
    private static final int MAX_SEGMENTS = 32768;
    private static final Segment EMPTY_SEGMENT = new Segment8(new byte[0]);
    private final List<Segment> completeSegments = new ArrayList<Segment>(4);
    private Segment lastSegment = EMPTY_SEGMENT;
    private int lastSegmentLength = 0;
    private int initialSize;

    public LargeTextBuffer(int initialSize) {
        this.initialSize = Math.max(initialSize, 65536);
    }

    private void addSegment(Segment segment) {
        if (this.completeSegments.size() == 32768) {
            throw new IllegalStateException("TinyTree capacity exceeded: more than 2^31 characters of text data");
        }
        this.completeSegments.add(segment);
    }

    private Segment getSegment(int n) {
        if (n == this.completeSegments.size()) {
            return this.lastSegment;
        }
        return this.completeSegments.get(n);
    }

    public void appendUnicodeString(UnicodeString chars) {
        if (chars.isEmpty()) {
            return;
        }
        if (this.lastSegment == EMPTY_SEGMENT) {
            int newWidth = chars.getWidth();
            int newLength = Math.max(this.initialSize, chars.length32()) & 0xFFFF;
            this.lastSegment = newWidth <= 8 ? new Segment8(new byte[newLength]) : (newWidth == 16 ? new Segment16(new char[newLength]) : new Segment24(new byte[newLength * 3]));
        }
        int spaceAvailableInLastSegment = 65536 - this.lastSegmentLength;
        long charsSupplied = chars.length();
        if (charsSupplied < (long)spaceAvailableInLastSegment) {
            this.extendLastSegment(chars);
        } else {
            long start = 0L;
            this.extendLastSegment(chars.substring(0L, spaceAvailableInLastSegment));
            charsSupplied -= (long)spaceAvailableInLastSegment;
            start += (long)spaceAvailableInLastSegment;
            while (charsSupplied > 65536L) {
                this.extendLastSegment(chars.substring(start, start + 65536L));
                charsSupplied -= 65536L;
                start += 65536L;
            }
            if (charsSupplied > 0L) {
                this.extendLastSegment(chars.substring(start, start + charsSupplied));
            }
        }
    }

    private void extendLastSegment(UnicodeString chars) {
        this.lastSegment = this.lastSegment.stretch(this.lastSegmentLength, this.lastSegmentLength + chars.length32(), chars.getWidth());
        if (this.lastSegment instanceof Segment8) {
            chars.copy8bit(((Segment8)this.lastSegment).bytes, this.lastSegmentLength);
        } else if (this.lastSegment instanceof Segment16) {
            chars.copy16bit(((Segment16)this.lastSegment).chars, this.lastSegmentLength);
        } else {
            assert (this.lastSegment instanceof Segment24);
            chars.copy24bit(((Segment24)this.lastSegment).bytes, this.lastSegmentLength * 3);
        }
        this.lastSegmentLength += chars.length32();
        if (this.lastSegmentLength == 65536) {
            this.addSegment(this.lastSegment);
            this.lastSegment = new Segment8(new byte[1024]);
            this.lastSegmentLength = 0;
        }
    }

    private void showSegmentLengths() {
        StringBuilder sb = new StringBuilder();
        for (Segment s : this.completeSegments) {
            sb.append(s.asUnicodeString().length()).append(", ");
        }
        sb.append(this.lastSegmentLength);
        System.err.println(sb);
    }

    public UnicodeString substring(int start, int end) {
        int firstSeg = start >> 16;
        int lastSeg = end - 1 >> 16;
        int lastCP = end & 0xFFFF;
        if (lastCP == 0) {
            lastCP = 65536;
        }
        if (firstSeg == lastSeg) {
            try {
                Segment seg = this.getSegment(firstSeg);
                return seg.substring(start & 0xFFFF, lastCP);
            }
            catch (ArrayIndexOutOfBoundsException e) {
                e.printStackTrace();
                throw e;
            }
        }
        UnicodeBuilder ub = new UnicodeBuilder();
        int segNr = firstSeg;
        ub.accept(this.getSegment(segNr++).substring(start & 0xFFFF, 65536));
        while (segNr < lastSeg) {
            ub.accept(this.getSegment(segNr++).asUnicodeString());
        }
        ub.accept(this.getSegment(lastSeg).substring(0, lastCP));
        return ub.toUnicodeString();
    }

    public void close() {
    }

    public int length() {
        if (this.lastSegment == EMPTY_SEGMENT) {
            return 0;
        }
        if (this.lastSegment == null) {
            return (this.completeSegments.size() - 1) * 65536 + this.lastSegmentLength;
        }
        return this.completeSegments.size() * 65536 + this.lastSegmentLength;
    }

    public void setLength(int newLength) {
        if (newLength < this.length()) {
            int segCount = this.completeSegments.size();
            if (newLength <= segCount * 65536) {
                this.lastSegment = this.completeSegments.get(segCount - 1);
                this.completeSegments.remove(segCount - 1);
            }
            this.lastSegmentLength = newLength & 0xFFFF;
        }
    }

    private static interface Segment {
        public int getWidth();

        public Segment stretch(int var1, int var2, int var3);

        public UnicodeString asUnicodeString();

        public UnicodeString substring(int var1, int var2);
    }

    private static class Segment8
    implements Segment {
        public byte[] bytes;

        public Segment8(byte[] bytes) {
            this.bytes = bytes;
        }

        @Override
        public int getWidth() {
            return 8;
        }

        @Override
        public Segment stretch(int oldLength, int newLength, int newWidth) {
            assert (newLength <= 65536);
            if (newWidth <= 8) {
                if (newLength > this.bytes.length) {
                    this.bytes = Arrays.copyOf(this.bytes, Math.max(newLength, Math.min(oldLength * 2, 65536)));
                }
                return this;
            }
            if (newWidth == 16) {
                char[] array16 = new char[newLength];
                StringTool.copy8to16(this.bytes, 0, array16, 0, oldLength);
                return new Segment16(array16);
            }
            byte[] array24 = new byte[newLength * 3];
            StringTool.copy8to24(this.bytes, 0, array24, 0, oldLength);
            return new Segment24(array24);
        }

        @Override
        public UnicodeString asUnicodeString() {
            return new Twine8(this.bytes);
        }

        @Override
        public UnicodeString substring(int start, int end) {
            return new Slice8(this.bytes, start, end);
        }
    }

    private static class Segment16
    implements Segment {
        public char[] chars;

        public Segment16(char[] chars) {
            this.chars = chars;
        }

        @Override
        public int getWidth() {
            return 16;
        }

        @Override
        public Segment stretch(int oldLength, int newLength, int newWidth) {
            assert (newLength <= 65536);
            if (newWidth <= 16) {
                if (newLength > this.chars.length) {
                    this.chars = Arrays.copyOf(this.chars, Math.max(newLength, Math.min(oldLength * 2, 65536)));
                }
                return this;
            }
            byte[] array24 = new byte[newLength * 3];
            StringTool.copy16to24(this.chars, 0, array24, 0, oldLength);
            return new Segment24(array24);
        }

        @Override
        public UnicodeString asUnicodeString() {
            return new Twine16(this.chars);
        }

        @Override
        public UnicodeString substring(int start, int end) {
            return new Slice16(this.chars, start, end);
        }
    }

    private static class Segment24
    implements Segment {
        public byte[] bytes;

        public Segment24(byte[] bytes) {
            this.bytes = bytes;
        }

        @Override
        public int getWidth() {
            return 24;
        }

        @Override
        public Segment stretch(int oldLength, int newLength, int newWidth) {
            assert (newLength <= 65536);
            if (newLength * 3 > this.bytes.length) {
                this.bytes = Arrays.copyOf(this.bytes, Math.max(newLength * 3, Math.min(oldLength * 6, 196608)));
            }
            return this;
        }

        @Override
        public UnicodeString substring(int start, int length) {
            return new Slice24(this.bytes, start, length);
        }

        @Override
        public UnicodeString asUnicodeString() {
            return new Twine24(this.bytes);
        }
    }
}

