package com.streamsicle;

import java.util.*;
//import javax.servlet.*;
// Import log4j classes.
import org.ten60.orchextra.*;


/**
 *  Title: Queue Description: A thread safe queue for streamsicle
 *  Copyright (c) 2001 Company: Streamsicle.com
 *
 *@author     Matt Hall
 *@created    February 16, 2003
 *@version    1.0
 */

public class RequestQueue {

    // The vector that will actually store the queue
    private Vector theQueue = null;
    // A hash of the files in the queue by file ID
    private HashMap fileIdHash = null;


    /**
     *  Constructor for the RequestQueue object
     */
    public RequestQueue() {
        theQueue = new Stack();
    }


    /**
     *  Gets the queue attribute of the RequestQueue object
     *
     *@return    The queue value
     */
    public Vector getQueue() {
        return (Vector) theQueue.clone();
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public int queueSize() {
        return theQueue.size();
    }


    /**
     *  Add a file to the queue. Check to see if the last queue item is a
     *  request, if not replace with this new item if it's a request, otherwise
     *  just add to the queue
     *
     *@param  fileToAdd  The feature to be added to the ToQueue attribute
     */
    public void addToQueue(QueueItem fileToAdd) {
        synchronized (theQueue) {
            Vector queue = theQueue;
            if (queue.size() > 0) {
                QueueItem requestCheck = (QueueItem) queue.lastElement();
                if (!requestCheck.isRequest() && fileToAdd.isRequest()) {
                    queue.removeElement(requestCheck);
                }
            }
            theQueue.addElement(fileToAdd);
            fileIdHash.put(new Integer(fileToAdd.getMP3File().getFileID()), fileToAdd);
        }
    }


    /**
     *  Description of the Method
     *
     *@param  fileToRemove  Description of the Parameter
     */
    public void removeFromQueue(MP3File fileToRemove) {
        removeFromQueue(fileToRemove.getFileID());
        return;
    }


    /**
     *  Description of the Method
     *
     *@param  fileID  Description of the Parameter
     */
    public void removeFromQueue(int fileID) {
        QueueItem file = (QueueItem) fileIdHash.get(new Integer(fileID));
        if (file == null) {
            // File not found in queue
            return;
        }

        synchronized (theQueue) {
            theQueue.removeElement(file);
            fileIdHash.remove(file);
        }
        return;
    }


    /**
     *  Take a file in the queue and move it to the top of the queue.
     *
     *@param  fileID  Description of the Parameter
     */
    public void moveToTop(int fileID) {
        QueueItem file = (QueueItem) fileIdHash.get(new Integer(fileID));
        if (file == null) {
            // File not found in queue
            return;
        }
        // at this point, file is a QueueItem, and is supposed to be placed at the top.
        synchronized (theQueue) {
            theQueue.removeElement(file);
            //take it out of its current position
            theQueue.insertElementAt(file, 0);
            // put it at the top
        }
    }

    /**
     *  Take a file in the queue and move it to the bottom of the queue.
     *
     *@param  fileID  Description of the Parameter
     */
    public void moveToBottom(int fileID) {
        QueueItem file = (QueueItem) fileIdHash.get(new Integer(fileID));
        if (file == null) {
            // File not found in queue
            return;
        }
        // at this point, file is a QueueItem, and is supposed to be placed at the top.
        synchronized (theQueue) {
            theQueue.removeElement(file);
            //take it out of its current position
            theQueue.insertElementAt(file, theQueue.size());
            // put it at the top
        }
    }

    /**
     * Move song in queue, positive numbers move up, negative moves down.
     * @param fileID
     * @param numSpaces
     */
    public void move(int fileID, int numSpaces) {
        QueueItem file = (QueueItem) fileIdHash.get(new Integer(fileID));
        if (file == null) {
            // File not found in queue
            return;
        }
        synchronized (theQueue) {
            int size = theQueue.size();
            int currPosition = theQueue.indexOf(file);
            int newPosition = Math.max(0,currPosition - numSpaces);
            // log.debug("Moving queue item at "+currPosition+" to "+newPosition);

            // Move the song
            theQueue.remove(file);
            theQueue.insertElementAt(file, newPosition);
        }
    }

    /**
     * Randomize the queue.
     */
    public void randomize() {
        Random rand = new Random();
        Vector newQueue = new Vector();
        synchronized (theQueue) {
            while (theQueue.size() > 0) {
                int getThis = rand.nextInt(theQueue.size());
                newQueue.add(theQueue.get(getThis));
                theQueue.remove(getThis);
            }
            theQueue = newQueue;
        }
    }


    /**
     *  Returns the next song in the queue, and removes that song from the queue
     *
     *@return    Description of the Return Value
     */
    public MP3File nextSong() {
        QueueItem nextSong = null;
        synchronized (theQueue) {
            nextSong = (QueueItem) theQueue.firstElement();
            theQueue.removeElement(nextSong);
            fileIdHash.remove(nextSong);
        }
        return nextSong.getMP3File();
    }

    /**
     * Reload the queue with the new file list and IDs
     * @param filenameHash
     */
    public void reloadQueue(Hashtable filenameHash) {
        synchronized (theQueue) {
            Enumeration queueElements = theQueue.elements();
            QueueItem file = null;
            Vector newQueue = new Vector();
            HashMap newHash = new HashMap();
            while (queueElements.hasMoreElements()) {
                file = (QueueItem) queueElements.nextElement();
                MP3File lookup = (MP3File) filenameHash.get(
                        file.getMP3File().getFile().getAbsolutePath().toLowerCase());
                if (lookup != null) {
                    QueueItem newQueueItem = new QueueItem(lookup);
                    newQueue.add(newQueueItem);
                    newHash.put(new Integer(lookup.getFileID()), newQueueItem);
                }
            }
            // Finally, replace the queue with whatever new files we could find
            theQueue = newQueue;
            fileIdHash = newHash;
        }
    }

    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public String toString() {
        StringBuffer sb = new StringBuffer();
        Enumeration queueElements = theQueue.elements();
        QueueItem file = null;
        while (queueElements.hasMoreElements()) {
            file = (QueueItem) queueElements.nextElement();
            sb.append(file.getMP3File().getFileID() + ", " + file.getMP3File() + " | ");
        }
        return sb.toString();
    }
}
