/******************************************************************************
 * (c) Copyright 2002,2003, 1060 Research Ltd
 *
 * This Software is licensed to You, the licensee, for use under the terms of
 * the 1060 Public License v1.0. Please read and agree to the 1060 Public
 * License v1.0 [www.1060research.com/license] before using or redistributing
 * this software.
 *
 * In summary the 1060 Public license has the following conditions.
 * A. You may use the Software free of charge provided you agree to the terms
 * laid out in the 1060 Public License v1.0
 * B. You are only permitted to use the Software with components or applications
 * that provide you with OSI Certified Open Source Code [www.opensource.org], or
 * for which licensing has been approved by 1060 Research Limited.
 * You may write your own software for execution by this Software provided any
 * distribution of your software with this Software complies with terms set out
 * in section 2 of the 1060 Public License v1.0
 * C. You may redistribute the Software provided you comply with the terms of
 * the 1060 Public License v1.0 and that no warranty is implied or given.
 * D. If you find you are unable to comply with this license you may seek to
 * obtain an alternative license from 1060 Research Limited by contacting
 * license@1060research.com or by visiting www.1060research.com
 *
 * NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC
 * LICENSE V1.0 FOR DETAILS
 *
 * THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ
 * THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license
 *
 * File:          $RCSfile: NKFHelperImpl.java,v $
 * Version:       $Name:  $ $Revision: 1.45 $
 * Last Modified: $Date: 2005/04/18 15:40:31 $
 *****************************************************************************/
package org.ten60.netkernel.layer1.nkf.impl;

import com.ten60.netkernel.urii.*;
import com.ten60.netkernel.urii.aspect.IAspectBoolean;
import com.ten60.netkernel.urrequest.*;
import com.ten60.netkernel.util.NetKernelException;
import com.ten60.netkernel.container.Container;
import com.ten60.netkernel.module.ModuleDefinition;
import com.ten60.netkernel.util.CheapMap;

import org.ten60.netkernel.layer1.representation.*;
import org.ten60.netkernel.layer1.meta.*;
import org.ten60.netkernel.layer1.nkf.*;

import java.net.URI;
import java.util.*;


/**
 *	The main helper implementation implements INKFCovenienceHelper
 * @author  tab
 */
public abstract class NKFHelperImpl implements INKFKernelHelper, INKFBasicHelper, INKFConvenienceHelper
{
	private Container mContainer;
	private IRequestorContext mContext;
	private NKFRequestReadOnlyImpl mRequest;
	private IURRepresentation mResponse;
	private Map mDependencies;
	private String mCWU;
	
	/** Creates a new instance of NKFHelperImpl */
	public NKFHelperImpl(Container aContainer, IRequestorContext aContext, URRequest aRequest)
	{	mContainer = aContainer;
		mContext = aContext;
		mResponse = null;
		if (aRequest!=null)
		{	mRequest = new NKFRequestReadOnlyImpl(aRequest);
			mCWU = mRequest.getCWU();
			mDependencies = new CheapMap(16);
		}
	}
	
	void destroy()
	{	mContainer=null;
		mContext=null;
		mRequest=null;
		mResponse=null;
		mDependencies=null;
	}
	
	IURRepresentation getResponse()
	{	IURRepresentation result = mResponse;
		if (result==null)
		{	result = VoidAspect.create();
		}
		return result;
	}
	
	void addDependencies(DependencyMeta aMeta)
	{	List added=new ArrayList(mDependencies.size());
		if (mDependencies!=null)
		{	for (Iterator i = mDependencies.values().iterator(); i.hasNext(); )
			{	IURRepresentation dependency = (IURRepresentation)i.next();
				boolean add=true;
				for (int j=added.size()-1; j>=0; j--)
				{	if (added.get(j)==dependency)
					{ add=false;
					  break;
					}
				}
				if (add)
				{	added.add(dependency);
					aMeta.addDependency(dependency);
				}
			}
		}
	}
	
	void declareDependency(URIdentifier aURI, IURRepresentation aRep)
	{	if (mDependencies!=null)
		{	mDependencies.put(aURI,aRep);
		}
	}
	
	//*************************
	// INKFKernelHelper API
	//*************************
	
	public Container getKernel()
	{	return mContainer;
	}
	
	public ModuleDefinition getOwningModule()
	{	return (ModuleDefinition)mContext;
	}
	
	public URRequest getThisKernelRequest() throws NKFException
	{	if (mRequest==null) throw new NKFException("No This Request");
		return mRequest.getKernelRequest();
	}

	
	public IURRepresentation issueRequest(URRequest aRequest) throws NetKernelException
	{	IURRepresentation result = getByValueArgument(aRequest);
		URRequest request = null;
		if (result!=null)
		{	if (!result.hasAspect(aRequest.getAspectClass()))
			{	request = getTransrepresentationRequest(result,aRequest);
			}
		}
		else
		{	request = aRequest;
		}
		if (request!=null)
		{	result=innerIssueSyncRequest(request);
			declareDependency(aRequest.getURI(),result);
		}
		return result;
	}
	
private IURRepresentation getByValueArgument(URRequest aRequest) throws NKFException
	{	IURRepresentation result=null;
		if (mRequest!=null)
		{	if (aRequest.getType()==URRequest.RQT_SOURCE && aRequest.argSize()==0)
			{	URIdentifier uri = aRequest.getURI();
				if (mDependencies!=null)
				{	result = (IURRepresentation)mDependencies.get(uri);
					if (result!=null && result.getMeta().isExpired())
					{	result=null;
					}
				}
				if (result==null)
				{	result = mRequest.getArgumentValue(uri.toString());
					if (result!=null)
					{	declareDependency(aRequest.getURI(),result);
					}
					else
					{	String fragment=uri.getFragment();
						if (fragment!=null)
						{	URIdentifier withoutFragment=uri.withoutFragment();
							IURRepresentation unfragmented = mRequest.getArgumentValue(withoutFragment.toString());
							if (unfragmented!=null)
							{	result = innerFragment(unfragmented, fragment, aRequest.getAspectClass());
							}
						}
					}
				}
			}
		}
		return result;
	}
	
	private URRequest getTransrepresentationRequest(IURRepresentation aValue, URRequest aRequest) throws NKFException
	{
		NKFRequestImpl req = (NKFRequestImpl)this.createSubRequest();
		req.setRequestType(INKFRequestReadOnly.RQT_TRANSREPRESENT);
		req.setURI(aRequest.getURI().toString());
		req.addSystemArgument(aValue);
		req.setAspectClass(aRequest.getAspectClass());
		return req.getKernelRequest();
	}
	
	public void setResponse(IURRepresentation aResponse)
	{	mResponse = aResponse;
	}
	
	//*************************
	// INKFBasicHelper API
	//*************************
	
	public INKFRequestReadOnly getThisRequest() throws NKFException
	{	if (mRequest==null) throw new NKFException("No This Request");
		return mRequest;
	}
	
	public INKFResponse createResponseFrom(IURRepresentation aRepresentation)
	{	declareDependency(URIdentifier.getUnique("var:"), aRepresentation);
		return new NKFResponseImpl(aRepresentation,this);
	}
	
	public INKFResponse createResponseFrom(IURAspect aAspect)
	{	return new NKFResponseImpl(aAspect,this);
	}
	
	public void setResponse(INKFResponse aResponse)
	{	mResponse = ((NKFResponseImpl)aResponse).getResponse();
	}

	public INKFRequest createSubRequest()
	{	NKFRequestImpl req= new NKFRequestImpl(mRequest,getOwningModule(),mCWU);
		return req;
	}
	
	public IURRepresentation issueSubRequest(INKFRequest aRequest) throws NKFException
	{	if (aRequest==null) throw new IllegalArgumentException("aRequest is null in issueSubRequest()");
		URRequest request = ((NKFRequestImpl)aRequest).getKernelRequest();
		ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
		try
		{	IURRepresentation result = this.issueRequest(request);
			return result;
		} catch (NetKernelException e)
		{	NKFException ex = new NKFException(e.getId(),e.getMessage(),request.toString());
			ex.addCause(e.getCause());
			throw ex;
		}
		finally
		{	Thread.currentThread().setContextClassLoader(contextClassLoader);
		}
	}

	public IURAspect issueSubRequestForAspect(INKFRequest aRequest) throws NKFException
	{	IURRepresentation rep = issueSubRequest(aRequest);
		IURAspect result = rep.getAspect( ((NKFRequestImpl)aRequest).getAspectClass() );
		return result;
	}
	
	public INKFAsyncRequestHandle issueAsyncSubRequest(INKFRequest aRequest) throws NKFException
	{
		if (aRequest==null) throw new IllegalArgumentException("aRequest is null in issueSubRequest()");
		URRequest kernelReq = ((NKFRequestImpl)aRequest).getKernelRequest();
		IURRepresentation result = getByValueArgument(kernelReq);
		URRequest request = null;
		if (result!=null)
		{	if (!result.hasAspect(kernelReq.getAspectClass()))
			{	request = getTransrepresentationRequest(result,kernelReq);
			}
		}
		else
		{	request = kernelReq;
		}
		NKFAsyncRequestHandleImpl subRequest;
		if (request!=null)
		{	subRequest = new NKFAsyncRequestHandleImpl((NKFRequestImpl)aRequest,this);
			request.setRequestor(subRequest);
			innerIssueAsyncRequest(request);
		}
		else
		{	// we have result already but must behave consistently
			subRequest = new NKFAsyncRequestHandleImpl((NKFRequestImpl)aRequest,this);
			subRequest.setResult(result);
		}
		return subRequest;	
	}
	
	
	public void setCWU(String aURI)
	{   mCWU=aURI;
	}
	
	public String getCWU()
	{   return mCWU;
	}
	
	//*************************
	// INKFConvenienceHelper API
	//*************************
	
	
	public void sinkAspect(String aURI, IURAspect aAspect) throws NKFException
	{	INKFRequest req=createSubRequest();
		req.setURI(aURI);
		req.setRequestType(INKFRequestReadOnly.RQT_SINK);
		IURRepresentation rep = createIntermediateRepresentationForAspect(aAspect);
		req.addSystemArgument(rep);
		issueSubRequest(req);
	}
	
	public IURRepresentation source(String aURI) throws NKFException
	{	INKFRequest req=createSubRequest();
		req.setURI(aURI);
		return issueSubRequest(req);
	}
	public IURRepresentation source(String aURI, Class aAspectClass) throws NKFException
	{	INKFRequest req=createSubRequest();
		req.setURI(aURI);
		req.setAspectClass(aAspectClass);
		return issueSubRequest(req);
	}
	
	public IURAspect sourceAspect(String aURI, Class aAspectClass) throws NKFException
	{	INKFRequest req=createSubRequest();
		req.setURI(aURI);
		req.setAspectClass(aAspectClass);
		IURRepresentation result=issueSubRequest(req);
		return result.getAspect(getNonNullAspectClass(aAspectClass));
	}

	public	IURAspect transrept(IURAspect aSource, Class aAspectClass) throws NKFException
	{	INKFRequest req=createSubRequest();
		req.setRequestType(INKFRequestReadOnly.RQT_TRANSREPRESENT);
		IURRepresentation rep= createIntermediateRepresentationForAspect(aSource);
		req.addSystemArgument(rep);
		req.setAspectClass(aAspectClass);
		req.setURI(URIdentifier.getUnique("literal:var").toString());
		IURRepresentation result=issueSubRequest(req);
		return result.getAspect(getNonNullAspectClass(aAspectClass));
	}
	
	public	IURAspect fragment(IURAspect aSource, String aFragment, String aMimeType, Class aAspectClass) throws NKFException
	{	IURRepresentation rep= createIntermediateRepresentationForAspect(aSource,aMimeType);
		IURRepresentation result=innerFragment(rep,aFragment,aAspectClass);
		return result.getAspect(getNonNullAspectClass(aAspectClass));
	}
	
	private IURRepresentation innerFragment(IURRepresentation aSource, String aFragment, Class aAspectClass) throws NKFException
	{	INKFRequest req=createSubRequest();
		req.setRequestType(INKFRequestReadOnly.RQT_FRAGMENT);
		req.addSystemArgument(aSource);
		req.setAspectClass(aAspectClass);
		req.setURI(URRequest.URI_SYSTEM.toString());
		req.setFragment(aFragment);
		return  issueSubRequest(req);
	}
	
	public String requestNew(String aURIBase, IURAspect aOptionalSource) throws NKFException
	{	INKFRequest req=createSubRequest();
		req.setRequestType(INKFRequestReadOnly.RQT_NEW);
		if (aOptionalSource!=null)
		{	IURRepresentation rep= createIntermediateRepresentationForAspect(aOptionalSource);
			req.addSystemArgument(rep);
		}
		req.setAspectClass(IAspectURI.class);
		req.setURI(aURIBase);
		IAspectURI result=(IAspectURI)issueSubRequest(req).getAspect(IAspectURI.class);
		return result.getURI().toString();
	}

	public boolean exists(String aURI) throws NKFException
	{	boolean result;
		if (aURI.startsWith("this:param:"))
		{	result=false;
			if (mRequest!=null)
			{	String arg = aURI.substring(11);
				String uri = mRequest.getArgument(arg);
				if (uri!=null)
				{	result = mRequest.getArgumentValue(uri)!=null;
					if (result==false)
					{	// passed by reference
						return exists(uri);
					}
				}
			}	
		}
		else
		{	INKFRequest req=createSubRequest();
			req.setURI(aURI);
			req.setRequestType(INKFRequestReadOnly.RQT_EXISTS);
			try
			{	IURRepresentation rep=issueSubRequest(req);
				result = ((IAspectBoolean)(rep.getAspect(IAspectBoolean.class))).isTrue();
			} catch (NKFException e)
			{	// failed so it doesn't
				result = false;
			}
		}
		return result;
	}
	
	public boolean delete(String aURI) throws NKFException
	{	INKFRequest req=createSubRequest();
		req.setURI(aURI);
		req.setRequestType(INKFRequestReadOnly.RQT_DELETE);
		IURRepresentation result=issueSubRequest(req);
		return ((IAspectBoolean)(result.getAspect(IAspectBoolean.class))).isTrue();
	}
	
	protected abstract void innerIssueAsyncRequest(URRequest aRequest);
	protected abstract IURRepresentation innerIssueSyncRequest(URRequest aRequest) throws NetKernelException;
	
	static IURRepresentation createIntermediateRepresentationForAspect(IURAspect aAspect)
	{	return innerCreateIntermediateRepresentationForAspect(aAspect,"content/unknown");
	}
	
	public  IURRepresentation createIntermediateRepresentationForAspect(IURAspect aAspect, String aMimeType)
	{	return innerCreateIntermediateRepresentationForAspect(aAspect,aMimeType);
	}
	static IURRepresentation innerCreateIntermediateRepresentationForAspect(IURAspect aAspect, String aMimeType)
	{	DependencyMeta meta = new DependencyMeta(aMimeType,0,0);
		IURRepresentation rep= new MonoRepresentationImpl(meta,aAspect);
		return rep;
	}
	
	public INKFKernelHelper getKernelHelper()
	{	return this;
	}
	
	private Class getNonNullAspectClass(Class aAspectClass)
	{	return (aAspectClass==null)?IURAspect.class : aAspectClass;
	}
	
}