/******************************************************************************
  (c) Copyright 2002,2005, 1060 Research Ltd                                   

  This Software is licensed to You, the licensee, for use under the terms of   
  the 1060 Public License v1.0. Please read and agree to the 1060 Public       
  License v1.0 [www.1060research.com/license] before using or redistributing   
  this software.                                                               

  In summary the 1060 Public license has the following conditions.             
  A. You may use the Software free of charge provided you agree to the terms   
  laid out in the 1060 Public License v1.0                                     
  B. You are only permitted to use the Software with components or applications
  that provide you with OSI Certified Open Source Code [www.opensource.org], or
  for which licensing has been approved by 1060 Research Limited.              
  You may write your own software for execution by this Software provided any  
  distribution of your software with this Software complies with terms set out 
  in section 2 of the 1060 Public License v1.0                                 
  C. You may redistribute the Software provided you comply with the terms of   
  the 1060 Public License v1.0 and that no warranty is implied or given.       
  D. If you find you are unable to comply with this license you may seek to    
  obtain an alternative license from 1060 Research Limited by contacting       
  license@1060research.com or by visiting www.1060research.com                 

  NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC  
  LICENSE V1.0 FOR DETAILS                                                     

  THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ     
  THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license      

  File:          $RCSfile: ThrottleAccessor.java,v $
  Version:       $Name:  $ $Revision: 1.3 $
  Last Modified: $Date: 2005/05/19 10:26:10 $
 *****************************************************************************/
package org.ten60.ura.util;

import com.ten60.netkernel.urii.*;
import com.ten60.netkernel.urii.aspect.NetKernelExceptionAspect;
import com.ten60.netkernel.scheduler.Scheduler;
import com.ten60.netkernel.urrequest.URResult;

import org.ten60.netkernel.layer1.nkf.*;
import org.ten60.netkernel.layer1.nkf.impl.NKFAccessorImpl;
import org.ten60.netkernel.xml.representation.*;
import org.ten60.netkernel.xml.xda.*;

import java.util.*;

/** Throttle provides a wrapping mechanism around a service for limiting the
 * concurrency. In the case of excessive concurrency demands requests are
 * queued in order until service is available. If the queue size grows beyond
 * a configured size the throttle will fail fast.
 * @author tab
 */
public class ThrottleAccessor extends NKFAccessorImpl implements INKFAsyncRequestListener
{
	/** map from throttle id to throttle state */
	private static final Map sIdToState = new HashMap();

	public static final String ARG_CONFIGURATION = "configuration";
	public static final String ARG_BASE_URI = "uri";
	public static final String ARG_THROTTLE_ID = "id";
	public static final String DEFAULT_CONFIG="ffcpl:/etc/ConfigThrottle.xml";	
	
	/** Per throttle state class
	 **/
	private static class ThrottleState
	{	/** number of requests currently in execution state*/
		public int mConcurrency = 0;
		/** list of requests pending execution */
		public final List mList = new ArrayList();
	}

    /** Creates a new instance of ThrottleAccessor */
    public ThrottleAccessor()
    {   super(4,true,0xFFFF); // accept all request types
    }
	
	public void processRequest(INKFConvenienceHelper context) throws Exception
	{	
		// parse configuration
		String configURI;
		if (context.getThisRequest().argumentExists(ARG_CONFIGURATION))
		{	configURI= "this:param:"+ARG_CONFIGURATION;
		}
		else
		{	configURI=DEFAULT_CONFIG;
		}
		IXDAReadOnly config = ((IAspectXDA)context.sourceAspect(configURI,IAspectXDA.class)).getXDA();
		int concurrency = Integer.parseInt(config.getText("concurrency",true));
		int queue = Integer.parseInt(config.getText("queue",true));

		// get throttle state
		String id = context.getThisRequest().getArgument(ARG_THROTTLE_ID);
		ThrottleState state = getThrottleFor(id);
		
		// decide what to do execute now, queue or fail
		boolean issueNow=false;
		synchronized(state)
		{	if (state.mConcurrency<concurrency)
			{	// execute now
				state.mConcurrency++;
				issueNow = true;
			}
			else if (state.mList.size()<queue)
			{	// queue request
				state.mList.add(context);
			}
			else
			{	// fail fast
				throw new NKFException("Throttle Overload","Queued requests on ["+id+"] exceeded maximum, request rejected", null);
			}
		}
		
		// issue request if possible
		if (issueNow)
		{	issueSubRequest(context);
		}
		
		// don't issue a response yet -  wait for response from sub-request
		context.setResponse(null);
	}
	
	/** Retrieve or create a throttle state for the given throttle id
	 **/
	private ThrottleState getThrottleFor(String aId)
	{	ThrottleState result;
		synchronized(sIdToState)
		{	result = (ThrottleState)sIdToState.get(aId);
			if (result==null)
			{	result = new ThrottleState();
				sIdToState.put(aId,result);
			}
		}
		return result;
	}
	
	/** asynchronously execute sub request based on this context */
	private void issueSubRequest(INKFConvenienceHelper context) throws NKFException
	{	INKFRequestReadOnly thisRequest = context.getThisRequest();
		INKFRequest subRequest = context.createSubRequest();
		subRequest.setURI(thisRequest.getArgument(ARG_BASE_URI));
		subRequest.setRequestType(thisRequest.getRequestType());
		subRequest.setAspectClass(thisRequest.getAspectClass());
		for (Iterator i=thisRequest.getArguments(); i.hasNext(); )
		{	String argName = (String)i.next();
			if (!argName.equals(ARG_BASE_URI) && !argName.equals(ARG_CONFIGURATION) && !argName.equals(ARG_THROTTLE_ID))
			{	String argURI = thisRequest.getArgument(argName);
				IURRepresentation argValue=thisRequest.getArgumentValue(argURI);
				if (argValue!=null)
				{	subRequest.addArgument(argName,argValue);
				}
				else
				{	subRequest.addArgument(argName,argURI);
				}
			}
		}
		context.issueAsyncSubRequest(subRequest).setListener(this);
	}
	
	/** called when sub-request completes with exception
	 */
	public void receiveException(NKFException aException, INKFRequest aRequest, INKFConvenienceHelper context) throws Exception
	{	// release potential queued requests
		String id = context.getThisRequest().getArgument(ARG_THROTTLE_ID);
		assessThrottle(id);
		
		// propage exception up
		throw aException;
	}
	
	/** called when sub-request completes successfully
	 */
	public void receiveRepresentation(IURRepresentation aRepresentation, INKFRequest aRequest, INKFConvenienceHelper context) throws Exception
	{	// release potential queued requests
		String id = context.getThisRequest().getArgument(ARG_THROTTLE_ID);
		assessThrottle(id);
		
		// propagate result
		INKFResponse response = context.createResponseFrom(aRepresentation);
		context.setResponse(response);
	}
	
	/** pull of a queued request if one is available and issue it
	 */
	private void assessThrottle(String aId)
	{	ThrottleState state = getThrottleFor(aId);
		boolean retry=true;
		while(retry)
		{	INKFConvenienceHelper context;
			synchronized(state)
			{	if (state.mList.size()>0)
				{	context=(INKFConvenienceHelper)state.mList.remove(0);
				}
				else
				{	context=null;
					state.mConcurrency--;
				}
			}
			try
			{	
				if (context!=null)
				{	issueSubRequest(context);
				}
				retry=false;
			}
			catch (NKFException e)
			{	// we cannot return errors in the execution of this request using NKF
				// we must report them back to the client of this request
				// This situation should never happen if the context is well-formed.
				// However NKF doesn't support this so we must go low-level
				Scheduler scheduler = (Scheduler)context.getKernelHelper().getKernel().getComponent(Scheduler.URI);
				IURRepresentation rep = NetKernelExceptionAspect.create(e);
				try
				{	URResult result = new URResult(context.getKernelHelper().getThisKernelRequest(),rep);
					scheduler.receiveAsyncException(result);
				}
				catch (NKFException e2)
				{ /* will not happen in this situation */ }
					
				
				// try next item
				retry=true;
			}
		}
	}
}