/* $Id: tupledesc.c,v 1.7 2005/08/17 01:24:06 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * The TupleDesc Python type implementation
 */
#include <setjmp.h>
#include <postgres.h>
#include <tcop/tcopprot.h>
#include <access/attnum.h>
#include <access/htup.h>
#include <access/hio.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <utils/array.h>
#include <utils/memutils.h>
#include <utils/palloc.h>
#include <utils/rel.h>
#include <utils/relcache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <abstract.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/relation.h>
#include <pypg/utils.h>
#include <pypg/conv.h>
#include <pypg/tif.h>

/*
 * TupleDesc_FetchAttrNumber - Fetch the AttrNumber from the attribute's name
 *
 * Returns 0 on failure to find an attribute, otherwise the appropriate
 * AttrNumber, which is the offset + 1.
 */
AttrNumber
AttrNumber_FromTupleDescAndNameStr(TupleDesc td, char *attname)
{
	register AttrNumber an;

	/* String is too long, can't be an attribute name */
	if (td->natts == 0 || strlen(attname) > NAMEDATALEN)
		return(0);

	for (an = td->natts; an > 0; --an)
	{
		register AttrNumber ao = AttrNumberGetAttrOffset(an);
		char *caname = NameStr(td->attrs[ao]->attname);
		if (strcmp(attname, caname) == 0)
			break;
	}

	return(an);
}

AttrNumber
AttrNumber_FromTupleDescAndPyObject(TupleDesc td, PyObj ob)
{
	AttrNumber an;

	if (PyInt_CheckExact(ob) || PyLong_CheckExact(ob))
	{
		an = (AttrNumber) PyInt_AsLong(ob);

		if (an < 0)
			an = an + td->natts + 1;
		else
			++an;
	}
	else
	{
		PyObj obstr;
		obstr = PyObject_Str(ob);
		if (obstr == NULL)
			return(0);
		
		an = AttrNumber_FromTupleDescAndNameStr(td, PyString_AS_STRING(obstr));
		DECREF(obstr);
	}

	return(an);
}

HeapTuple
PyPgTupleDesc_AttTuple(PyObj self, long attnum)
{
	HeapTuple ht;
	TupleDesc td;

	Assert(self != NULL && attnum >= 0);

	td = PyPgTupleDesc_FetchTupleDesc(self);
	ht = heap_addheader(Natts_pg_attribute, true,
			sizeof(FormData_pg_attribute), td->attrs[attnum]);

	return(ht);
}

AttrNumber
PyPgTupleDesc_AttOffset(PyObj self, PyObj ob)
{
	AttrNumber an;
	TupleDesc tupd;
	Assert(self != NULL && ob != NULL);

	tupd = PyPgTupleDesc_FetchTupleDesc(self);
	Assert(tupd != NULL);

	an = AttrNumber_FromTupleDescAndPyObject(tupd, ob);
	return(an);
}

PyObj
TupleDesc_Keys(TupleDesc td)
{
	PyObj rob = NULL;
	PyObj ob;
	int16 ca;
	rob = PyList_New(td->natts);

	for (ca = 0; ca < td->natts; ++ca)
	{
		ob = PyString_FromString(NameStr(td->attrs[ca]->attname));
		if (ob == NULL)
		{
			Py_DECREF(rob);
			return(NULL);
		}
		PyList_SET_ITEM(rob, ca, ob);
	}

	return(rob);
}

bool
TupleDesc_HasKey(TupleDesc td, const char *key)
{
	int16 ca;

	for (ca = 0; ca < td->natts; ++ca)
	{
		if (!strcmp(NameStr(td->attrs[ca]->attname), key))
			return(true);
	}
	return(false);
}

PyObj
tupd_keys(PyObj self)
{
	PyObj rob;
	rob = TupleDesc_Keys(PyPgTupleDesc_FetchTupleDesc(self));
	return(rob);
}

PyObj
tupd_haskey(PyObj self, PyObj name)
{
	TupleDesc td = PyPgTupleDesc_FetchTupleDesc(self);
	PyObj str, rob;

	str = PyObject_Str(name);
	rob = TupleDesc_HasKey(td, PyString_AS_STRING(str)) ? Py_True : Py_False;
	Py_DECREF(str);

	Py_INCREF(rob);
	return(rob);
}

static PyMethodDef PyPgTupleDesc_Methods[] = {
	{"keys", (PyCFunction) tupd_keys, METH_NOARGS,
	"get a list of the TupleDesc's keys"},
	{"has_key", (PyCFunction) tupd_haskey, METH_O,
	"check if TupleDesc has an attribute with a name the same as what's given"},
	{NULL}
};

static int
tupd_length(PyObj self)
{
	return(PyPgTupleDesc_FetchTupleDesc(self)->natts);
}

static PyObj
tupd_item(PyObj self, int attr)
{
	PyObj rob;
	HeapTuple ht;

	if (attr >= tupd_length(self))
	{
		PyErr_Format(PyExc_IndexError,
			"Postgres.TupleDesc index(%d) is out of range", attr);
		return(NULL);
	}

	ht = PyPgTupleDesc_AttTuple(self, attr);
	rob = PyPgObject_FromPyPgTypeAndHeapTuple(pg_attribute_PyPgType(), ht);

	return(rob);
}

static PyObj
tupd_slice(PyObj self, int from, int to)
{
	PyObj rob;
	int c, len;
	/* XXX: Need more error checking here */
	/* TODO: Return a new TupleDesc */

	len = self->ob_type->tp_as_sequence->sq_length(self);
	to = to > len ? len : to;

	rob = PyTuple_New(to - from);
	for (c = from; c < to; ++c)
	{
		PyTuple_SetItem(rob, c - from, tupd_item(self, c));
	}

	return(rob);
}

static int
tupd_contains(PyObj self, PyObj ob)
{
	if (PyPgTupleDesc_AttOffset(self, ob) >= 0)
		return(1);

	return(0);
}

static PySequenceMethods PyPgTupleDescAsSequence = {
	tupd_length,			/* sq_length */
	NULL,						/* sq_concat */
	NULL,						/* sq_repeat */
	tupd_item,				/* sq_item */
	tupd_slice,				/* sq_slice */
	NULL,						/* sq_ass_item */
	NULL,						/* sq_ass_slice */
	tupd_contains,			/* sq_contains */
	NULL,						/* sq_inplace_concat */
	NULL,						/* sq_inplace_repeat */
};

static PyObj
tupd_subscript(PyObj self, PyObj ob)
{
	AttrNumber num = PyPgTupleDesc_AttOffset(self, ob);
	if (num == 0)
	{
		PyErr_SetObject(PyExc_KeyError, ob);
		return(NULL);
	}

	return(self->ob_type->tp_as_sequence->sq_item(self, num-1));
}

static PyMappingMethods PyPgTupleDescAsMapping = {
	tupd_length,				/* mp_length */
	tupd_subscript,			/* mp_subscript */
	NULL,							/* mp_ass_subscript */
};

static void
tupd_dealloc(PyObj self)
{
	FreeTupleDesc(PyPgTupleDesc_FetchTupleDesc(self));
	self->ob_type->tp_free(self);
}

static int
tupd_compare(PyObj self, PyObj ob)
{
	TupleDesc pri, sec;

	if (!PyPgTupleDesc_Check(ob))
		return(-1);

	pri = PyPgTupleDesc_FetchTupleDesc(self);
	sec = PyPgTupleDesc_FetchTupleDesc(ob);

	if (pri->natts > sec->natts)
		return(1);
	else if (pri->natts < sec->natts)
		return(-1);
	else if (equalTupleDescs(pri, sec))
		return(0);
		
	return(-1);
}

static PyObj
tupd_str(PyObj self)
{
	PyObj rob;
	TupleDesc td;
	unsigned long i, natts;
	
	td = PyPgTupleDesc_FetchTupleDesc(self);
	natts = td->natts;

	for (i = 0; i < natts; ++i)
	{
		Form_pg_attribute att;
		Oid atttypid;
		char *str = NULL;

		if (i == 0)
			rob = PyString_FromString("");
		else
			PyString_ConcatAndDel(&rob, PyString_FromString(" "));

		att = td->attrs[i];
		atttypid = att->atttypid;

		str = NameStr(att->attname);
	
		if (strlen(str))
		{
			PyString_ConcatAndDel(&rob, PyString_FromString(str));
			PyString_ConcatAndDel(&rob, PyString_FromString(" "));
		}
		
		str = PgTypeName_FromOid(atttypid);
		PyString_ConcatAndDel(&rob, PyString_FromString(str));
		pfree(str);

		if (att->atttypmod > 0)
		{
			char *tmod = NULL;
			asprintf(&tmod, "(%d)", att->atttypmod);
			PyString_ConcatAndDel(&rob, PyString_FromString(tmod));
			free(tmod);
		}

		if (att->attndims > 0)
		{
			char *dims = NULL;
			asprintf(&dims, "[%d]", att->attndims);
			PyString_ConcatAndDel(&rob, PyString_FromString(dims));
			free(dims);
		}

		if (att->attnotnull)
			PyString_ConcatAndDel(&rob, PyString_FromString(" NOT NULL"));

		if (att->atthasdef)
		{
			AttrDefault *defval;
			uint16 num_defval;
			int ii;

			defval = td->constr->defval;
			num_defval = td->constr->num_defval;
			for (ii = 0; ii < num_defval; ++ii)
			{
				if (defval[ii].adnum == i)
					break;
			}

			if (ii < num_defval)
			{
				char *def;
				asprintf(&def, " DEFAULT '%s'", defval[ii].adbin);
				PyString_ConcatAndDel(&rob, PyString_FromString(def));
				free(def);
			}
		}

		if (i+1 < natts)
			PyString_ConcatAndDel(&rob, PyString_FromString(","));
	}

	return(rob);
}

static PyObj
tupd_repr(PyObj self)
{
	PyObj str, rob;
	str = self->ob_type->tp_str(self);
	if (str == NULL)
		return(NULL);
	rob = PyString_FromFormat("(%s)", PyString_AS_STRING(str));
	Py_DECREF(str);
	return(rob);
}

/*
 * tupd_call - create a Postgres.HeapTuple based on the descriptor
 */
static PyObj
tupd_call(PyObj self, PyObj args, PyObj kw)
{
	TupleDesc td;
	HeapTuple ht;
	PyObj rob;

	if (PyCFunctionErr_NoKeywordsAllowed(kw))
		return(NULL);

	td = PyPgTupleDesc_FetchTupleDesc(self);
	if (PyTuple_GET_SIZE(args) != td->natts)
	{
		PyErr_Format(PyExc_TypeError, "Tuple descriptor has %d attributes, "
			"given %d for HeapTuple construction",
			td->natts, PyTuple_GET_SIZE(args));
		return(NULL);
	}

	ht = HeapTuple_FromTupleDescAndPySequence(td, args);
	rob = PyPgHeapTuple_New(self, ht);
	return(rob);
}

static PyObj
tupd_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj rob;

	PyErr_SetString(PyExc_NotImplementedError,
			"custom TupleDesc creation not supported");
	return(NULL);

	rob = (PyObj) subtype->tp_alloc(subtype, 0);

	/* TODO: fill in custom tuple descriptor creation  */

	return(rob);
}

const char PyPgTupleDesc_Doc[] = "Postgres Tuple Descriptor";

PyTypeObject PyPgTupleDesc_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.TupleDesc",			/* tp_name */
	sizeof(struct PyPgTupleDesc),	/* tp_basicsize */
	0,										/* tp_itemsize */
	tupd_dealloc,						/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	tupd_compare,						/* tp_compare */
	tupd_repr,							/* tp_repr */
	NULL,									/* tp_as_number */
	&PyPgTupleDescAsSequence,		/* tp_as_sequence */
	&PyPgTupleDescAsMapping,		/* tp_as_mapping */
	NULL,									/* tp_hash */
	tupd_call,							/* tp_call */
	tupd_str,							/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	(char *) PyPgTupleDesc_Doc,	/* tp_doc */
	NULL,									/* tp_traverse */
	NULL,									/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	PySeqIter_New,						/* tp_iter */
	NULL,									/* tp_iternext */
	PyPgTupleDesc_Methods,			/* tp_methods */
	NULL,									/* tp_members */
	NULL,									/* tp_getset */
	NULL,									/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	tupd_new,							/* tp_new */
};

PyObj
PyPgTupleDesc_Initialize(PyObj self, TupleDesc tupd)
{
	if (self == NULL)
		return(NULL);

	PyPgTupleDesc_FixTupleDesc(self, tupd);
	return(self);
}

PyObj
PyPgTupleDesc_FromRelationId(Oid rid)
{
	Relation rel;
	TupleDesc td = NULL;
	PyObj rob = NULL;

	if (!RelationIsValid(rel))
		return(NULL);

	rel = RelationIdGetRelation(rid);
	td = CreateTupleDescCopy(RelationGetDescr(rel));
	RelationClose(rel);

	if (td == NULL)
		return(NULL);

	rob = PyPgTupleDesc_New(td);
	if (rob == NULL)
		FreeTupleDesc(td);

	return(rob);
}
/*
 * vim: ts=3:sw=3:noet:
 */
