package map;

import isj.ISJUtil;
import java.awt.Shape;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import emap.Emap;
import emap.Sej;
import emap.Smbc;

/**
 * 国土数値情報の行政界・海岸線（面）から作成された1つの市区町村を管理するクラスです。
 * @author Kumano Tatsuo
 * 2005/11/11
 */
public class City {
	/**
	 * 数値地図2500（空間データ基盤）を読み込んだかどうか
	 */
	private boolean has2500;

	/**
	 * 外接長方形
	 */
	private final Rectangle2D bounds;

	/**
	 * 高精度の領域
	 */
	private Shape fineShape;

	/**
	 * 市区町村コード
	 */
	private final String id;

	/**
	 * 市区町村名
	 */
	private final String label;

	/**
	 * 都道府県名
	 */
	private final String prefecture;

	/**
	 * 領域
	 */
	private final Shape shape;

	/**
	 * 数値地図2500（空間データ基盤）のURL
	 */
	private final URL url;

	/**
	 * 街区レベル位置参照情報
	 */
	private Map<String, Point2D> isj;

	/**
	 * 街区レベル位置参照情報のラベル位置
	 */
	private Map<Point2D, String> isjLabels;

	/**
	 * 三井住友銀行の一覧
	 * @since 3.10
	 */
	private Collection<PointData> smbc;

	/**
	 * セブンイレブンの一覧
	 * @since 3.11
	 */
	private Collection<PointData> sej;

	/**
	 * 市区町村を初期化します。
	 * @param shape 領域
	 * @param label 市区町村名
	 * @param id 市区町村コード
	 * @param url 数値地図2500（空間データ基盤）のURL
	 * @param prefecture 都道府県名
	 */
	public City(final Shape shape, final String label, final String id, final URL url,
			final String prefecture) {
		this.shape = shape;
		this.bounds = shape.getBounds2D();
		this.label = label;
		this.id = id;
		this.url = url;
		this.prefecture = prefecture;
		this.isjLabels = new HashMap<Point2D, String>();
	}

	/**
	 * 街区レベル位置参照情報のラベル位置を空にします。
	 */
	public void clearIsjLabels() {
		this.isjLabels.clear();
	}

	/**
	 * 高精度の領域を開放します。
	 */
	public void freeFineShape() {
		this.fineShape = null;
	}

	/**
	 * 街区レベル位置参照情報を開放します。
	 */
	public void freeIsj() {
		this.isj = null;
	}

	/**
	 * @return 外接長方形
	 */
	public Rectangle2D getBounds() {
		return this.bounds;
	}

	/**
	 * @return 高精度の領域
	 */
	public Shape getFineShape() {
		return this.fineShape;
	}

	/**
	 * @return 市区町村コード
	 */
	public String getId() {
		return this.id;
	}

	/**
	 * @return 街区レベル位置参照情報
	 */
	public Map<String, Point2D> getIsj() {
		return this.isj;
	}

	/**
	 * @return 街区レベル位置参照情報のラベル位置
	 */
	public Map<Point2D, String> getIsjLabels() {
		return this.isjLabels;
	}

	/**
	 * @return 市区町村名
	 */
	public String getLabel() {
		return this.label;
	}

	/**
	 * @return 領域
	 */
	public Shape getShape() {
		return this.shape;
	}

	/**
	 * @return 数値地図2500（空間データ基盤）のURL
	 */
	public URL getURL() {
		return this.url;
	}

	/**
	 * @return 数値地図2500（空間データ基盤）を読み込んだかどうか
	 */
	public boolean has2500() {
		return this.has2500;
	}

	/**
	 * @return 高精度の領域を持っているかどうか
	 */
	public boolean hasFineShape() {
		return this.fineShape != null;
	}

	/**
	 * @return 街区レベル位置参照情報を持っているかどうか
	 */
	public boolean hasIsj() {
		return this.isj != null;
	}

	/**
	 * 街区レベル位置参照情報をダウンロードし、読み込みます。
	 * @throws IOException 
	 */
	public void loadIsj() throws IOException {
		this.isj = ISJUtil.loadIsj(this.id);
	}

	/**
	 * @param shape 高精度の領域
	 */
	public void setFineShape(final Shape shape) {
		this.fineShape = shape;
	}

	/**
	 * @param has2500 数値地図2500（空間データ基盤）を読み込んだかどうか 
	 */
	public void setHas2500(final boolean has2500) {
		this.has2500 = has2500;
	}

	/**
	 * @since 3.10
	 * @return 三井住友銀行の一覧
	 */
	public Collection<PointData> getSmbc() {
		return this.smbc;
	}

	/**
	 * @since 3.11
	 * @return セブンイレブンの一覧
	 */
	public Collection<PointData> getSej() {
		return this.sej;
	}

	/**
	 * @return 三井住友銀行の一覧を持っているかどうか
	 */
	public boolean hasSmbc() {
		return this.smbc != null;
	}

	/**
	 * @return セブンイレブンの一覧を持っているかどうか
	 */
	public boolean hasSej() {
		return this.sej != null;
	}

	/**
	 * 三井住友銀行の一覧を読み込みます。
	 * @throws IOException 
	 * @throws MalformedURLException 
	 * @since 3.10
	 */
	public void loadSmbc() throws MalformedURLException, IOException {
		this.smbc = new ArrayList<PointData>();
		loadEmap(new Smbc(), this.smbc, Const.Smbc.CACHE_DIR, Const.Smbc.PREFIX, Const.Smbc.SUFFIX,
				"三井住友");
	}

	/**
	 * セブンイレブンの一覧を読み込みます。
	 * @throws IOException 
	 * @throws MalformedURLException 
	 * @since 3.10
	 */
	public void loadSej() throws MalformedURLException, IOException {
		this.sej = new ArrayList<PointData>();
		loadEmap(new Sej(), this.sej, Const.Sej.CACHE_DIR, Const.Sej.PREFIX, Const.Sej.SUFFIX,
				"セブイレ");
	}

	/**
	 * 三井住友銀行、セブンイレブンの一覧を読み込みます。
	 * @param emap 解析エンジン
	 * @param points 結果を格納するオブジェクト
	 * @param cacheDir キャッシュディレクトリ
	 * @param prefix 接頭語
	 * @param suffix 接尾語
	 * @param attribute 表示する属性
	 * @throws MalformedURLException
	 * @throws IOException
	 * @throws FileNotFoundException
	 * @throws UnsupportedEncodingException
	 */
	private void loadEmap(final Emap emap, final Collection<PointData> points,
			final String cacheDir, final String prefix, final String suffix, final String attribute)
			throws MalformedURLException, IOException, FileNotFoundException,
			UnsupportedEncodingException {
		File csvFile = new File(cacheDir + File.separator + prefix + String.valueOf(this.id)
				+ suffix);
		if (!csvFile.exists()) {
			if (!new File(cacheDir).exists()) {
				new File(cacheDir).mkdir();
			}
			final Map<String, URL> prefectures = emap.getPrefectures();
			for (final Map.Entry<String, URL> entry : prefectures.entrySet()) {
				if (this.prefecture.contains(entry.getKey())) {
					System.out.println("DEBUG: getting " + entry.getValue());
					final Map<String, URL> cities = emap.getCities(entry.getValue());
					for (final Map.Entry<String, URL> entry2 : cities.entrySet()) {
						final String[] strings = entry2.getKey().split(",");
						if (strings.length == 2) {
							final String city = strings[1];
							if (city.equals(this.label)) {
								final PrintWriter out = new PrintWriter(new File(cacheDir
										+ File.separator + prefix + String.valueOf(this.id)
										+ suffix), "SJIS");
								final Map<String, String> addresses = emap.getAddresses(entry2
										.getValue());
								final Map<String, Point2D> tempIsj = new HashMap<String, Point2D>();
								for (final Map.Entry<String, Point2D> entry4 : this.isj.entrySet()) {
									String[] strings2 = entry4.getKey().split(",");
									if (strings2.length == 4) {
										final String prefecture1 = strings2[0];
										final String city2 = strings2[1];
										String tyome = strings2[2];
										tyome = tyome.replaceFirst("二十丁目$", "20");
										tyome = tyome.replaceFirst("十九丁目$", "19");
										tyome = tyome.replaceFirst("十八丁目$", "18");
										tyome = tyome.replaceFirst("十七丁目$", "17");
										tyome = tyome.replaceFirst("十六丁目$", "16");
										tyome = tyome.replaceFirst("十五丁目$", "15");
										tyome = tyome.replaceFirst("十四丁目$", "14");
										tyome = tyome.replaceFirst("十三丁目$", "13");
										tyome = tyome.replaceFirst("十二丁目$", "12");
										tyome = tyome.replaceFirst("十一丁目$", "11");
										tyome = tyome.replaceFirst("十丁目$", "10");
										tyome = tyome.replaceFirst("九丁目$", "9");
										tyome = tyome.replaceFirst("八丁目$", "8");
										tyome = tyome.replaceFirst("七丁目$", "7");
										tyome = tyome.replaceFirst("六丁目$", "6");
										tyome = tyome.replaceFirst("五丁目$", "5");
										tyome = tyome.replaceFirst("四丁目$", "4");
										tyome = tyome.replaceFirst("三丁目$", "3");
										tyome = tyome.replaceFirst("二丁目$", "2");
										tyome = tyome.replaceFirst("一丁目$", "1");
										final String number = strings2[3];
										final String string = (tyome.matches(".*[0-9]$")) ? prefecture1
												+ city2 + tyome + "-" + number
												: prefecture1 + city2 + tyome + number;
										tempIsj.put(string, entry4.getValue());
									} else {
										System.out.println("WARNING: 街区レベル位置参照情報のデータ形式が不正です。"
												+ entry4.getKey());
									}
								}
								for (final Map.Entry<String, String> entry3 : addresses.entrySet()) {
									String address = entry3.getKey();
									address = address.replaceAll("−", "-");
									address = address.replaceAll("ー", "-");
									address = address.replaceAll("０", "0");
									address = address.replaceAll("１", "1");
									address = address.replaceAll("２", "2");
									address = address.replaceAll("３", "3");
									address = address.replaceAll("４", "4");
									address = address.replaceAll("５", "5");
									address = address.replaceAll("６", "6");
									address = address.replaceAll("７", "7");
									address = address.replaceAll("８", "8");
									address = address.replaceAll("９", "9");
									final String caption = entry3.getValue().replaceAll(",", "");
									final Pattern pattern = Pattern.compile("([^-]+-[0-9]+)-");
									final Matcher matcher = pattern.matcher(address);
									if (matcher.find()) {
										address = matcher.group(1);
									}
									if (tempIsj.containsKey(address)) {
									} else {
										address = address.replaceFirst("-.+", "");
										if (tempIsj.containsKey(address)) {
										} else {
											address = replaceTyomeToHyphen(address,
													"([^0-9]+[0-9]+)丁目([0-9]+)");
											if (tempIsj.containsKey(address)) {
											} else {
												final String address2 = address.replace("ケ丘", "ヶ丘");
												if (tempIsj.containsKey(address2)) {
													address = address2;
												} else {
													if (address.matches(".+[0-9]+番[0-9]+号$")) {
														address = address
																.replaceAll("番[0-9]+号", "");
													}
													if (tempIsj.containsKey(address)) {
													} else {
														System.out.println("DEBUG: no match "
																+ address);
														continue;
													}
												}
											}
										}
									}
									final Point2D point = tempIsj.get(address);
									address = address.replaceAll(",", "");
									out.println(address + "," + caption + "," + point.getX() + ","
											+ point.getY());
									System.out.println("DEBUG: found " + address + ", " + caption);
								}
								out.close();
							}
						} else {
							System.out.println("WARNING: データ形式が不正です。" + entry2);
						}
					}
					break;
				}
			}
			csvFile = new File(cacheDir + File.separator + prefix + String.valueOf(this.id)
					+ suffix);
		}
		if (csvFile.exists()) {
			final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(csvFile),
					"SJIS"));
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final String[] items = line.split(",");
				if (items.length == 4) {
					final double x = Double.parseDouble(items[2]);
					final double y = Double.parseDouble(items[3]);
					final PointData pointData = new PointData(attribute,
							PointData.CLASSIFICATION_UNKNOWN, x, y);
					pointData.setAttribute(attribute);
					points.add(pointData);
				} else {
					System.out.println("WARNING: データ形式が不正です。" + line);
				}
			}
			scanner.close();
		}
	}

	/**
	 * 「(aaa)bbb(ccc)ddd」を「aaa-ccc」に置換します。
	 * @param string 置換前の文字列
	 * @param regex 正規表現。丸括弧が2つあるべきです。
	 * @return 置換後の文字列
	 */
	private String replaceTyomeToHyphen(final String string, final String regex) {
		final Pattern pattern = Pattern.compile(regex);
		final Matcher matcher = pattern.matcher(string);
		if (matcher.find()) {
			return matcher.group(1) + "-" + matcher.group(2);
		} else {
			return string;
		}
	}

	/**
	 * @return 都道府県名
	 */
	public String getPrefecture() {
		return this.prefecture;
	}
}
