﻿////' This class permits you to perform direct connections to FTP sites in Visual Basic. NET.
////' The class supports the following FTP commands:
////'   - Upload a file
////'   - Download a file
////'   - Create a directory
////'   - Remove a directory
////'   - Change directory
////'   - Remove a file
////'   - Rename a file
////'   - Set the user name of the remote user
////'   - Set the password of the remote user

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net;
using System.IO;
using System.Net.Sockets;

namespace ConsoleApplication4
{


    //FTP Class
    class clsFTP
    {
#region Class Variable Declarations
        public string RemoteHost { get; set; }
        public Int32 RemotePort { get; set; }
        public string RemotePath { get; set; }
        public string RemoteUser { get; set; }
        public string RemotePassword { get; set; }

        private Int32 m_iBytes;
        private Socket m_objClientSocket;

        private Int32 m_iRetValue;
        private bool m_bLoggedIn;
        private string m_sMes;
        private string m_sReply;

        ////'Set the size of the packet that is used to read and to write data to the FTP server 
        ////'to the following specified size.
        private const int BLOCK_SIZE = 512;
        private byte[] m_aBuffer = new byte[BLOCK_SIZE];
        private Encoding ASCII = Encoding.ASCII;
        public bool flag_bool;

        // General variable declaration
        public string MessageString { get; set; }

        FileStream output;
#endregion

#region Class Constructors

        ////' Main class constructor
        public clsFTP()
        {
            this.RemoteHost = "microsoft";
            this.RemotePath = ".";
            this.RemoteUser = "anonymous";
            this.RemotePassword = "";
            this.MessageString = "";
            this.RemotePort = 21;
            m_bLoggedIn = false;
        }

        ////' Parameterized constructor
        public clsFTP(string sRemoteHost, string sRemotePath, string sRemoteUser, string sRemotePassword, Int32 iRemotePort = 21)
        {
            this.RemoteHost = sRemoteHost;
            this.RemotePath = sRemotePath;
            this.RemoteUser = sRemoteUser;
            this.RemotePassword = sRemotePassword;
            this.MessageString = "";
            this.RemotePort = iRemotePort;
            this.m_bLoggedIn = false;
        }

#endregion

#region public Subs and Functions

        //'return a list of files from the file system. return these files in a string() array.
        public String[] GetFileList(string sMask)
        {

            Socket cSocket;
            Int32 bytes;
            Char seperator = '\r';
            string[] mess;

            this.m_sMes = "";

            //'Check to see if you are logged on to the FTP server.
            if (!m_bLoggedIn)
            {
                Login();
            }

            cSocket = CreateDataSocket();

            //'Send an FTP command. 
            SendCommand("NLST " + sMask);

            if (!(m_iRetValue == 150 || m_iRetValue == 125))
            {
                output.Close();
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }

            m_sMes = "";
            while (true)
            {
                Array.Clear(m_aBuffer, 0, m_aBuffer.Length);
                bytes = cSocket.Receive(m_aBuffer, m_aBuffer.Length, 0);
                m_sMes += ASCII.GetString(m_aBuffer, 0, bytes);

                if (bytes < m_aBuffer.Length)
                {
                    break;
                }
            }

            mess = m_sMes.Split(seperator);
            cSocket.Close();
            ReadReply();

            if (m_iRetValue != 226)
            {
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }

            return mess;
        }

        //' Get the size of the file on the FTP server.
        public long GetFileSize(string sFileName)
        {
            long size;

            if (!m_bLoggedIn)
            {
                Login();
            }
            //'Send an FTP command. 
            SendCommand("SIZE " + sFileName);
            size = 0;

            if (m_iRetValue == 213) {
                size = Int64.Parse(m_sReply.Substring(4));
            }
            else
            {
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }

            return size;
        }

        //'Log on to the FTP server.
        public bool Login()
        {

            m_objClientSocket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);

//            var ep = new IPEndPoint(Dns.GetHostEntry(this.RemoteHost).AddressList[0], this.RemotePort);

            try
            {
                m_objClientSocket.Connect(this.RemoteHost,this.RemotePort);
            }
            catch (Exception ex)
            {
                MessageString = m_sReply;
                throw new IOException("Cannot connect to the remote server. (" + ex.Message + ")" );
            }

            ReadReply();
            if (m_iRetValue != 220)
            {
                CloseConnection();
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }

            //'Send an FTP command to send a user logon ID to the server.
            SendCommand("USER " + this.RemoteUser);
            if (!(m_iRetValue == 331 || m_iRetValue == 230))
            {
                Cleanup();
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }

            if (m_iRetValue != 230)
            {
                //'Send an FTP command to send a user logon password to the server.
                SendCommand("PASS " + this.RemotePassword);
                if (!(m_iRetValue == 230 || m_iRetValue == 202))
                {
                    Cleanup();
                    MessageString = m_sReply;
                    throw new IOException(m_sReply.Substring(4));
                }
            }

            m_bLoggedIn = true;
            //'Call the ChangeDirectory user-defined function to change the directory to the 
            //'remote FTP folder that is mapped.
            ChangeDirectory( this.RemotePath );

            //'return the final result.
            return m_bLoggedIn;
        }

        //'if the value of mode is true, set binary mode for downloads. Otherwise, set ASCII mode.
        public void SetBinaryMode(bool bMode)
        {
            if (bMode)
            {
                //'Send the FTP command to set the binary mode.
                //'(TYPE is an FTP command that is used to specify representation type.)
                SendCommand("TYPE I");
            }
            else
            {
                //'Send the FTP command to set ASCII mode. 
                //'(TYPE is an FTP command that is used to specify representation type.)
                SendCommand("TYPE A");
            }

            if (m_iRetValue != 200)
            {
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }
        }

        //' Download a file to the local directory of the assembly. Keep the same file name.
        public void DownloadFile(string sFileName)
        {
            DownloadFile(sFileName, "", false);
        }

        //' Download a remote file to the local directory of the Assembly. Keep the same file name.
        public void DownloadFile(string sFileName, bool bResume)
        {
            DownloadFile(sFileName, "", bResume);
        }

        //'Download a remote file to a local file name. You must include a path.
        //'The local file name will be created or will be overwritten, but the path must exist.
        public void DownloadFile(string sFileName, string sLocalFileName)
        {
            DownloadFile(sFileName, sLocalFileName, false);
        }

        //' Download a remote file to a local file name. You must include a path. Set the 
        //' resume flag. The local file name will be created or will be overwritten, but the path must exist.
        public void DownloadFile(string sFileName, string sLocalFileName, bool bResume)
        {
            Stream st;
            Socket cSocket;
            long offset;
            long npos;

            if ( !m_bLoggedIn )
            {
                Login();
            }

            SetBinaryMode(true);

            if (sLocalFileName.Equals(""))
            {
                sLocalFileName = sFileName;
            }

            if (! File.Exists(sLocalFileName))
            {
                st = File.Create(sLocalFileName);
                st.Close();
            }

            output = new FileStream(sLocalFileName, FileMode.Open);
            cSocket = CreateDataSocket();
            offset = 0;

            if (bResume)
            {
                offset = output.Length;

                if (offset > 0)
                {
                    //'Send an FTP command to restart.
                    SendCommand("REST " + offset);
                    if (m_iRetValue != 350)
                    {
                        offset = 0;
                    }
                }

                if (offset > 0)
                {
                    npos = output.Seek(offset, SeekOrigin.Begin);
                }
            }
            //'Send an FTP command to retrieve a file.
            SendCommand("RETR " + sFileName);

            if (!(m_iRetValue == 150 || m_iRetValue == 125))
            {
                output.Close();
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }

            while (true)
            {
                Array.Clear(m_aBuffer, 0, m_aBuffer.Length);
                m_iBytes = cSocket.Receive(m_aBuffer, m_aBuffer.Length, 0);
                output.Write(m_aBuffer, 0, m_iBytes);

                if (m_iBytes <= 0)
                {
                    break;
                }
            }

            output.Close();
            if (cSocket.Connected)
            {
                cSocket.Close();
            }

            ReadReply();
            if (!(m_iRetValue == 226 || m_iRetValue == 250))
            {
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }

        }

        //'This is a function that is used to upload a file from your local hard disk to your FTP site.
        public void UploadFile(string sFileName)
        {
            UploadFile(sFileName, false);
        }

        //' This is a function that is used to upload a file from your local hard disk to your FTP site 
        //' and { set the resume flag.
        public void UploadFile(string sFileName, bool bResume)
        {
            Socket cSocket;
            long offset;
            FileStream input;
            bool bFileNotFound;

            if (!m_bLoggedIn)
            {
                Login();
            }

            cSocket = CreateDataSocket();
            offset = 0;

            if (bResume) {
                try
                {
                    SetBinaryMode(true);
                    offset = GetFileSize(sFileName);
                }
                catch( Exception )
                {
                    offset = 0;
                }
            }

            if (offset > 0) {
                SendCommand("REST " + offset);
                if (m_iRetValue != 350) {

                    //'The remote server may not support resuming.
                    offset = 0;
                }
            }
            //'Send an FTP command to store a file.
            SendCommand("STOR " + Path.GetFileName(sFileName));
            if (! (m_iRetValue == 125 || m_iRetValue == 150)) {
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }

            //'Check to see if the file exists before the upload.
            bFileNotFound = false;
            if (File.Exists(sFileName)) {
                //' Open the input stream to read the source file.
                input = new FileStream(sFileName, FileMode.Open);
                if (offset != 0) {
                    input.Seek(offset, SeekOrigin.Begin);
                }

                //'Upload the file. 
                m_iBytes = input.Read(m_aBuffer, 0, m_aBuffer.Length);
                while (m_iBytes > 0){
                    cSocket.Send(m_aBuffer, m_iBytes, 0);
                    m_iBytes = input.Read(m_aBuffer, 0, m_aBuffer.Length);
                }
                input.Close();
            }
            else
            {
                bFileNotFound = true;
            }

            if (cSocket.Connected) {
                cSocket.Close();
            }

            //'Check the return value if the file was not found.
            if (bFileNotFound) {
                MessageString = m_sReply;
                throw new IOException("The file: " + sFileName + " was not found. " + "Cannot upload the file to the FTP site");
            }

            ReadReply();
            if (! (m_iRetValue == 226 || m_iRetValue == 250)) {
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }
        }

        //' Delete a file from the remote FTP server.
        public bool DeleteFile(string sFileName)
        {
            bool bResult;

            bResult = true;
            if ( !m_bLoggedIn )
            {
                Login();
            }
            //'Send an FTP command to delete a file.
            SendCommand("DELE " + sFileName);
            if (m_iRetValue != 250)
            {
                bResult = false;
                MessageString = m_sReply;
            }

            //' return the final result.
            return bResult;
        }

        //' Rename a file on the remote FTP server.
        public bool RenameFile(string sOldFileName, string sNewFileName)
        {
            bool bResult;

            bResult = true;
            if ( !m_bLoggedIn )
            {
                Login();
            }

            //'Send an FTP command to rename from a file.
            SendCommand("RNFR " + sOldFileName);
            if (m_iRetValue != 350)
            {
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }

            //'Send an FTP command to rename a file to a new file name.
            //'It will overwrite if newFileName exists. 
            SendCommand("RNTO " + sNewFileName);
            if (m_iRetValue != 250)
            {
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }
            //' return the final result.
            return bResult;
        }

        //'This is a function that is used to create a directory on the remote FTP server.
        public bool CreateDirectory(string sDirName)
        {
            bool bResult;

            bResult = true;
            if ( !m_bLoggedIn )
            {
                Login();
            }

            //'Send an FTP command to make directory on the FTP server.
            SendCommand("MKD " + sDirName);
            if (m_iRetValue != 257)
            {
                bResult = false;
                MessageString = m_sReply;
            }

            //' return the final result.
            return bResult;
        }

        //' This is a function that is used to delete a directory on the remote FTP server.
        public bool RemoveDirectory(string sDirName)
        {
            bool bResult;

            bResult = true;
            //'Check if logged on to the FTP server
            if ( !m_bLoggedIn )
            {
                Login();
            }
            //'Send an FTP command to remove directory on the FTP server.
            SendCommand("RMD " + sDirName);
            if (m_iRetValue != 250)
            {
                bResult = false;
                MessageString = m_sReply;
            }

            //' return the final result.
            return bResult;
        }

        //'This is a function that is used to change the current working directory on the remote FTP server.
        public bool ChangeDirectory(string sDirName)
        {
            bool bResult;

            bResult = true;
            //'Check if you are in the root directory.
            if (sDirName.Equals("."))
            {
                return bResult;
            }
            //'Check if logged on to the FTP server
            if ( !m_bLoggedIn )
            {
                Login();
            }
            //'Send an FTP command to change directory on the FTP server.
            SendCommand("CWD " + sDirName);
            if (m_iRetValue != 250)
            {
                bResult = false;
                MessageString = m_sReply;
            }

            RemotePath = sDirName;

            //' return the final result.
            return bResult;
        }

        //' Close the FTP connection of the remote server.
        public void CloseConnection()
        {
            if (m_objClientSocket != null)
            {
                //'Send an FTP command to end an FTP server system.
                SendCommand("QUIT");
            }

            Cleanup();
        }

#endregion

#region private Subs and Functions

        //' Read the reply from the FTP server.
        private void ReadReply()
        {
            m_sMes = "";
            m_sReply = ReadLine();
            m_iRetValue = Int32.Parse(m_sReply.Substring(0, 3));
        }

        //' Clean up some variables.
        private void Cleanup()
        {
            if (m_objClientSocket != null)
            {
                m_objClientSocket.Close();
                m_objClientSocket = null;
            }

            m_bLoggedIn = false;
        }

        //' Read a line from the FTP server.
        private string ReadLine(bool bClearMes = false)
        {
            char seperator = '\r';
            string[] mess;

            if (bClearMes)
            {
                m_sMes = "";
            }
            while (true)
            {
                Array.Clear(m_aBuffer, 0, BLOCK_SIZE);
                m_iBytes = m_objClientSocket.Receive(m_aBuffer, BLOCK_SIZE, SocketFlags.None);
                m_sMes += ASCII.GetString(m_aBuffer, 0, m_iBytes);
                if (m_iBytes < m_aBuffer.Length)
                {
                    break;
                }
            }

            mess = m_sMes.Split(seperator);
            if (m_sMes.Length > 2)
            {
                m_sMes = mess[mess.Length - 2];
            }
            else
            {
                m_sMes = mess[0];
            }

            if (!(m_sMes.Substring(3, 1).Equals(" ")))
            {
                return ReadLine(true);
            }
            Console.WriteLine(m_sMes);
            return m_sMes;
        }

        //' This is a function that is used to send a command to the FTP server that you are connected to.
        private void SendCommand(string sCommand)
        {
            Console.WriteLine(sCommand);

            sCommand = sCommand + "\r\n";
            byte[] cmdbytes = ASCII.GetBytes(sCommand);
            m_objClientSocket.Send(cmdbytes, cmdbytes.Length, 0);
            ReadReply();
        }

        //' Create a data socket.
        private Socket CreateDataSocket()
        {
            Int32 index1;
            Int32 index2;
            Int32 len;
            Int32 partCount;
            Int32 i;
            Int32 port;
            string ipData;
            string ipAddress;
            Int32[] parts = new Int32[6];
            Socket s;
            IPEndPoint ep;
            string[] bufs;

            //'Send an FTP command to use passive data connection. 
            SendCommand("PASV");
            if (m_iRetValue != 227)
            {
                MessageString = m_sReply;
                throw new IOException(m_sReply.Substring(4));
            }

            index1 = m_sReply.IndexOf("(");
            index2 = m_sReply.IndexOf(")");
            ipData = m_sReply.Substring(index1 + 1, index2 - index1 - 1);

            len = ipData.Length;
            partCount = 0;
//            buf = "";
            bufs = ipData.Split(',');

            for (i = 0; i < bufs.Length; i++)
            {
                try
                {
                    parts[i] = Int32.Parse(bufs[i]);
                }
                catch (Exception)
                {
                    MessageString = m_sReply;
                    throw new IOException("Malformed PASV reply: " + m_sReply);
                }
            }

            ipAddress = parts[0] + "." + parts[1] + "." + parts[2] + "." + parts[3];

            //' Make this call in Visual Basic .NET 2002. You want to 
            //' bitshift the number by 8 bits. In Visual Basic .NET 2002 you must
            //' multiply the number by 2 to the power of 8.
            //'port = parts(4) * (2 ^ 8)

            //' Make this call and { comment out the previous line for Visual Basic .NET 2003.
            port = parts[4] << 8;

            //' Determine the data port number.
            port = port + parts[5];

            s = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
//            ep = new IPEndPoint(Dns.GetHostEntry(ipAddress).AddressList[0], port);

            try
            {
                s.Connect(ipAddress, port);
            }
            catch (Exception)
            {
                MessageString = m_sReply;
                throw new IOException("Cannot connect to remote server.");
                //'if you cannot connect to the FTP server that is 
                //'specified, make the boolean variable false.
            }

            //'if you can connect to the FTP server that is specified, make the boolean variable true.
            flag_bool = true;
            return s;
        }
    }
#endregion
}
