#!/bin/bash
#License - Attribution-NonCommercial 4.0 International

# Notes:

# --with-newlib
# Since a working C library is not yet available, this ensures that the inhibit_libc constant
# is defined when building libgcc. This prevents the compiling of any code that requires libc support.

# patches for building binutils, gcc and gdb.

set -e

# --------------------
# Global variables
export TARGET=arm-none-symbianelf
GCCC=gcc-10.1.0
BINUTILS=binutils-2.33.1
GDB=gdb-8.3.1

WGET=wget
# --------------------
if [ ! -d ./$GDB ] ; then
    $WGET http://ftp.gnu.org/pub/gnu/gdb/$GDB.tar.xz
    tar -xf $GDB.tar.xz
fi
if [ ! -d ./$GCCC ] ; then
    $WGET http://gcc.gnu.org/pub/gcc/releases/$GCCC/$GCCC.tar.xz
    tar -xf $GCCC.tar.xz
fi
if [ ! -d ./$BINUTILS ] ; then
    $WGET http://ftp.gnu.org/pub/gnu/binutils/$BINUTILS.tar.xz
    tar -xf $BINUTILS.tar.xz
fi

# --------------------
# Installation folder
export PREFIX=/usr/local/$GCCC
export PATH=$PATH:$PREFIX/bin
export CONFIGURE=$GCCC/libstdc++-v3/configure
unset CFLAGS
export CFLAGS+="-pipe"

if [ ! -f ./first-pass-finished ] ; then

# ------------------
echo "Bulding binutils pass started"

find='^inline uint16_t'
replace='__attribute__((always_inline)) inline uint16_t'
sed -i -e 's/'"$find"'/'"$replace"'/g' $BINUTILS/libctf/swap.h

find='^inline uint32_t'
replace='__attribute__((always_inline)) inline uint32_t'
sed -i -e 's/'"$find"'/'"$replace"'/g' $BINUTILS/libctf/swap.h

find='^inline uint64_t'
replace='__attribute__((always_inline)) inline uint64_t'
sed -i -e 's/'"$find"'/'"$replace"'/g' $BINUTILS/libctf/swap.h

touch first-pass-started
if [ -d ./build-binutils ] ; then
    rm -rf ./build-binutils
    mkdir build-binutils
else
    mkdir build-binutils
fi

cd build-binutils
../$BINUTILS/configure --target=$TARGET --prefix=$PREFIX --disable-option-checking \
--enable-ld --enable-gold --enable-lto --enable-vtable-verify \
--enable-werror=no --without-headers --disable-nls --disable-shared \
--disable-libquadmath --enable-plugins --enable-multilib

make
make install-strip

cd ..
touch first-pass-finished
echo "Bulding binutils pass finished"

fi

if [ ! -f ./build-gcc-finished ] ; then

# _____________
echo "Copyng gcc dependency libs started"

MPC=mpc-1.1.0
ISL=isl-0.22.1
GMP=gmp-6.1.2
MPFR=mpfr-4.0.2

if [ ! -d ./$MPC ] ; then
    $WGET http://ftp.gnu.org/pub/gnu/mpc/$MPC.tar.gz
    tar -xf $MPC.tar.gz
fi
if [ ! -d ./$ISL ] ; then
    $WGET http://isl.gforge.inria.fr/$ISL.tar.xz
    tar -xf $ISL.tar.xz
fi
if [ ! -d ./$GMP ] ; then
    $WGET http://ftp.gnu.org/pub/gnu/gmp/$GMP.tar.xz
    tar -xf $GMP.tar.xz
fi
if [ ! -d ./$MPFR ] ; then
    $WGET http://ftp.gnu.org/pub/gnu/mpfr/$MPFR.tar.xz
    tar -xf $MPFR.tar.xz
fi
if [ ! -d $GCCC/mpc ] ; then
    cp -Rp $MPC $GCCC/mpc
fi
if [ ! -d $GCCC/isl ] ; then
    cp -Rp $ISL $GCCC/isl
fi
if [ ! -d $GCCC/gmp ] ; then
    cp -Rp $GMP $GCCC/gmp
fi
if [ ! -d $GCCC/mpfr ] ; then
    cp -Rp $MPFR $GCCC/mpfr
fi

echo "Copyng gcc dependency libs finished"

# _____________
unset CFLAGS
export CFLAGS+="-pipe"
if [ -d ./build-gcc ] ; then
    rm -rf ./build-gcc
    mkdir build-gcc
else
    mkdir build-gcc
fi

echo "Bulding gcc started"

# patch for the EOF, SEEK_CUR, and SEEK_END integer constants
# because autoconf can't set them
find='as_fn_error $? "computing EOF failed" "$LINENO" 5'
replace='$as_echo "computing EOF failed" "$LINENO" >\&5'
sed -i -e 's/'"$find"'/'"$replace"'/g' $CONFIGURE
find='as_fn_error $? "computing SEEK_CUR failed" "$LINENO" 5'
replace='$as_echo "computing SEEK_CUR failed" "$LINENO" >\&5'
sed -i -e 's/'"$find"'/'"$replace"'/g' $CONFIGURE
find='as_fn_error $? "computing SEEK_END failed" "$LINENO" 5'
replace='$as_echo "computing SEEK_END failed" "$LINENO" >\&5'
sed -i -e 's/'"$find"'/'"$replace"'/g' $CONFIGURE

# patch for the void, int, short and long
# because autoconf can't set them
find='if ac_fn_c_compute_int "$LINENO" "(long int) (sizeof (void \*))" "ac_cv_sizeof_void_p"        "$ac_includes_default"'
replace='if ac_fn_c_compute_int "$LINENO" "(long int) (sizeof (void \*))" "ac_cv_sizeof_void_p"'
sed -i -e 's/'"$find"'/'"$replace"'/g' $CONFIGURE

find='if ac_fn_c_compute_int "$LINENO" "(long int) (sizeof (long))" "ac_cv_sizeof_long"        "$ac_includes_default"'
replace='if ac_fn_c_compute_int "$LINENO" "(long int) (sizeof (long))" "ac_cv_sizeof_long"'
sed -i -e 's/'"$find"'/'"$replace"'/g' $CONFIGURE

find='if ac_fn_c_compute_int "$LINENO" "(long int) (sizeof (int))" "ac_cv_sizeof_int"        "$ac_includes_default"'
replace='if ac_fn_c_compute_int "$LINENO" "(long int) (sizeof (int))" "ac_cv_sizeof_int"'
sed -i -e 's/'"$find"'/'"$replace"'/g' $CONFIGURE

find='if ac_fn_c_compute_int "$LINENO" "(long int) (sizeof (short))" "ac_cv_sizeof_short"        "$ac_includes_default"'
replace='if ac_fn_c_compute_int "$LINENO" "(long int) (sizeof (short))" "ac_cv_sizeof_short"'
sed -i -e 's/'"$find"'/'"$replace"'/g' $CONFIGURE

find='if ac_fn_c_compute_int "$LINENO" "(long int) (sizeof (char))" "ac_cv_sizeof_char"        "$ac_includes_default"'
replace='if ac_fn_c_compute_int "$LINENO" "(long int) (sizeof (char))" "ac_cv_sizeof_char"'
sed -i -e 's/'"$find"'/'"$replace"'/g' $CONFIGURE

# find=''
# replace=''
# sed -i -e 's/'"$find"'/'"$replace"'/g' $CONFIGURE

find='^#include <stdint\.h>'
replace='\/\/ #include <stdint.h>'
sed -i -e 's/'"$find"'/'"$replace"'/g' $GCCC/libstdc++-v3/libsupc++/new_opa.cc

# msys2 / MINGW-packages / mingw-w64-gcc / 0015-makedeps-properly-handle-win-paths.patch
find='^\t    case '\'':'\'':'
replace='\t    \/\/ case '\'':'\'':'
sed -i -e 's/'"$find"'/'"$replace"'/g' $GCCC/libcpp/mkdeps.c



touch build-gcc-started
cd build-gcc
../$GCCC/configure  --target=$TARGET --prefix=$PREFIX  --without-headers \
	--enable-languages="c,c++,lto" --enable-poison-system-directories \
	--enable-lto --with-newlib \
	--with-gnu-as --with-gnu-ld --with-dwarf2 \
	--disable-hosted-libstdcxx --disable-libstdcxx-pch \
	--disable-option-checking --disable-threads --disable-nls \
	--disable-win32-registry --disable-libssp --disable-shared \
	--enable-interwork --enable-tls --enable-multilib \
	--enable-wchar_t --enable-extra-sgxxlite-multilibs --enable-c99 \
	--enable-long-long
	# --with-sysroot

make
make install-strip

cd ..
touch build-gcc-finished
echo "Bulding gcc finished"

fi

unset CFLAGS
export CFLAGS+="-pipe"
unset LIBS
export LIBS+="-lssp"
if [ -d ./build-gdb ] ; then
    rm -rf ./build-gdb
    mkdir build-gdb
else
    mkdir build-gdb
fi

# ______________________

find='^struct maverick_regs DSPregs\[16\];'
replace='static struct maverick_regs DSPregs[16];'
sed -i -e 's/'"$find"'/'"$replace"'/g' $GDB/sim/arm/maverick.c

find='^union maverick_acc_regs DSPacc\[4\];'
replace='\/\/ union maverick_acc_regs DSPacc[4];'
sed -i -e 's/'"$find"'/'"$replace"'/g' $GDB/sim/arm/maverick.c

find='^ARMword DSPsc;'
replace='static ARMword DSPsc;'
sed -i -e 's/'"$find"'/'"$replace"'/g' $GDB/sim/arm/maverick.c

find='^struct maverick_regs     DSPregs\[16\];'
replace='static struct maverick_regs     DSPregs[16];'
sed -i -e 's/'"$find"'/'"$replace"'/g' $GDB/sim/arm/wrapper.c

find='^union maverick_acc_regs  DSPacc\[4\];'
replace='\/\/ union maverick_acc_regs  DSPacc[4];'
sed -i -e 's/'"$find"'/'"$replace"'/g' $GDB/sim/arm/wrapper.c

find='^ARMword DSPsc;'
replace='static ARMword DSPsc;'
sed -i -e 's/'"$find"'/'"$replace"'/g' $GDB/sim/arm/wrapper.c

find='^ARMword isize;'
replace='#ifdef MODE32\nARMword    isize;\n#endif'
sed -i -e 's/'"$find"'/'"$replace"'/g' $GDB/sim/arm/armemu.c

touch gdb-started
cd build-gdb

../$GDB/configure --target=$TARGET --prefix=$PREFIX --disable-nls --disable-shared
make
make install

cd ..
touch gdb-finished

rundll32 powrprof.dll,SetSuspendState 0,1,0
