﻿'Copyright (C) 2010 pepetaro, All rights reserved.
'This library is free software; you can redistribute it and/or
'modify it under the terms of the GNU Lesser General Public
'License as published by the Free Software Foundation; either
'version 3 of the License, or (at your option) any later version.

'This library is distributed in the hope that it will be useful,
'but WITHOUT ANY WARRANTY; without even the implied warranty of
'MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
'Lesser General Public License for more details.

'You should have received a copy of the GNU Lesser General Public
'License along with this library; if not, write to the Free Software
'Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

Namespace TwitterServices
    ''' <summary>xAuthやPINを使った認証を行います。</summary>
    Public NotInheritable Class AuthenticateService
        Friend Sub New(ByVal tw As Twitter)
            Me.tw = tw
        End Sub
        <DebuggerBrowsable(DebuggerBrowsableState.Never)> _
        Dim tw As Twitter

        '''<summary>
        '''認証成功時の応答でユーザー情報を取得する場合のキー。設定しない場合は、AuthUsernameもブランクのままとなる
        '''</summary>
        Private userIdentKey As String

        '''<summary>
        '''OAuthの署名作成用秘密コンシューマーデータ
        '''</summary>
        Private authorizedUsername As String

        '''<summary>
        '''OAuth認証の開始要求（リクエストトークン取得）。PIN入力用の前段
        '''</summary>
        '''<remarks>
        '''呼び出し元では戻されたurlをブラウザで開き、認証完了後PIN入力を受け付けて、リクエストトークンと共にAuthenticatePinFlowを呼び出す
        '''</remarks>
        '''<param name="requestTokenUrl">リクエストトークンの取得先URL</param>
        '''<param name="authorizeUrl">ブラウザで開く認証用URLのベース</param>
        '''<param name="requestToken">[OUT]認証要求で戻されるリクエストトークン。使い捨て</param>
        '''<param name="authUri">[OUT]requestUriを元に生成された認証用URL。通常はリクエストトークンをクエリとして付加したUri</param>
        '''<returns>取得結果真偽値</returns>
        Public Function AuthenticatePinFlowRequest(ByVal requestTokenUrl As String, _
                                            ByVal authorizeUrl As String, _
                                            ByRef requestToken As String, _
                                            ByRef authUri As Uri) As Boolean
            'PIN-based flow
            authUri = GetAuthenticatePageUri(requestTokenUrl, authorizeUrl, requestToken)
            If authUri Is Nothing Then Return False
            Return True
        End Function

        '''<summary>
        '''OAuth認証のアクセストークン取得。PIN入力用の後段
        '''</summary>
        '''<remarks>
        '''事前にAuthenticatePinFlowRequestを呼んで、ブラウザで認証後に表示されるPINを入力してもらい、その値とともに呼び出すこと
        '''</remarks>
        '''<param name="accessTokenUrl">アクセストークンの取得先URL</param>
        '''<param name="requestToken">AuthenticatePinFlowRequestで取得したリクエストトークン</param>
        '''<param name="pinCode">Webで認証後に表示されるPINコード</param>
        '''<returns>取得結果真偽値</returns>
        Public Function AuthenticatePinFlow(ByVal accessTokenUrl As String, _
                                            ByVal requestToken As String, _
                                            ByVal pinCode As String) As Boolean
            'PIN-based flow
            If String.IsNullOrEmpty(requestToken) Then Throw New Exception("Sequence error.(requestToken is blank)")

            'アクセストークン取得
            Dim accessTokenData As NameValueCollection = GetOAuthToken(New Uri(accessTokenUrl), pinCode, requestToken, Nothing)

            If accessTokenData IsNot Nothing Then
                tw.AccessToken = New PublicSecretPair
                tw.AccessToken.Public = accessTokenData.Item("oauth_token")
                tw.AccessToken.Secret = accessTokenData.Item("oauth_token_secret")
                'サービスごとの独自拡張対応
                If Me.userIdentKey <> "" Then
                    authorizedUsername = accessTokenData.Item(Me.userIdentKey)
                Else
                    authorizedUsername = ""
                End If
                If tw.AccessToken.Public = "" Then Return False
                Return True
            Else
                Return False
            End If
        End Function

        '''<summary>
        '''OAuth認証のアクセストークン取得。xAuth方式
        '''</summary>
        '''<param name="accessTokenUrl">アクセストークンの取得先URL</param>
        '''<param name="username">認証用ユーザー名</param>
        '''<param name="password">認証用パスワード</param>
        '''<returns>取得結果真偽値</returns>
        ''' <remarks>テスト環境がないので全くの未テスト</remarks>
        Public Function AuthenticateXAuth(ByVal accessTokenUrl As String, ByVal username As String, ByVal password As String) As Boolean
            'ユーザー・パスワードチェック
            If String.IsNullOrEmpty(username) OrElse String.IsNullOrEmpty(password) Then
                Throw New Exception("Sequence error.(username or password is blank)")
            End If
            'xAuthの拡張パラメータ設定
            Dim parameter As New Dictionary(Of String, String)
            parameter.Add("x_auth_mode", "client_auth")
            parameter.Add("x_auth_username", username)
            parameter.Add("x_auth_password", password)

            'アクセストークン取得
            Dim accessTokenData As NameValueCollection = GetOAuthToken(New Uri(accessTokenUrl), "", "", parameter)

            If accessTokenData IsNot Nothing Then
                tw.AccessToken = New PublicSecretPair
                tw.AccessToken.Public = accessTokenData.Item("oauth_token")
                tw.AccessToken.Secret = accessTokenData.Item("oauth_token_secret")
                'サービスごとの独自拡張対応
                If Me.userIdentKey <> "" Then
                    authorizedUsername = accessTokenData.Item(Me.userIdentKey)
                Else
                    authorizedUsername = ""
                End If
                If tw.AccessToken.Public = "" Then Return False
                Return True
            Else
                Return False
            End If
        End Function

        '''<summary>
        '''OAuth認証のリクエストトークン取得。リクエストトークンと組み合わせた認証用のUriも生成する
        '''</summary>
        '''<param name="requestTokenUrl">リクエストトークンの取得先URL</param>
        '''<param name="authorizeUrl">ブラウザで開く認証用URLのベース</param>
        '''<param name="requestToken">[OUT]取得したリクエストトークン</param>
        '''<returns>取得結果真偽値</returns>
        Private Function GetAuthenticatePageUri(ByVal requestTokenUrl As String, _
                                            ByVal authorizeUrl As String, _
                                            ByRef requestToken As String) As Uri
            Const tokenKey As String = "oauth_token"

            'リクエストトークン取得
            Dim reqTokenData As NameValueCollection = GetOAuthToken(New Uri(requestTokenUrl), "", "", Nothing)
            If reqTokenData IsNot Nothing Then
                requestToken = reqTokenData.Item(tokenKey)
                'Uri生成
                Dim ub As New UriBuilder(authorizeUrl)
                ub.Query = String.Format("{0}={1}", tokenKey, requestToken)
                Return ub.Uri
            Else
                Return Nothing
            End If
        End Function

        '''<summary>
        '''OAuth認証のトークン取得共通処理
        '''</summary>
        '''<param name="requestUri">各種トークンの取得先URL</param>
        '''<param name="pinCode">PINフロー時のアクセストークン取得時に設定。それ以外は空文字列</param>
        '''<param name="requestToken">PINフロー時のリクエストトークン取得時に設定。それ以外は空文字列</param>
        '''<param name="parameter">追加パラメータ。xAuthで使用</param>
        '''<returns>取得結果のデータ。正しく取得出来なかった場合はNothing</returns>
        Private Function GetOAuthToken(ByVal requestUri As Uri, ByVal pinCode As String, ByVal requestToken As String, ByVal parameter As Dictionary(Of String, String)) As NameValueCollection
            Dim webReq As HttpWebRequest = Nothing
            'HTTPリクエスト生成。PINコードもパラメータも未指定の場合はGETメソッドで通信。それ以外はPOST
            If String.IsNullOrEmpty(pinCode) AndAlso parameter Is Nothing Then
                webReq = TwitterConnection.CreateRequest("GET", requestUri, Nothing)
            Else
                webReq = TwitterConnection.CreateRequest("POST", requestUri, parameter) 'ボディに追加パラメータ書き込み
            End If
            'OAuth関連パラメータ準備。追加パラメータがあれば追加
            Dim query As New Dictionary(Of String, String)
            If parameter IsNot Nothing Then
                For Each kvp As KeyValuePair(Of String, String) In parameter
                    query.Add(kvp.Key, kvp.Value)
                Next
            End If
            'PINコードが指定されていればパラメータに追加
            If Not String.IsNullOrEmpty(pinCode) Then query.Add("oauth_verifier", pinCode)
            'OAuth関連情報をHTTPリクエストに追加
            tw.twiCon.AppendOAuthInfo(webReq, query, requestToken, "")
            'HTTP応答取得
            Try
                Dim contentText As String = ""
                Dim status As HttpStatusCode = tw.twiCon.GetResponse(webReq, contentText, Nothing)
                If status = HttpStatusCode.OK Then
                    Return tw.twiCon.ParseQueryString(contentText)
                Else
                    Return Nothing
                End If
            Catch ex As Exception
                Return Nothing
            End Try
        End Function

    End Class
End Namespace