/**
 * This load profile represents an increase in load from zero to the maximum load, which is then maintained for a
 * period. The time spent at maxLoad can be adjusted all the way down to zero, giving us a variety of possible curves:
 *
 *  ▲                 ▲
 *  │                 │   / maxLoad
 *  │                 │  /
 *  │ ─── maxLoad     │ /
 *  │/                │/
 *  └────►            └────►
 *
 * The different variations are available based on the ramp duration and the total duration:
 *
 *  1 Ramp duration is less than total duration: The profile ramps up to maxLoad and holds for the remaining duration.
 *  2 Ramp duration is equal to the total duration: The profile ramps up to maxLoad over the total duration.
 *  3 Ramp duration is greater than total duration: This is an exceptional case and the profile will clip the ramp
 *    duration to match the total duration, giving us the same characteristics as (2).
 */
module.exports = {
  id: 'ramp',
  params: ['maxLoad', 'totalDurationMs', 'rampDurationMs', 'elapsedMs'],

  /**
   * A ramp is calculated using a linear function. The constant component of the profile, if it exists, is just
   * returning maxLoad.
   *
   * @param  {Number} params.maxLoad - The maximum load this load profile should evaluate to.
   * @param  {Number} params.totalDurationMs - The total duration, in milliseconds, of the load profile.
   * @param  {Number} [params.rampDurationMs] - The duration of the load profile, in milliseconds, during which the load
   *                                            is "ramping up" from zero to maxLoad. This duration should be less than
   *                                            or equal to totalDurationMs. totalDurationMs will be used instead if
   *                                            this value is greater.
   * @param  {Number} params.elapsedMs - The time, in milliseconds, at which the
   * @return {Number} The evaluated load for this point in time.
   */
  evaluate ({ maxLoad, totalDurationMs, rampDurationMs = 0, elapsedMs }) {
    // Total duration must be greater than 0 for our evaluation logic to be valid, otherwise we run a risk of dividing
    // by zero. On top of that, a negative total duration just makes no sense. 😐
    if (totalDurationMs <= 0) {
      throw new RangeError('Total duration must be a non-zero positive integer.');
    }

    // Other timings should be positive, but they can be zero as well. A zero ramp duration gives us a constant load
    // profile.
    if (rampDurationMs < 0 || elapsedMs < 0) {
      throw new RangeError('Ramp duration and elapsed time should be positive integers or 0.');
    }

    // A maxLoad of 0 isn't necessarily invalid, but it's best to avoid weird behavior unless we have a specific use
    // case for it.
    if (maxLoad <= 0) {
      throw new RangeError('Maximum load must be a non-zero positive integer.');
    }

    // This invocation is invalid if the elapsed time is greater than the total time
    if (elapsedMs > totalDurationMs) {
      throw new RangeError('Elapsed time cannot be more than total duration.');
    }

    // Ramp duration is optional, so default to the totalDuration if it's not provided. This creates a zero to maxLoad
    // ramp over the whole duration as a special case.
    if (rampDurationMs === 0) {
      rampDurationMs = totalDurationMs;
    }

    // Evaluate the ramp only if we're still within the ramp window. Everything after that is constant.
    if (elapsedMs < rampDurationMs) {
      const effectiveRamp = Math.min(rampDurationMs, totalDurationMs);
      const elapsedPercent = elapsedMs / effectiveRamp;
      const proportialLoad = elapsedPercent * maxLoad;

      // Make sure we're returning an integer. Javascript's round method rounds away from 0 if the fractional portion is
      // > 0.5, towards 0 if it's < 0.5, and towards +∞ if it's exactly 0.5. We also want to make sure we never return
      // anything less than 1 -- even if the load is 0 we want to have some load instead of zero load.
      return Math.max(Math.round(proportialLoad), 1);
    }

    return maxLoad;
  }
};
