
--
-- Copyright (C) 2020  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--

with system;
with gl, gl.binding, gl.pointers;
with glu, glu.binding, glu.pointers;
with glext, glext.binding, glext.pointers;

with interfaces.c;
use type interfaces.c.unsigned_short;

with matutils; use matutils;

with ada.finalization;
with unchecked_deallocation;


	with text_io;

package body mrectobj is -- for textured rectangular exterior, eg. partition wall

-- Intended for objects bigger than 1 unit on a side...
-- maps texture coordinates UV to cover a portion of a side;
-- thus multiple copies of *.png file may be visible



procedure initialize( rect: in out rectangle ) is
begin
	rect.vert := new varray;
	rect.txuv := new tarray;
	rect.elem := new earray;
	rect.norm := new varray;
end initialize;

procedure vfree is new unchecked_deallocation(varray,vap);
procedure tfree is new unchecked_deallocation(tarray,tap);
procedure efree is new unchecked_deallocation(earray,eap);

procedure finalize( rect: in out rectangle ) is
begin
	vfree( rect.vert );
	tfree( rect.txuv );
	efree( rect.elem );
	vfree( rect.norm );
	--text_io.put_line("rect Free");
end finalize;





procedure setrect( rect: rectangle;
nu, nv : integer;
xc,yc,zc, xr,yr,zr : float;
xm,xp,ym,yp,zm,zp : out float
) is

	k, ejj, j : integer := 0;
	jj : glushort;
	umax,vmax : float; --25nov14

	ax,ay,az,bx,by,bz,nx,ny,nz: float;

begin

	xm  := xc-xr;
	xp  := xc+xr;
	ym  := yc-yr;
	yp  := yc+yr;
	zm  := zc-zr;
	zp  := zc+zr;

	-- Z+ ccw exterior normal
	rect.vert(k+ 1):=xm;  rect.vert(k+ 2):=ym;  rect.vert(k+ 3):=zp; --LL
	rect.vert(k+ 4):=xp;  rect.vert(k+ 5):=ym;  rect.vert(k+ 6):=zp; --LR
	rect.vert(k+ 7):=xp;  rect.vert(k+ 8):=yp;  rect.vert(k+ 9):=zp; --UR
	rect.vert(k+10):=xm;  rect.vert(k+11):=yp;  rect.vert(k+12):=zp; --UL
-------- begin insert ---------------------------------------
	ax:=rect.vert(k+1)-rect.vert(k+4);
	ay:=rect.vert(k+2)-rect.vert(k+5);
	az:=rect.vert(k+3)-rect.vert(k+6);
	bx:=rect.vert(k+7)-rect.vert(k+4);
	by:=rect.vert(k+8)-rect.vert(k+5);
	bz:=rect.vert(k+9)-rect.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- up X lf(ea) = No = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rect.norm(k+ 1):=nx;  rect.norm(k+ 2):=ny;  rect.norm(k+ 3):=nz;
	rect.norm(k+ 4):=nx;  rect.norm(k+ 5):=ny;  rect.norm(k+ 6):=nz;
	rect.norm(k+ 7):=nx;  rect.norm(k+ 8):=ny;  rect.norm(k+ 9):=nz;
	rect.norm(k+10):=nx;  rect.norm(k+11):=ny;  rect.norm(k+12):=nz;
-------- end insert ---------------------------------------
	k:=k+12;

	-- Y+  ccw exterior normal
	rect.vert(k+ 1):=xm;  rect.vert(k+ 2):=yp;  rect.vert(k+ 3):=zp; --NE
	rect.vert(k+ 4):=xp;  rect.vert(k+ 5):=yp;  rect.vert(k+ 6):=zp; --NW
	rect.vert(k+ 7):=xp;  rect.vert(k+ 8):=yp;  rect.vert(k+ 9):=zm; --SW
	rect.vert(k+10):=xm;  rect.vert(k+11):=yp;  rect.vert(k+12):=zm; --SE
-------- begin insert ---------------------------------------
	ax:=rect.vert(k+1)-rect.vert(k+4);
	ay:=rect.vert(k+2)-rect.vert(k+5);
	az:=rect.vert(k+3)-rect.vert(k+6);
	bx:=rect.vert(k+7)-rect.vert(k+4);
	by:=rect.vert(k+8)-rect.vert(k+5);
	bz:=rect.vert(k+9)-rect.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- so X ea = up
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rect.norm(k+ 1):=nx;  rect.norm(k+ 2):=ny;  rect.norm(k+ 3):=nz;
	rect.norm(k+ 4):=nx;  rect.norm(k+ 5):=ny;  rect.norm(k+ 6):=nz;
	rect.norm(k+ 7):=nx;  rect.norm(k+ 8):=ny;  rect.norm(k+ 9):=nz;
	rect.norm(k+10):=nx;  rect.norm(k+11):=ny;  rect.norm(k+12):=nz;
-------- end insert ---------------------------------------
	k:=k+12;


	-- Z-  ccw exterior normal
	rect.vert(k+ 1):=xp;  rect.vert(k+ 2):=ym;  rect.vert(k+ 3):=zm; --LL
	rect.vert(k+ 4):=xm;  rect.vert(k+ 5):=ym;  rect.vert(k+ 6):=zm; --LR
	rect.vert(k+ 7):=xm;  rect.vert(k+ 8):=yp;  rect.vert(k+ 9):=zm; --UR
	rect.vert(k+10):=xp;  rect.vert(k+11):=yp;  rect.vert(k+12):=zm; --UL
-------- begin insert ---------------------------------------
	ax:=rect.vert(k+1)-rect.vert(k+4);
	ay:=rect.vert(k+2)-rect.vert(k+5);
	az:=rect.vert(k+3)-rect.vert(k+6);
	bx:=rect.vert(k+7)-rect.vert(k+4);
	by:=rect.vert(k+8)-rect.vert(k+5);
	bz:=rect.vert(k+9)-rect.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- up X lf(we) = So = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rect.norm(k+ 1):=nx;  rect.norm(k+ 2):=ny;  rect.norm(k+ 3):=nz;
	rect.norm(k+ 4):=nx;  rect.norm(k+ 5):=ny;  rect.norm(k+ 6):=nz;
	rect.norm(k+ 7):=nx;  rect.norm(k+ 8):=ny;  rect.norm(k+ 9):=nz;
	rect.norm(k+10):=nx;  rect.norm(k+11):=ny;  rect.norm(k+12):=nz;
-------- end insert ---------------------------------------
	k:=k+12;

	-- Y-  ccw exterior normal
	rect.vert(k+ 1):=xm;  rect.vert(k+ 2):=ym;  rect.vert(k+ 3):=zm; --SE
	rect.vert(k+ 4):=xp;  rect.vert(k+ 5):=ym;  rect.vert(k+ 6):=zm; --SW
	rect.vert(k+ 7):=xp;  rect.vert(k+ 8):=ym;  rect.vert(k+ 9):=zp; --NW
	rect.vert(k+10):=xm;  rect.vert(k+11):=ym;  rect.vert(k+12):=zp; --NE
-------- begin insert ---------------------------------------
	ax:=rect.vert(k+1)-rect.vert(k+4);
	ay:=rect.vert(k+2)-rect.vert(k+5);
	az:=rect.vert(k+3)-rect.vert(k+6);
	bx:=rect.vert(k+7)-rect.vert(k+4);
	by:=rect.vert(k+8)-rect.vert(k+5);
	bz:=rect.vert(k+9)-rect.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- no X ea = Dn = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rect.norm(k+ 1):=nx;  rect.norm(k+ 2):=ny;  rect.norm(k+ 3):=nz;
	rect.norm(k+ 4):=nx;  rect.norm(k+ 5):=ny;  rect.norm(k+ 6):=nz;
	rect.norm(k+ 7):=nx;  rect.norm(k+ 8):=ny;  rect.norm(k+ 9):=nz;
	rect.norm(k+10):=nx;  rect.norm(k+11):=ny;  rect.norm(k+12):=nz;
-------- end insert ---------------------------------------
	k:=k+12;


	-- east X- ccw exterior normal
	rect.vert(k+ 1):=xm;  rect.vert(k+ 2):=ym;  rect.vert(k+ 3):=zm; --LL
	rect.vert(k+ 4):=xm;  rect.vert(k+ 5):=ym;  rect.vert(k+ 6):=zp; --LR
	rect.vert(k+ 7):=xm;  rect.vert(k+ 8):=yp;  rect.vert(k+ 9):=zp; --UR
	rect.vert(k+10):=xm;  rect.vert(k+11):=yp;  rect.vert(k+12):=zm; --UL
-------- begin insert ---------------------------------------
	ax:=rect.vert(k+1)-rect.vert(k+4);
	ay:=rect.vert(k+2)-rect.vert(k+5);
	az:=rect.vert(k+3)-rect.vert(k+6);
	bx:=rect.vert(k+7)-rect.vert(k+4);
	by:=rect.vert(k+8)-rect.vert(k+5);
	bz:=rect.vert(k+9)-rect.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- up X lf(so) = Ea = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rect.norm(k+ 1):=nx;  rect.norm(k+ 2):=ny;  rect.norm(k+ 3):=nz;
	rect.norm(k+ 4):=nx;  rect.norm(k+ 5):=ny;  rect.norm(k+ 6):=nz;
	rect.norm(k+ 7):=nx;  rect.norm(k+ 8):=ny;  rect.norm(k+ 9):=nz;
	rect.norm(k+10):=nx;  rect.norm(k+11):=ny;  rect.norm(k+12):=nz;
-------- end insert ---------------------------------------
	k:=k+12;

	-- west X+ ccw exterior normal
	rect.vert(k+ 1):=xp;  rect.vert(k+ 2):=ym;  rect.vert(k+ 3):=zp; --LL
	rect.vert(k+ 4):=xp;  rect.vert(k+ 5):=ym;  rect.vert(k+ 6):=zm; --LR
	rect.vert(k+ 7):=xp;  rect.vert(k+ 8):=yp;  rect.vert(k+ 9):=zm; --UR
	rect.vert(k+10):=xp;  rect.vert(k+11):=yp;  rect.vert(k+12):=zp; --UL
-------- begin insert ---------------------------------------
	ax:=rect.vert(k+1)-rect.vert(k+4);
	ay:=rect.vert(k+2)-rect.vert(k+5);
	az:=rect.vert(k+3)-rect.vert(k+6);
	bx:=rect.vert(k+7)-rect.vert(k+4);
	by:=rect.vert(k+8)-rect.vert(k+5);
	bz:=rect.vert(k+9)-rect.vert(k+6);
	cross(bx,by,bz, ax,ay,az, nx,ny,nz); -- up X lf(no) = We = outward
	normalize(nx,ny,nz);
	----------------------------------------------------------
	rect.norm(k+ 1):=nx;  rect.norm(k+ 2):=ny;  rect.norm(k+ 3):=nz;
	rect.norm(k+ 4):=nx;  rect.norm(k+ 5):=ny;  rect.norm(k+ 6):=nz;
	rect.norm(k+ 7):=nx;  rect.norm(k+ 8):=ny;  rect.norm(k+ 9):=nz;
	rect.norm(k+10):=nx;  rect.norm(k+11):=ny;  rect.norm(k+12):=nz;
-------- end insert ---------------------------------------
	k:=k+12;


	-- texture UV coords for cube:
	for i in 0..5 loop
		j := i*8;
		if i=0 or i=2 then --fix Z (z+,z-)
			umax:= float(nu);
			vmax:= float(nv);
		elsif i=1 or i=3 then --fix Y (y+,y-)
			umax:= float(nu);
			vmax:= float(nv);
		else --fix X (x-,x+)
			umax:= float(nu);
			vmax:= float(nv);
		end if;

		rect.txuv(j+1):=0.0;   rect.txuv(j+2):=0.0;
		rect.txuv(j+3):=umax;  rect.txuv(j+4):=0.0;
		rect.txuv(j+5):=umax;  rect.txuv(j+6):=vmax;
		rect.txuv(j+7):=0.0;   rect.txuv(j+8):=vmax;


	end loop;

	-- element indices:
	for i in 0..5 loop
		jj:=glushort(i*4);
		j := i*6;
		rect.elem(j+1):=jj+0;
		rect.elem(j+2):=jj+1;
		rect.elem(j+3):=jj+2;
		rect.elem(j+4):=jj+2;
		rect.elem(j+5):=jj+3;
		rect.elem(j+6):=jj+0;
	end loop;


end setrect;



--
-- note:  the shaders for these objects must have two 
-- input "layouts", as well as whatever uniforms are needed:
--
-- layout(location=0) in vec3 vertPosName
-- layout(location=1) in vec3 vertRgbName
--
-- ...where their actual names can be whatever is convenient
--
use gl;
use glext;
use glext.binding;
use gl.binding;

procedure draw( rect: rectangle; vertbuff, uvbuff, elembuff : gluint ) is
begin

	-- 0th attribute:  vertices
	glBindBuffer(gl_array_buffer, vertbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nvert), rect.vert(1)'address, gl_static_draw);
	glEnableVertexAttribArray(0);
	glVertexAttribPointer(0,3,gl_float,gl_false,0, system.null_address);

	-- 1st attribute:  texture UV
	glBindBuffer(gl_array_buffer, uvbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nuv), rect.txuv(1)'address, gl_static_draw);
	glEnableVertexAttribArray(1);
	glVertexAttribPointer(1,2,gl_float,gl_false,0, system.null_address);

	-- element indices:
	glBindBuffer(gl_element_array_buffer, elembuff);
	glBufferData(gl_element_array_buffer, glsizeiptr(2*nelm), rect.elem(1)'address, gl_static_draw);

	glEnable(gl_blend);
	glBlendFunc(gl_src_alpha, gl_one_minus_src_alpha);

	glDrawElements( gl_triangles, glint(nvert), gl_unsigned_short, system.null_address );

	glDisableVertexAttribArray(0);
	glDisableVertexAttribArray(1);

end draw;





procedure ldraw( rect: rectangle; vertbuff, uvbuff, normbuff, elembuff : gluint ) is
begin

	-- 0th attribute:  vertices
	glBindBuffer(gl_array_buffer, vertbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nvert), rect.vert(1)'address, gl_static_draw);
	glEnableVertexAttribArray(0);
	glVertexAttribPointer(0,3,gl_float,gl_false,0, system.null_address);

	-- 1st attribute:  texture UV
	glBindBuffer(gl_array_buffer, uvbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nuv), rect.txuv(1)'address, gl_static_draw);
	glEnableVertexAttribArray(1);
	glVertexAttribPointer(1,2,gl_float,gl_false,0, system.null_address);

	-- 2nd attribute:  normals
	glBindBuffer(gl_array_buffer, normbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nvert), rect.norm(1)'address, gl_static_draw);
	glEnableVertexAttribArray(2);
	glVertexAttribPointer(2,3,gl_float,gl_false,0, system.null_address);


	-- element indices:
	glBindBuffer(gl_element_array_buffer, elembuff);
	glBufferData(gl_element_array_buffer, glsizeiptr(2*nelm), rect.elem(1)'address, gl_static_draw);

	glEnable(gl_blend);
	glBlendFunc(gl_src_alpha, gl_one_minus_src_alpha);

	glDrawElements( gl_triangles, glint(nvert), gl_unsigned_short, system.null_address );

	glDisableVertexAttribArray(0);
	glDisableVertexAttribArray(1);
	glDisableVertexAttribArray(2);

end ldraw;








end mrectobj;

