<?php
/**
 * A simple description for this script
 *
 * PHP Version 5.2.4 or Upper version
 *
 * @package    Hangul Parser
 * @author     Hidehito NOZAWA aka Suin <xoops.suinyeze.com>
 * @copyright  2008 Hidehito NOZAWA
 * @license    http://www.gnu.org/licenses/gpl-2.0.html GNU GPL v2.0
 *
 * @example
 *
	$hangul =& HangulParser::getInstance();
	$hangul->setTableClassName('HangulRealConvertTable');
	$word = $hangul->parseWord('abc한국어');
	foreach ( $word as $char )
	{
		if ( is_object($char) )
		{
			var_dump($char->parts);
		}
	}
 *
 */

class HangulParser
{
	const START = 44032;
	const END = 55203;
	const INITIAL = 588;
	const MIDDLE = 28;

	protected $hangulClassName = 'HangulChar';
	protected $tableClassName = 'HangulConvertTable';

	protected function __construct()
	{
	}

	public static function &getInstance()
	{
		static $instance;

		if ( $instance == null )
		{
			$class = __CLASS__;
			$instance = new $class();
		}

		return $instance;
	}

	/*
	 * 2文字以上のハングルを分解する。
	 * 引数：(string) 2文字以上のハングル。
	 * 戻り値：(array) 分解した情報を含むオブジェクトの集合
	**/
	public function parseWord($word)
	{
		$chars = $this->split($word);

		$ret = array();

		foreach ( $chars as $char )
		{
			$ret[] = $this->parseChar($char);
		}

		return $ret;
	}

	/*
	 * ハングル1文字を分解する。
	 * 引数：(string) ハングル1文字
	 * 戻り値：(object/string) 分解した情報を含むオブジェクトまたは、ハングル以外の文字列
	**/
	public function parseChar($char)
	{
		$code = $this->encode($char);

		if ( self::START <= $code and $code <= self::END )
		{
			$parts = $this->parse($code);
			$hangul = new $this->hangulClassName($char, $parts);
			return $hangul;
		}

		return $char;
	}

	/*
	 * ハングルクラスの名前を変更する。
	 * 引数：(string) クラスの名前
	**/
	public function setHangulClassName($className)
	{
		$this->hangulClassName = $className;
	}

	/*
	 * テーブルクラスの名前を変更する。
	 * 引数：(string) クラスの名前
	**/
	public function setTableClassName($className)
	{
		$this->tableClassName = $className;
	}

	protected function detectInitial($code)
	{
		$initialTalbe  = $this->getInitialTable();
		$initialNumber = $this->getInitialNumber($code);
		return $initialTalbe[$initialNumber];
	}

	protected function detectMiddle($code)
	{
		$middleTable  = $this->getMiddleTable();
		$middleNumber = $this->getMiddleNumber($code);
		return $middleTable[$middleNumber];
	}

	protected function detectFinal($code)
	{
		$finalTalbe  = $this->getFinalTalbe();
		$finalNumber = $this->getFinalNumber($code);
		return $finalTalbe[$finalNumber];
	}

	protected function split($string)
	{
		$strlen = mb_strlen($string);

		while ( $strlen )
		{
			$array[] = mb_substr($string, 0, 1, 'UTF-8');
			$string = mb_substr($string, 1, $strlen, 'UTF-8');
			$strlen = mb_strlen($string);
		}

		return $array;
	}

	protected function encode($char)
	{
		$ret = $this->entitize($char);
		$ret = str_replace('&#', '', $ret);
		$ret = str_replace(';', '', $ret);
		$ret = intval($ret);
		return $ret;
	}

	protected function decode($code)
	{
		$ret = '&#'.$code.';';
		$ret = $this->unentitize($ret);
		return $ret;
	}

	protected function entitize($string)
	{
		return mb_convert_encoding($string, 'HTML-ENTITIES', 'UTF-8');
	}

	protected function unentitize($string)
	{
		return mb_convert_encoding($string, 'UTF-8', 'HTML-ENTITIES');
	}

	protected function getInitialNumber($code)
	{
		return floor(($code - self::START) / self::INITIAL);
	}

	protected function getInitialTable()
	{
		return call_user_func(array($this->tableClassName, 'getInitialTable'));
	}

	protected function getMiddleNumber($code)
	{
		$initialNumber = $this->getInitialNumber($code);
		$middleNumber  = $code - self::START;
		$middleNumber  = $middleNumber - (self::INITIAL * $initialNumber);
		$middleNumber  = $middleNumber / self::MIDDLE;
		$middleNumber  = floor($middleNumber);
		$middleNumber  = intval($middleNumber);
		return $middleNumber;
	}

	protected function getMiddleTable()
	{
		return call_user_func(array($this->tableClassName, 'getMiddleTable'));
	}

	protected function getFinalNumber($code)
	{
		$initialNumber = $this->getInitialNumber($code);
		$middleNumber  = $this->getMiddleNumber($code);
		$finalNumber   = $code - self::START - (self::INITIAL * $initialNumber) - (self::MIDDLE * $middleNumber);
		return $finalNumber;
	}

	protected function getFinalTalbe()
	{
		return call_user_func(array($this->tableClassName, 'getFinalTalbe'));
	}

	protected function parse($code)
	{
		$initialChar = $this->detectInitial($code);
		$middleChar  = $this->detectMiddle($code);
		$finalChars  = $this->detectFinal($code);

		$chars = array();
		$chars[] = $initialChar;
		$chars[] = $middleChar;

		foreach ( $finalChars as $finalChar )
		{
			if ( !empty($finalChar) )
			{
				$chars[] = $finalChar;
			}
		}

		return $chars;
	}
}

?>
