package net.sf.jhpdf.io

import org.junit.Test
import static org.junit.Assert.assertThat
import static org.hamcrest.CoreMatchers.is
import static net.sf.jhpdf.test.matcher.Matchers.equalToArray

import net.sf.jhpdf.io.HPdfReadStream.SeekMode

class MemStreamTest {
    @Test
    void readAfterWrite() {
        // setup
        def s = new HPdfMemStream(0)
        def buf1 = "test string".getBytes("ISO-8859-1")
        def buf2 = new byte[buf1.length]
        // work
        s.write(buf1, buf1.length)
        s.seek(0, SeekMode.SET)
        s.read(buf2, buf2.length)
        s.close()
        // verify
        assertThat(buf2, equalToArray(buf1))
    }

    @Test
    void partialWrite() {
        // setup
        def s = new HPdfMemStream(0)
        def buf1 = "test string".getBytes("ISO-8859-1")
        def buf2 = new byte[buf1.length]
        // work
        s.write(buf1, 4)
        s.write(buf1, 4, buf1.length)
        s.seek(0, SeekMode.SET)
        s.read(buf2, buf2.length)
        s.close()
        // verify
        assertThat(buf2, equalToArray(buf1))
    }

    @Test
    void overRead() {
        // setup
        def buf1 = "test string".getBytes("ISO-8859-1")
        def s = new HPdfMemStream(buf1.length)
        def buf2 = new byte[10]
        // work
        s.write(buf1, buf1.length)
        s.seek(8, SeekMode.SET)
        def len = s.read(buf2, buf2.length)
        s.close()
        // verify
        assertThat len, is(buf1.length - 8)
    }

    @Test
    void positionAfterRewrite() {
        // setup
        def buf1 = "test string".getBytes("ISO-8859-1")
        def s = new HPdfMemStream(0)
        // work
        s.write(buf1, buf1.length)
        s.seek(0, SeekMode.SET)
        s.rewrite("TEST".getBytes("ISO-8859-1"), 4)
        assertThat s.tell(), is(4L)
        s.close()
    }

    @Test
    void rewritedContent() {
        def buf1 = "test string".getBytes("ISO-8859-1")
        def s = new HPdfMemStream(0)
        def buf2 = new byte[buf1.length]
        s.write(buf1, buf1.length)
        s.seek(0, SeekMode.SET)
        s.rewrite("TEST".getBytes("ISO-8859-1"), 4)
        s.seek(0, SeekMode.SET)
        def len = s.read(buf2, buf2.length)
        s.close()
        // verify
        assertThat(buf2, equalToArray("TEST string".getBytes("ISO-8859-1")))
    }

    @Test
    void expandOnWrite() {
        // setup
        def buf1 = "test string".getBytes("ISO-8859-1")
        def s = new HPdfMemStream(5)
        // work
        s.write(buf1, buf1.length)
        s.seek(0, SeekMode.END)
        // verify
        assertThat s.tell(), is(buf1.length as long)
        s.close()
    }

    @Test
    void expandOnRewrite() {
        // setup
        def buf1 = "test_".getBytes("ISO-8859-1")
        def buf2 = " string".getBytes("ISO-8859-1")
        def s = new HPdfMemStream(buf1.length + 2)
        // work
        s.write(buf1, buf1.length)
        s.seek(4, SeekMode.SET)
        s.rewrite(buf2, buf2.length)
        s.seek(0, SeekMode.SET)
        def buf3 = new byte[buf1.length + buf2.length - 1]
        s.read(buf3, buf3.length)
        s.close()
        // verify
        assertThat new String(buf3, "ISO-8859-1"), is("test string")
    }

    @Test
    void readLast() {
        // setup
        def buf1 = "test string".getBytes("ISO-8859-1")
        def s = new HPdfMemStream(0)
        // work
        s.write(buf1, buf1.length)
        s.seek(-1, SeekMode.END)
        def buf2 = new byte[10]
        s.read(buf2, 1)
        s.close()
        // verify
        assertThat buf2[0], is('g'.charAt(0) as byte)
    }

    @Test
    void checkEof() {
        // setup
        def buf1 = "test string".getBytes("ISO-8859-1")
        def s = new HPdfMemStream(0)
        // work
        s.write(buf1, buf1.length)
        s.seek(0, SeekMode.SET)
        def buf2 = new byte[20]
        s.read(buf2, buf1.length)
        def len = s.read(buf2, 1)
        // verify
        assertThat len, is(-1)
        s.close()
    }

    @Test
    void writeAppend() {
        // setup
        def buf1 = "test ".getBytes("ISO-8859-1")
        def buf2 = "string".getBytes("ISO-8859-1")
        def s = new HPdfMemStream(0)
        // work
        s.write(buf1, buf1.length)
        s.write(buf2, buf2.length)
        def buf3 = new byte[100]
        s.seek(0, SeekMode.SET)
        def len = s.read(buf3, 0, buf3.length)
        s.close()
        // verify
        assertThat new String(buf3, 0, len, "ISO-8859-1"), is("test string")
    }
}
