/*
 * JHPdf Free PDF Library : HPdfTrueTypeFont.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject.font;

import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.HPdfUtil;
import net.sf.jhpdf.encoder.HPdfBasicEncoder;
import net.sf.jhpdf.encoder.HPdfEncoder.HPdfEncoderType;
import net.sf.jhpdf.encrypt.HPdfEncrypt;
import net.sf.jhpdf.graphics.HPdfTextWidth;
import net.sf.jhpdf.io.HPdfWriteStream;
import net.sf.jhpdf.pdfobject.HPdfArray;
import net.sf.jhpdf.pdfobject.HPdfDict;
import net.sf.jhpdf.pdfobject.HPdfFont;
import net.sf.jhpdf.pdfobject.HPdfStreamDict;
import net.sf.jhpdf.pdfobject.HPdfXref;
import net.sf.jhpdf.pdfobject.font.HPdfFontDef.HPdfFontDefType;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Class represents PDF TrueType Font.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public class HPdfTrueTypeFont extends HPdfFontImpl {

    private static final Logger logger = LoggerFactory.getLogger(HPdfTrueTypeFont.class);
    
    
    public HPdfTrueTypeFont(HPdfTrueTypeFontDef fontDef, HPdfBasicEncoder encoder, HPdfXref xref) {
        super(fontDef, encoder, xref);
        
        logger.trace("HPdfTrueTypeFont#ctor");
        
        /* check whether the fontdef object and the encoder object is valid. */
        if (encoder.getType() != HPdfEncoderType.SINGLE_BYTE) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_ENCODER_TYPE, 0);
        }
        
        if (fontDef.getFontDefType() != HPdfFontDefType.TRUETYPE) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_FONTDEF_TYPE, 0);
        }
        
        this.setFontType(HPdfFontType.TRUETYPE);
        this.setWritingMode(HPdfWritingMode.HORIZONTAL);
        
        /* singlebyte-font has a widths-array which is an array of 256 signed
         * short integer.
         * in the case of type1-font, widths-array for all letters is made in
         * constructer. but in the case of true-type-font, the array is
         * initialized at 0, and set when the corresponding character was used
         * for the first time.
         */
        this.setWidthsCount(256);
        
        this.setUsedCount(256);
        
        this.addName("Type", "Font");
        this.addName("BaseFont", fontDef.getBaseFont());
        this.addName("Subtype", "TrueType");
        
        this.addNumber("FirstChar", (int) encoder.getFirstChar());
        this.addNumber("LastChar", (int) encoder.getLastChar());
        if (fontDef.getMissingWidth() != 0) {
            this.addNumber("MissingWidth", fontDef.getMissingWidth());
        }
        
        xref.add(this);
    }

    private void createDescriptor() {
        logger.trace("HPdfTrueTypeFont#createDescriptor");
        
        HPdfTrueTypeFontDef fontDef = (HPdfTrueTypeFontDef) this.getFontDef();
        if (fontDef.getDescriptor() == null) {
            HPdfDict descriptor = new HPdfDict();
            this.getXref().add(descriptor);
            descriptor.addName("Type", "FontDescriptor");
            descriptor.addNumber("Ascent", fontDef.getAscent());
            descriptor.addNumber("Descent", fontDef.getDescent());
            descriptor.addNumber("CapHeight", fontDef.getCapHeight());
            descriptor.addNumber("Flags", fontDef.getFlags());
            
            descriptor.add("FontBBox", HPdfArray.boxToArray(fontDef.getFontBbox()));
            
            descriptor.addName("FontName", fontDef.getBaseFont());
            descriptor.addNumber("ItalicAngle", fontDef.getItalicAngle());
            descriptor.addNumber("StemV", fontDef.getStemv());
            descriptor.addNumber("XHeight", fontDef.getXHeight());
            
            if (fontDef.getCharset() != null) {
                descriptor.addName("CharSet", fontDef.getCharset());
            }
            
            if (fontDef.isEmbedding()) {
                HPdfStreamDict fontData = new HPdfStreamDict(this.getXref());
                
                fontDef.saveFontData(fontData.getStreamForWrite());
                
                descriptor.add("FontFile2", fontData);
                fontData.addNumber("Length1", fontDef.getLength1());
                fontData.addNumber("Length2", 0);
                fontData.addNumber("Length3", 0);
                
                fontData.setFilterSet(this.getFilterSet());
            }
            
            fontDef.setDescriptor(descriptor);
        }
        this.add("FontDescriptor", fontDef.getDescriptor());
    }
    
    private int charWidth(char code) {
        if (this.lookupUsed(code) == 0) {
            this.putUsed(code, (byte) 1);
            this.putWidth(code, this.getFontDef().getWidth(code));
        }
        return this.lookupWidth(code);
    }

    @Override
    protected HPdfTextWidth textWidthImpl(String text, int len) {
        logger.trace("HPdfTrueTypeFont#textWidth");
        
        HPdfTextWidth ret = new HPdfTextWidth(0, 0, 0, 0);
        
        char b = 0x0;
        if (this.getWidths() != null) {
            for (int i = 0; i < len; ++i) {
                b = text.charAt(i);
                ret.incNumChars();
                ret.addWidth(this.charWidth(b));
                
                if (HPdfUtil.isWhiteSpace(b)) {
                    ret.incNumSpace();
                    ret.incNumWords();
                }
            }
        } else {
            throw new HPdfException(HPdfErrorCode.HPDF_FONT_INVALID_WIDTHS_TABLE, 0);
        }
        
        /* 2006.08.19 add. */
        if (HPdfUtil.isWhiteSpace(b)) {
            ; /* do nothing. */
        } else {
            ret.incNumWords();
        }
        
        return ret;
    }


    @Override
    public int measureTextImpl(String text, int len, float width, float fontSize,
            float charSpace, float wordSpace, boolean wordWrap,
            float[] realWidth) {
        logger.trace("HPdfTrueTypeFont#measureText");
        
        float w = 0;
        int tmpLen = 0;
        
        for (int i = 0; i < len; ++i) {
            char b = text.charAt(i);
            
            if (HPdfUtil.isWhiteSpace(b)) {
                tmpLen = i + 1;
                
                if (realWidth != null) {
                    realWidth[0] = w;
                }
                
                w += wordSpace;
            } else if (!wordWrap) {
                tmpLen = i;
                
                if (realWidth != null) {
                    realWidth[0] = w;
                }
            }
            
            w += this.charWidth(b) * fontSize / 1000;
            
            /* 2006.08.04 break when it encountered  line feed */
            if (w > width || b == '\n') {
                return tmpLen;
            }
            
            if (i > 0) {
                w += charSpace;
            }
        }
        
        /* all of text can be put in the specified width */
        if (realWidth != null) {
            realWidth[0] = w;
        }
        
        return len;
    }

    @Override
    protected void beforeWrite(HPdfWriteStream stream, HPdfEncrypt e) {
        super.beforeWrite(stream, e);
        logger.trace("HPdfTrueTypeFont#beforeWrite");
        this.createDescriptor();
    }

    @Override
    protected void onWrite(HPdfWriteStream stream) {
        super.onWrite(stream);
        logger.trace("HPdfTrueTypeFont#onWrite");
        
        HPdfBasicEncoder encoder = (HPdfBasicEncoder) this.getEncoder();
        
        /* Widths entry */
        stream.writeEscapeName("Widths");
        stream.writeStr(" [\n");
        
        StringBuilder buf = new StringBuilder();
        for (int i = encoder.getFirstChar(); i <= encoder.getLastChar(); ++i) {
            buf.append(this.lookupWidth(i));
            buf.append(' ');
            
            if ((i + 1) % 16 == 0) {
                buf.append('\n');
                stream.writeStr(buf.toString());
                buf.setLength(0);
            }
        }
        buf.append("]\n");
        stream.writeStr(buf.toString());
        
        encoder.writeValue(stream);
    }
    
}
