/*
 * JHPdf Free PDF Library : HPdfMarkupAnnotation.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject.annotation;

import net.sf.jhpdf.encoder.HPdfEncoder;
import net.sf.jhpdf.graphics.HPdfCMYKColor;
import net.sf.jhpdf.graphics.HPdfRGBColor;
import net.sf.jhpdf.graphics.HPdfRect;
import net.sf.jhpdf.pdfobject.HPdfAnnotation;
import net.sf.jhpdf.pdfobject.HPdfArray;
import net.sf.jhpdf.pdfobject.HPdfDate;
import net.sf.jhpdf.pdfobject.HPdfDict;
import net.sf.jhpdf.pdfobject.HPdfString;
import net.sf.jhpdf.pdfobject.HPdfXref;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Abstract class represents PDF Annotations with markup.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public abstract class HPdfMarkupAnnotation extends HPdfAnnotation {

    private static final Logger logger = LoggerFactory.getLogger(HPdfMarkupAnnotation.class);
    
    public enum HPdfAnnotIntent {
        FREETEXTCALLOUT("FreeTextCallout"),
        FREETEXTTYPEWRITER("FreeTextTypeWriter"),
        LINEARROW("LineArrow"),
        LINEDIMENSION("LineDimension"),
        POLYGONCLOUD("PolygonCloud"),
        POLYLINEDIMENSION("PolyLineDimension"),
        POLYGONDIMENSION("PolygonDimension"),
        ;
        
        String name;
        HPdfAnnotIntent(String name) {
            this.name = name;
        }
        String getName() {
            return this.name;
        }
    };
    
    protected HPdfMarkupAnnotation(HPdfXref xref, HPdfRect rect, String text, HPdfEncoder encoder) {
        super(xref, rect);
        
        logger.trace("HPdfMarkupAnnotation#ctor");
        
        HPdfString s = new HPdfString(text, encoder);
        
        // MEMO: no Subtype definition.
        
        this.add("Contents", s);
    }
    
    public final void setTitle(String name) {
        logger.trace("HPdfMarkupAnnotation#setTitle");
        
        this.add("T", new HPdfString(name, null));
    }
    
    public final void setCreationDate(HPdfDate value) {
        logger.trace("HPdfMarkupAnnotation#setCreationDate");
        
        this.setInfoDateAttr(HPdfInfoType.CREATION_DATE, value);
    }
    
    public final void setTransparency(float value) {
        logger.trace("HPdfMarkupAnnotation#setTransparency");
        
        this.addReal("CA", value);
    }
    
    public final void setIntent(HPdfAnnotIntent intent) {
        logger.trace("HPdfMarkupAnnotation#setIntent");
        
        this.addName("IT", intent.getName());
    }
    
    public final void setPopup(HPdfAnnotation popup) {
        logger.trace("HPdfMarkupAnnotation#setPopup");
        
        this.add("Popup", popup);
    }
    
    /**
     * IC with RGB entry
     * @param color
     */
    public final void setInteriorRGBColor(HPdfRGBColor color) {
        logger.trace("HPdfMarkupAnnotation#setInteriorRGBColor");
        
        HPdfArray cArray = new HPdfArray();
        
        this.add("IC", cArray);
        cArray.addReal(color.getR());
        cArray.addReal(color.getG());
        cArray.addReal(color.getB());
    }
    
    /**
     * IC with CMYK entry
     * @param color
     */
    public final void setInteriorCMYKColor(HPdfCMYKColor color) {
        logger.trace("HPdfMarkupAnnotation#setInteriorCMYKColor");
        
        HPdfArray cArray = new HPdfArray();
        
        this.add("IC", cArray);
        cArray.addReal(color.getC());
        cArray.addReal(color.getM());
        cArray.addReal(color.getY());
        cArray.addReal(color.getK());
    }
    
    /**
     * IC with Gray entry
     * @param color
     */
    public final void setInteriorGrayColor(float color) {
        logger.trace("HPdfMarkupAnnotation#setInteriorGrayColor");
        
        HPdfArray cArray = new HPdfArray();
        
        this.add("IC", cArray);
        cArray.addReal(color);
    }
    
    /**
     * IC with No Color entry
     * @param color
     */
    public final void setInteriorTransparent() {
        logger.trace("HPdfMarkupAnnotation#setInteriorTransparent");
        
        HPdfArray cArray = new HPdfArray();
        
        this.add("IC", cArray);
    }

    /**
     * RD entry : this is the difference between Rect and the text annotation rectangle
     * @param rect
     */
    public final void setRectDiff(HPdfRect rect) {
        logger.trace("HPdfMarkupAnnotation#setRectDiff");
        
        HPdfArray array = new HPdfArray();
        
        this.add("RD", array);
        
        HPdfRect rect_ = rect.clone(); // MEMO: original code 'rect' is copy of caller's.
        rect_.reviseVertically();
        
        array.addReal(rect_.getLeft());
        array.addReal(rect_.getBottom());
        array.addReal(rect_.getRight());
        array.addReal(rect_.getTop());
    }
    
    /**
     * BE entry
     * @param cloudIntensity
     */
    public final void setCloudEffect(int cloudIntensity) {
        logger.trace("HPdfMarkupAnnotation#setCloudEffect");
        
        HPdfDict borderEffect = new HPdfDict();
        
        this.add("BE", borderEffect);
        borderEffect.addName("S", "C");
        borderEffect.addNumber("I", cloudIntensity);
    }
}
