/*
 * JHPdf Free PDF Library : HPdfStreamDict.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jhpdf.HPdfConst;
import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.encrypt.HPdfEncrypt;
import net.sf.jhpdf.io.HPdfReadStream;
import net.sf.jhpdf.io.HPdfStream.HPdfFilterFlag;
import net.sf.jhpdf.io.HPdfMemStream;
import net.sf.jhpdf.io.HPdfWriteStream;

/**
 * Class represents PDF Dict having stream.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public class HPdfStreamDict extends HPdfFilterableDict {
    
    private static final Logger logger = LoggerFactory.getLogger(HPdfStreamDict.class);
    
    private HPdfMemStream innerStream =
            new HPdfMemStream(HPdfConst.HPDF_STREAM_BUF_SIZ);
    
    /**
     * ctor.
     */
    public HPdfStreamDict(HPdfXref xref) {
        super();
        logger.trace("HPdfStreamDict#ctor.");
        
        /* only stream object is added to xref automatically */
        xref.add(this);
        HPdfNumber length = new HPdfNumber(0);
        xref.add(length);
        this.add("Length", length);
    }
    
    public HPdfReadStream getStreamForRead() {
        return this.innerStream;
    }
    
    public HPdfWriteStream getStreamForWrite() {
        return this.innerStream;
    }
    
    @Override
    protected void beforeWrite(HPdfWriteStream stream, HPdfEncrypt e) {
        super.beforeWrite(stream, e);
        if (this.getFilterSet().size() == 0) {
            try {
                removeElement("Filter");
            } catch (HPdfException ex) {
                // NOP
            }
        } else {
            HPdfArray array = (HPdfArray) getDictBody().get("Filter");
            if (array == null) {
                array = new HPdfArray();
                add("Filter", array);
            }
            array.clear();
            // OK, Java runtime has it's own FlateDecoder...
            if (this.getFilterSet().contains(HPdfFilterFlag.FLATE_DECODE)) {
                array.addName("FlateDecode");
            }
            if (this.getFilterSet().contains(HPdfFilterFlag.DCT_DECODE)) {
                array.addName("DCTDecode");
            }
            if (this.getFilterSet().contains(HPdfFilterFlag.CCITT_DECODE)) {
                array.addName("CCITTFaxDecode");
            }
            if (this.getFilterParams() != null) {
                addFilterParams(this.getFilterParams());
            }
        }
    }
    
    private void addFilterParams(HPdfDict filterParam) {
        /* prepare params object */
        HPdfArray paramArray = (HPdfArray) getDictBody().get("DecodeParms");
        if (paramArray == null) {
            paramArray = new HPdfArray();
            /* add parameters */
            this.add("DecodeParms", paramArray);
        }
        paramArray.add(filterParam);
    }
    
    @Override
    protected void afterWrite(HPdfWriteStream stream, HPdfEncrypt e) {
        /* get "length" element */
        HPdfNumber length = (HPdfNumber) getDictBody().get("Length");
        if (length == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_DICT_STREAM_LENGTH_NOT_FOUND, 0);
        }
        
        /* "length" element must be indirect-object */
        if (!length.isTypeIndirect()) {
            throw new HPdfException(HPdfErrorCode.HPDF_DICT_ITEM_UNEXPECTED_TYPE, 0);
        }
        
        stream.writeStr("\012stream\015\012"); /* Acrobat 8.15 requires both \r and \n here */
        
        long strptr = stream.getSize();
        
        if (e != null) {
            e.reset();
        }
        
        this.getStreamForRead().writeToStream(stream, this.getFilterSet(), e);
        
        super.afterWrite(stream, e);
    }
}
