/*
 * JHPdf Free PDF Library : HPdfDate.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.HPdfException;

/**
 * Class represents date in PDF.<br>
 * note: This class basically represents Gregorian Calender's date and time.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public class HPdfDate {
    
    // Original logic in HPDF_Info_SetInfoDateAttr is almost implemented in this class.
    
    private int year; // gregorian year(but not sensitive for Julian Day)
    private int month; // 1-12
    private int day;
    private int hour;
    private int minutes;
    private int seconds;
    private char ind;
    private int offHour;
    private int offMinutes;
    
    public HPdfDate() {
        this(new java.util.Date());
    }
    
    public HPdfDate(Date d) {
        Calendar c = new GregorianCalendar();
        c.setTimeInMillis(d.getTime());
        commonInit(c);
    }
    
    public HPdfDate(Calendar c) {
        commonInit(c);
    }
    
    private final void commonInit(Calendar c) {
        this.year = c.get(Calendar.YEAR);
        this.month = c.get(Calendar.MONTH) - Calendar.JANUARY + 1;
        this.day = c.get(Calendar.DATE);
        this.hour = c.get(Calendar.HOUR_OF_DAY);
        this.minutes = c.get(Calendar.MINUTE);
        this.seconds = c.get(Calendar.SECOND);
        
        int offset = c.getTimeZone().getOffset(c.getTimeInMillis());
        
        if (offset == 0) {
            // equals to UTC
            this.ind = 'Z';
            this.offHour = 0;
            this.offMinutes = 0;
        } else {
            if (offset < 0) {
                this.ind = '-';
                offset = -offset;
            } else {
                this.ind = '+';
            }
            this.offHour = offset / 3600000;
            this.offMinutes = ((offset / 60000) % 60);
        }
        
    }
    
    /**
     * get year value.
     * @return year value. <strong>A.D.1 = 1.</strong>.
     */
    public int getYear(){
        return this.year;
    }
    
    /**
     * set year value.
     * @param year value to be set. <strong>A.D.1 = 1.</strong>.
     */
    public void setYear(int year){
        this.year = year;
    }
    
    /**
     * get month value.
     * @return month value. <strong>between 1 to 12</strong>.
     */
    public int getMonth(){
        return this.month;
    }
    
    /**
     * set month value.
     * @param month value to be set. <strong>between 1 to 12</strong>.
     */
    public void setMonth(int month){
        this.month = month;
    }
    
    /**
     * get day value.
     * @return day value. between 1 to 31.
     */
    public int getDay(){
        return this.day;
    }
    
    /**
     * set day value. <br>no year-month specific check.
     * @param day value to be set. between 1 to 31.
     */
    public void setDay(int day){
        this.day = day;
    }
    
    /**
     * get hour value.
     * @return hour value. between 0 to 23.
     */
    public int getHour(){
        return this.hour;
    }

    /**
     * set hour value.
     * @param hour value to be set. between 0 to 23.
     */
    public void setHour(int hour){
        this.hour = hour;
    }
    
    /**
     * get minutes value.
     * @return minutes value. between 0 to 59.
     */
    public int getMinutes(){
        return this.minutes;
    }
    
    /**
     * set minutes value.
     * @param minutes value to be set. between 0 to 59.
     */
    public void setMinutes(int minutes){
        this.minutes = minutes;
    }
    
    /**
     * get seconds value.
     * @return seconds value. between 0 to 59.
     */
    public int getSeconds(){
        return this.seconds;
    }
    
    /**
     * set seconds value.
     * @param seconds value to be set. between 0 to 59.
     */
    public void setSeconds(int seconds){
        this.seconds = seconds;
    }
    
    /**
     * get indicator(the relationship of local time to Universal Time (UT)).
     * @return indicator character(+/-/Z/' ')<br>
     * if return character == ' ', no UT relationship is specified.
     */
    public char getInd(){
        return this.ind;
    }
    
    /**
     * set indicator.
     * @param ind indicator character(+/-/Z)
     */
    public void setInd(char ind){
        this.ind = ind;
    }
    
    /**
     * get offset hour(s) of local time to Universal Time.
     * @return offset hour(s) of local time to Universal Time
     */
    public int getOffhour(){
        return this.offHour;
    }
    
    /**
     * set offset hour(s) of local time to Universal Time.
     * @param offHour offset hour(s) value.
     */
    public void setOffhour(int offHour){
        this.offHour = offHour;
    }
    
    /**
     * get offset minute(s) of local time to Universal Time.
     * @return offset minute(s) of local time to Universal Time
     */
    public int getOffminutes(){
        return this.offMinutes;
    }
    
    /**
     * set offset minute(s) of local time to Universal Time.<br>
     * @param offHour offset minute(s) value.
     */
    public void setOffminutes(int offMinutes){
        this.offMinutes = offMinutes;
    }
    
    /**
     * validates all fields value.<br>
     * setters of this class make no validation,
     * so before output an instance, you should check it with this method.
     * @throws HPdfException (HPDF_INVALID_DATE_TIME) whatever wrong in validation.
     */
    public void validate() {
        if (this.month < 1 || 12 < this.month ||
            this.day < 1 ||
            this.hour < 0 || 23 < this.hour ||
            this.minutes < 0 || 59 < this.minutes ||
            this.seconds < 0 || 59 < this.seconds ||
            "+-Z ".indexOf(this.ind) < 0 ||
            this.offHour < 0 || 23 < this.offHour ||
            this.offMinutes < 0 || 59 < this.offMinutes) {
                throw new HPdfException(HPdfErrorCode.HPDF_INVALID_DATE_TIME, 0);
        }
        switch (this.month) {
        case 1: // fall through
        case 3: // fall through
        case 5: // fall through
        case 7: // fall through
        case 8: // fall through
        case 10: // fall through
        case 12:
            if (this.day > 31) {
            }
            break;
        case 4: // fall through
        case 6: // fall through
        case 9: // fall through
        case 11:
            if (this.day > 30) {
                throw new HPdfException(HPdfErrorCode.HPDF_INVALID_DATE_TIME, 0);
            }
            break;
        case 2:
            if (this.day > 29 || (this.day == 29 &&
                (this.year % 4 !=0 ||
                (this.year % 100 == 0 && this.year % 400 != 0)))) {
                throw new HPdfException(HPdfErrorCode.HPDF_INVALID_DATE_TIME, 0);
            }
            break;
        default:  // invalid month number
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_DATE_TIME, 0);
        }
        // TODO: time offset to be checked detailed?
    }
    
    /**
     * make String representation.<br>
     * about format, see PDF Specification's section 'Common Data Structures' - 'Dates'
     * @return String representation of this instance.
     */
    public String toString() {
        String s = String.format("D:%04d%02d%02d%02d%02d%02d",
            this.year, this.month, this.day,
            this.hour, this.minutes, this.seconds);
        if (this.ind != ' ') {
            s += String.format("%c%02d'%02d'",
                this.ind, this.offHour, this.offMinutes);
        }
        return s;
    }
}
