#pragma once
#include <iostream>
#include <vector>

namespace chocolat{namespace filter{

/** Interface of stream filter function object */
class IStreamFilter
{
public:
	/** this is the body of filter. */
	virtual void operator()(std::istream& is, std::ostream& os)=0;
	/** your filter class needs 'clone' method to be stored in pipe. */
	virtual IStreamFilter* clone()const =0;
	/** this is defined for sub-class. */
	virtual ~IStreamFilter(){}
};

/** It is used in expression, so it's cleared after evaluation. */
class PipeExpression
{
public:
	/** connect input stream to pipe */
	friend PipeExpression operator>>(std::istream& is, PipeExpression pipe);
	/** clear the pointer of input. */
	PipeExpression();
	/** copy constructor uses virtual method 'transTo' by default behaves move.*/
	/** this method, using const cast.*/
	PipeExpression(const PipeExpression& pipe);
	/** this is used to move own filters. */
	virtual void transTo(PipeExpression& pipe);
	/** delete all own filter object. */
	virtual ~PipeExpression();
	/** push the clone of the pointed filter. */
	void push(const IStreamFilter* p_f);
	/** output to stream */
	std::ostream& operator>>(std::ostream& os);
	/** connect to pipe, returns new pipe object. */
	PipeExpression operator>>(const PipeExpression& ref)const;
	/** connect to filter, returns new pipe object.*/
	PipeExpression operator>>(const IStreamFilter& f)const;
	
	/** connected input stream */
	std::istream* is;
	/** connected filter list */
	std::vector<IStreamFilter*> list;
};

PipeExpression
operator>>(std::istream& is, PipeExpression pipe);

/** connect input stream to filter, returns pipe */
PipeExpression
operator>>(std::istream& is, const IStreamFilter& f);

/** This class is used to save the pipe.
 *  If you need temporary, use PipeExpression.
 */
class Pipe : public PipeExpression
{
public:
	/** filter pipe copies filter to other */
	virtual void transTo(PipeExpression& pipe);
	/** in assignment, it copies the filter objects. */
	Pipe& operator=(const PipeExpression& pipe);
private:
	/** copy constructor is prohibited. */
	Pipe(const PipeExpression& copy);
};
		
}}
