#include "number_format.hpp"
#include "string_aligner.hpp"

const HexadecimalFormat HexadecimalFormat::invalid()
{
	HexadecimalFormat h;
	h.m_is_valid = false;
	return h;
}
/** construct from value.*/
HexadecimalFormat::HexadecimalFormat(unsigned long value)
		:m_value(value),
		 m_width(4),
		 m_filling('0'),
		 m_upper(true),
		 m_is_valid(true)
{
}
bool HexadecimalFormat::isValid()const
{
	return m_is_valid;
}
/** construct from string.*/
HexadecimalFormat::HexadecimalFormat(std::string line)
		:m_width(4),
		 m_filling('0'),
		 m_upper(true),
		 m_is_valid(true)
{
	unsigned long value=0;
	for (unsigned i=0; i<line.length(); ++i){
		char c = line[i];
		if (isdigit(c)){
			value<<=4;
			value += (c - '0');
		}else if (('a'<= c && c<='f')){
			value<<=4;
			value += (c - 'a') + 10;
		}else if ('A'<= c && c<='F'){
			value<<=4;
			value += (c - 'A') + 10;
		} else if (c == 'x' || c=='X' || c=='$' || c == '#') {
			continue;
		} else {
			break;
		}
	}
	m_value = value;
}
bool HexadecimalFormat::isAccept(char c)const
{
	if (isdigit(c)){
		return true;
	}else if (('a'<= c && c<='f')){
		return true;
	}else if ('A'<= c && c<='F'){
		return true;
	} else if (c == 'x' || c=='X' || c=='$' || c == '#') {
		return true;
	} else {
		return false;
	}
}
/** copy.*/
HexadecimalFormat::HexadecimalFormat(const HexadecimalFormat& that)
		:m_value(that.m_value),
		 m_width(that.m_width),
		 m_filling(that.m_filling),
		 m_upper(that.m_upper),
		 m_is_valid(that.m_is_valid)
{
}
/** downcast in context of unsigned value.*/
HexadecimalFormat::operator unsigned long()const
{
	return m_value;
}
/** assign.*/
HexadecimalFormat& HexadecimalFormat::operator=(const HexadecimalFormat& rh)
{
	m_value=rh.m_value;
	m_width=rh.m_width;
	m_filling=rh.m_filling;
	return *this;
}
/** assign by value.*/
HexadecimalFormat& HexadecimalFormat::operator=(unsigned long value)
{
	m_value = value;
	return *this;
}
/** add the value, ignore format.*/
HexadecimalFormat HexadecimalFormat::operator+(const HexadecimalFormat& rh)const
{
	return HexadecimalFormat(m_value+rh.m_value);
}
/** subtract, ignore format.*/
HexadecimalFormat HexadecimalFormat::operator-(const HexadecimalFormat& rh)const
{
	return HexadecimalFormat(m_value-rh.m_value);
}
/** add value.*/
HexadecimalFormat& HexadecimalFormat::operator+=(unsigned long value)
{
	m_value += value;
	return *this;
}
/** subtract value.*/
HexadecimalFormat& HexadecimalFormat::operator-=(unsigned long value)
{
	m_value -= value;
	return *this;
}
/** set format width.*/
HexadecimalFormat& HexadecimalFormat::width(unsigned w)
{
	m_width = w;
	return *this;
}
/** set format width.*/
HexadecimalFormat HexadecimalFormat::width(unsigned w)const
{
	HexadecimalFormat result(*this);
	result.m_width = w;
	return result;
}
HexadecimalFormat& HexadecimalFormat::upper(bool is_upper)
{
	m_upper = is_upper;
	return *this;
}
HexadecimalFormat HexadecimalFormat::low()const
{
	HexadecimalFormat v= m_value%256;
	return v.width(2);
}
HexadecimalFormat HexadecimalFormat::high()const
{
	HexadecimalFormat v= (m_value/256)%256;
	return v.width(2);
}
std::string HexadecimalFormat::fillings(unsigned n)const
{
	std::string result="";
	for (unsigned i=0; i<n;++i){
		result+=m_filling;
	}
	return result;
}
std::string HexadecimalFormat::to_s()const
{
	std::string content="";
	for (unsigned long v=m_value; v>0; v/=16){
		char s;
		s = v%16;
		if (s < 10){
			s+='0';
			content = s + content;
		} else {
			s+=(m_upper ? 'A':'a') - 10;
			content = s + content;
		}
	}
	if (m_value == 0){
		content = "0";
	}
	StringAligner right(m_width, StringAligner::RIGHT, m_filling);
	return right(content);
	// if (content.length() < m_width){
	// 	return fillings(m_width - content.length()) + content;
	// } else if (content.length() > m_width){
	// 	return content.substr(content.length() - m_width, m_width);
	// } else {
	// 	return content;
	// }
}
/** display.*/
std::ostream& operator<<(std::ostream& os,
						 const HexadecimalFormat& rh)
{
	os << rh.to_s();
	return os;
}
/** read*/
std::istream& operator>>(std::istream& is,
						 HexadecimalFormat& rh)
{
	std::string word="";
	while (is && rh.isAccept(is.peek())){
		word += is.get();
	}
	rh = HexadecimalFormat(word);
	return is;
}


BinaryFormat::BinaryFormat()
{
}
BinaryFormat::BinaryFormat(unsigned value)
		:m_value(value),
		 m_width(8),
		 m_off('0'),
		 m_on('1'),
		 m_length(4),
		 m_delimiter(' ')
{
}
BinaryFormat::BinaryFormat(const BinaryFormat& that)
		:m_value(that.m_value),
		 m_width(that.m_width),
		 m_off(that.m_off),
		 m_on(that.m_on),
		 m_length(that.m_length),
		 m_delimiter(that.m_delimiter)
{
}
bool BinaryFormat::operator [](unsigned index)const
{
	unsigned value = 1<<index;
	return m_value & value;
}
BinaryFormat::operator unsigned()
{
	return m_value;
}
BinaryFormat BinaryFormat::width(unsigned w)const
{
	BinaryFormat that(*this);
	that.m_width=w;
	return that;
}
BinaryFormat BinaryFormat::length(unsigned len)const
{
	BinaryFormat that(*this);
	that.m_length=len;
	return that;
}
void BinaryFormat::display(std::ostream& os)const
{
	std::string result;
	unsigned v=m_value;
	for (unsigned i=0; i<m_width; ++i){
		if (i%m_length==0 && i!=0){
			result = m_delimiter+result;
		}
		result = (v%2 == 0 ? m_off:m_on) + result;
		v>>=1;
	}
	os << result;
}
std::ostream& operator<<(std::ostream& os, const BinaryFormat& rh)
{
	rh.display(os);
	return os;
}

DecimalFormat::DecimalFormat(int value)
		:m_value(value),
		 m_width(0),
		 m_filling(' ')
{
}
/** construct from string.*/
DecimalFormat::DecimalFormat(std::string line)
		:m_width(0),
		 m_filling(' ')
{
	int value=0;
	int sign = 1;
	for (unsigned i=0; i<line.length(); ++i){
		char c = line[i];
		if (isdigit(c)){
			value *= 10;
			value += (c - '0');
		} else if (i == 0 && c == '-') {
			sign = -1;
			continue;
		} else if (i == 0 && c == '0') {
			m_filling = '0';
			continue;
		} else {
			break;
		}
	}
	m_value = value * sign;
}
DecimalFormat::DecimalFormat(const DecimalFormat& that)
		:m_value(that.m_value),
		 m_width(that.m_width),
		 m_filling(that.m_filling)
{
}

DecimalFormat::operator int()
{
	return m_value;
}
	
DecimalFormat DecimalFormat::width(unsigned w)const
{
	DecimalFormat result(*this);
	result.m_width = w;
	return result;
}
std::string DecimalFormat::to_s()const
{
	std::string content="";
	unsigned v=m_value;
	if (m_value == 0){
		content = "0";
	} else {
		if (m_value < 0) {
			content+="-";
			v = -m_value;
		}
		for (; v>0; v/=10){
			char s;
			s = v%10 + '0';
			content = s + content;
		}
	}
	StringAligner right(m_width, StringAligner::RIGHT, m_filling);
	return right(content);
}
void DecimalFormat::display(std::ostream& os)const
{
	os << to_s();
}
std::ostream&
operator<<(std::ostream& os, const DecimalFormat& rh)
{
	rh.display(os);
	return os;
}
