#pragma once

#include "basic_type.hpp"
#include "nes_memory_map.hpp"
#include "instruction.hpp"
#include "nes_memory_mapper.hpp"
#include "utils/hex_form.hpp"

/**  emulate. */
class StateMachine
{
	enum Constant {
		HIGHEST_BIT = 7,
	};
	enum ProcessorStateType {
		PS_C=0, PS_Z, PS_I, PS_D,
		PS_B, PS_U, PS_V, PS_N,
		PS_NUM,
	};
	enum RegisterType {
		R_A=0, R_X, R_Y, R_SP, R_PS,
		R_COND, 
		REG_NUM,
	};
	enum IoRegisterType {
		IO_PPU=0, IO_VRAM, IO_CONST,
		IO_APU, IO_PAD,
		IO_NUM,
	};
	
	Word m_pc; 
	Byte m_reg[REG_NUM];

	NesMemoryMapper m_mapper;
	Word m_return_addr; // to test whether return is corresponding.
	bool m_cond_flag;

	const StateMachine* mp_parent;
	std::vector<Word> m_call_stack;
	
	Byte m_temp;
public:
	StateMachine(Word pc, const NesMemoryMapper& mapper)
			:m_pc(pc),
			 m_mapper(mapper),
			 m_return_addr(pc),
			 m_cond_flag(false),
			 mp_parent(this)
			 
	{
		for (unsigned i=0; i<REG_NUM; ++i) {
			m_reg[i] = 0;
		}
	}
	/// <<copy>>
	StateMachine(const StateMachine& that)
			:m_pc(that.m_pc),
			 m_mapper(that.m_mapper),
			 m_return_addr(that.m_return_addr),
			 m_cond_flag(that.m_cond_flag),
			 mp_parent(&that),
			 m_call_stack(that.m_call_stack)
	{
		for (unsigned i=0; i<REG_NUM; ++i) {
			m_reg[i] = that.m_reg[i];
		}
	}
	void copy(const StateMachine& that)
	{
		m_pc=that.m_pc;
		
		m_mapper.copy(that.m_mapper);
		m_return_addr = that.m_return_addr;
		m_cond_flag = that.m_cond_flag;

		unsigned size = that.m_call_stack.size();
		m_call_stack.resize(size);
		
		for (unsigned i=0; i<size; ++i) {
			m_call_stack[i]=that.m_call_stack[i];
		}
		
		for (unsigned i=0; i<REG_NUM; ++i) {
			m_reg[i] = that.m_reg[i];
		}
	}
	const StateMachine& parent()const { return *mp_parent; }
	
	unsigned depth()const 
	{
		if (mp_parent == this) {
			return 0;
		} else {
			return mp_parent->depth()+1;
		}
	}
	
	void execute(PInstruction instr);
	void restoreMemory()
	{
		if (mp_parent == this) {
			return;
		}
		for (unsigned i=0; i<REG_NUM; ++i) {
			m_reg[i] = mp_parent->m_reg[i];
		}
		m_mapper.copy(mp_parent->m_mapper);
		
	}
	
	const NesMemoryMapper& mapper()const {return m_mapper; }
	Byte a()const { return m_reg[R_A]; }
	Byte x()const { return m_reg[R_X]; }
	Byte y()const { return m_reg[R_Y]; }
	Word stackAddress()const
	{
		return bit8::addressIndex(nes::STACK_OFFSET, m_reg[R_SP]); 
	}
	Word returnAddress()const { return m_return_addr; }
	
	Byte sp()const { return m_reg[R_SP]; }
	void setSP(Byte val) { m_reg[R_SP] = val; }
	Word pc()const { return m_pc; }
	void setPC(Word addr)
    {
        m_pc =addr;
    }
	bool ps(ProcessorStateType bit) {return bit8::byteBit(m_reg[R_PS],bit);}
	void setPS(ProcessorStateType bit, bool val)
	{
		return bit8::setByteBit(m_reg[R_PS],bit, val);
	}
	PInstruction nextInstruction()
	{
        
		MemoryID id=m_mapper.id(m_pc);
        return m_mapper.getInstruction(id);
        /* index behaviour is not good, when reading instruction.
		Byte byte=m_mapper.data(id);
		Byte low=m_mapper.indexedData(id, 1);
		Byte high=m_mapper.indexedData(id, 2);
		PInstruction ptr(new Instruction(m_pc, byte, low, high));
		return ptr;        */
	}
	bool isCorrespondReturn()const { return m_return_addr+1 == m_pc; }
	
	void setupMatchCase(PInstruction instr)
	{
		if (m_cond_flag) {
			m_pc = instr->operand();
		} else {
			m_pc = instr->next();
		}
	}
	void setupOtherCase(PInstruction instr)
	{
		if (m_cond_flag) {
			m_pc = instr->next();
		} else {
			m_pc = instr->operand();
		}
	}
	void resetCall()
	{
		MemoryID id = m_mapper.id(nes::STACK_OFFSET);
		m_return_addr=m_mapper.indexedData(id, m_reg[R_SP]+1);
		m_return_addr+=m_mapper.indexedData(id, m_reg[R_SP]+2)<<8;
	}
	void dumpCallStack()
	{
		std::cout << "stack trace: ";
		unsigned size=m_call_stack.size();
		for (unsigned i=0; i<size-1; i+=2){
			std::cout << "  [" << HexForm(m_call_stack[size-i-1])
					  << " > " << HexForm(m_call_stack[size-i-2])
					  << "],";
		}
		std::cout << std::endl;
	}
    MemoryID targetID(AddressingMode mode, Word operand);

private:
	Word rts();
	void call(Word dest_addr, Word ret_addr);
	// push instruction
	void push(Byte byte) 
	{
		MemoryID id = m_mapper.id(nes::STACK_OFFSET);
		m_mapper.indexedData(id, m_reg[R_SP]) = byte;
		m_reg[R_SP]--;
	}
	// pull instruction
	Byte pull() {
		m_reg[R_SP]++;
		MemoryID id = m_mapper.id(nes::STACK_OFFSET);
		return m_mapper.indexedData(id, m_reg[R_SP]);
	}
	void pushWord(Word word);
	Word pullWord();	
	// read the value of operand.
	Byte& target(AddressingMode mode, Word operand);
	
	void cond(Byte val)
	{
		setPS(PS_N, bit8::signBit(val));
		setPS(PS_Z, val == 0);
	}
	bool isOverflowAddtion(Byte a, Byte b, Byte result);
	bool isOverflowSubtraction(Byte a, Byte b, Byte result);
	void compare(Byte left, Byte right);
};
