#pragma once

#include "basic_type.hpp"
#include "ines_file.hpp"
#include "instruction.hpp"

#include <map>
#include <vector>
#include <boost/shared_ptr.hpp>

class NesMemoryMapper
{
	enum IoRegisterType {
		IO_PPU=0, IO_VRAM, IO_CONST,
		IO_APU, IO_PAD,
		IO_NUM,
	};
	const INesFile& m_cart; // cartridge
	const std::vector<Word>& m_origins;
	unsigned m_bank_number_map[nes::PAGE_NUM];
	
	std::vector<Byte> m_RAM;
	std::map<Word,Byte> m_SRAM; // sparse list.
	/// io registers.
	Byte m_io[IO_NUM];
	/// dummy byte address to temporary value.
	Byte m_dummy;
public:
	NesMemoryMapper(const INesFile& cartridge,
					const std::vector<Word>& origins)
			:m_cart(cartridge),
			 m_origins(origins),
			 m_RAM(nes::RAM_SIZE)
	{
		clearMemory();
		m_io[IO_PPU]=0xFF;
		m_io[IO_PAD]=0x01;
		initRomMap();
	}
	/** <<copy constructor>> */
	NesMemoryMapper(const NesMemoryMapper& that)
			:m_cart(that.m_cart),
			 m_origins(that.m_origins),
			 m_RAM(nes::RAM_SIZE)
	{
		copy(that);
	}
	void copy(const NesMemoryMapper& that)
	{
		std::copy(that.m_RAM.begin(), that.m_RAM.end(), m_RAM.begin());
		std::copy(that.m_io, that.m_io+IO_NUM,m_io);
		std::copy(that.m_bank_number_map,
				  that.m_bank_number_map + nes::PAGE_NUM,
				  m_bank_number_map);
		for (auto i=that.m_SRAM.begin(); i!=that.m_SRAM.end(); ++i) {
			m_SRAM[i->first]=i->second;
		}
	}
	// interrupt vectors.
	std::vector<Word> interrupts()const
	{
		std::vector<Word> v;
		v.push_back(nmi());
		v.push_back(reset());
		v.push_back(irq());
		return v;
	}
	Word nmi()const { return wordData(id(nes::NMI_ADDR)); }
	Word reset()const { return wordData(id(nes::RESET_ADDR)); }
	Word irq()const { return wordData(id(nes::IRQ_ADDR)); }

	// returns identical number in the cartridge.
	MemoryID size()const {
		return m_cart.prgBanks().size()*nes::BANK_SIZE+memory_id::MEMORY_ID_OFFSET;
	}
	// addr -> id
	MemoryID id(Word addr)const
	{
		assert(addr<=nes::CODE_END);
		if (addr<nes::CODE_OFFSET){
			return addr;
		} else {
            long bank_num = loadedBankNumberByAddress(addr);
            long bank_offset = bank_num*nes::BANK_SIZE
                + memory_id::MEMORY_ID_OFFSET;
            long offset_addr = addr%nes::BANK_SIZE;
			return bank_offset + offset_addr;
        }
	}
	/** <<map>> */
	Word hardAddress(MemoryID id)const
	{
		return origin(loadedBankNumberByID(id)) + id%nes::BANK_SIZE;
	}
	Byte data(MemoryID id)const { return dataImplement(id); }
	Byte& data(MemoryID id) { return const_cast<Byte&>(dataImplement(id)); }
	ByteSequence dataRange(MemoryID begin, MemoryID end)const
	{
		ByteSequence bytes;
		if (isRomID(begin) && isRomID(end)) {
			for (MemoryID i=begin; i<end; ++i) {
				bytes.push_back(data(i));
			}
		}
		return bytes;
	}
	MemoryID index(MemoryID id, Byte idx)
	{
		return bit8::addressIndexLong(id, idx);
	}
	Byte& indexedData(MemoryID id, Byte idx)
	{
		return data(index(id, idx));
	}
	Word wordData(MemoryID id)const
	{
		return bit8::lowHigh(data(id),data(id+1));
	}
    MemoryID preindirectID(MemoryID id, Byte idx)
    {
		Word w = wordData(index(id,idx));
		return this->id(w);
    }
	MemoryID postindirectID(MemoryID id, Byte idx)
	{
		Word w = wordData(id);
		MemoryID dest_id = this->id(w);
		return (index(dest_id,idx));
	}
	// ex. lda ($50, x)
	Byte& preindirectData(MemoryID id, Byte idx)
	{
		Word w = wordData(index(id,idx));
		MemoryID dest_id = this->id(w);
		return data(dest_id);
	}
	// ex. lda ($50), y
	Byte& postindirectData(MemoryID id, Byte idx)
	{
		Word w = wordData(id);
		MemoryID dest_id = this->id(w);
		return data(index(dest_id,idx));
	}
	// bank_num -> origin addr
	Word origin(unsigned bank_num)const
	{
		assert(bank_num < m_origins.size());
		return m_origins[bank_num];
	}
	// id -> instruction
	PInstruction getInstruction(MemoryID id)const
	{
		if (!isRomID(id)) {
            PInstruction ptr(new Instruction(
                                     Instruction::getInvalid()));
			return ptr;
		}
		unsigned bank_num = loadedBankNumberByID(id);
		unsigned idx = id % nes::BANK_SIZE;
        
        Byte byte=data(id);
        Byte low=id+1 < size()? data(id+1) : 0;
        Byte high=id+2 < size()? data(id+2) : 0;

        PInstruction ptr(new Instruction(idx+origin(bank_num),
                                         byte, low, high));

        ptr->setOperandID(this->id(ptr->operand()));

        // when crossing bank, fail.
        //PInstruction ptr(new Instruction(idx+origin(bank_num),
        //                                 bank(bank_num), idx));
        return ptr;
	}
	unsigned lastBankNumber()const { return m_cart.prgBanks().size()-1; }
	bool isRomID(MemoryID id)const
	{
		return (nes::CODE_OFFSET <= id && id < size());
	}
private:
	void clearMemory()
	{
		std::fill(m_RAM.begin(), m_RAM.end(), 0);
		std::fill(m_io, m_io + IO_NUM, 0);
	}
	void initRomMap()
	{
		m_bank_number_map[0] = 0;
		m_bank_number_map[1] = 1;
		m_bank_number_map[2] = lastBankNumber()-1;
		m_bank_number_map[3] = lastBankNumber();
	}
	// id -> bank_num
	unsigned loadedBankNumberByID(MemoryID id)const
	{
		if (!isRomID(id)) {
			return 0;
		}
		return (id - memory_id::MEMORY_ID_OFFSET)/nes::BANK_SIZE;
	}
	// addr -> map idx -> bank num.
	unsigned loadedBankNumberByAddress(Word addr)const
	{
		assert(addr>=nes::CODE_OFFSET);
		assert(addr<=nes::CODE_END);
        
		unsigned hard_bank_idx
            = NesMemoryMapHelper::pageIndexByAddress(addr);
		return m_bank_number_map[hard_bank_idx];
	}
	// bank num -> bank
	const ByteSequence& bank(unsigned n)const {
		assert(n < m_cart.prgBanks().size());
		return *(m_cart.prgBanks()[n]);
	}
	
	const Byte& dataImplement(MemoryID id)const
	{
		if (id < nes::RAM_END) {
			return m_RAM[id % nes::RAM_SIZE];
		} else if (id < nes::IO_REG_END) {
			switch (id%8) {
			case 2:
				return m_io[IO_PPU];
			case 3:
				return m_io[IO_VRAM];
			default:
				return m_dummy;
			}
		} else if (id < nes::IO_REG_END2) {
			if (id == 0x4015) {
				return m_io[IO_APU];
			} else if (id == 0x4016 || id == 0x4017) {
				return m_io[IO_PAD];
			} else {
				return m_dummy;
			}
		} else if (id < nes::EXPANSION_ROM_END) { // expansion ROM
			return m_dummy;
		} else if (id < nes::SRAM_END) {
			std::map<Word,Byte>& sram
                = const_cast<std::map<Word,Byte>&>(m_SRAM);
			if (sram.find(id) == sram.end()) {
				sram[id]=0;
			}
			return sram[id];
		} else if (id < size()) {
			unsigned bank_num = loadedBankNumberByID(id);
			unsigned idx = id%nes::BANK_SIZE;
			return bank(bank_num)[idx];
		} else {
			assert("id >= size()"==false);
			return m_dummy;
		}
	}
};
