#pragma once

#include <vector>

/*
  synopsis: 
  		INesHeader header;
		header.read(fin);
		header.display(std::cout);

  detail:
  From iNES document, the format is:
     SS SS SS SS PP CC MS M0  	 00 00 00 00 00 00 00 00
	 
	 Signature [0:3]:"NES "
	 PRG-ROM   [4]  : 00-FF (n * 16kb bank of PRG code)
	 CHR-ROM   [5]  : 00-FF (n *  8kb bank of CHR data)
	 SWitch    [6]  : 0-F value.(lower 4 bits). [4TBV]
		4 Screen VRAM
		Trainer
		Battery
		Vertical Mirroring
	 Mapper    [6:7]: 00-FF
	 
  In NESASM, the header infos are written by:
     .inesprg n
	 .ineschr n
	 .inesmap n
	 .inesmir n
 */
class INesHeader
{
public:
	enum ByteIndex {
		SIGNATURE_LENGTH = 4,
		PRG_BYTE = 4, CHR_BYTE = 5, MIR_BYTE = 6,
		HEADER_LENGTH = 16,
	};
	enum MirroringFlagMask {
		VERTICAL_MIR = 0x01, BATTERY = 0x02, TRAINER = 0x04, FOUR_SCR =0x08,
	};
private:
	std::vector<Byte> m_bytes;
public:
	INesHeader():m_bytes(HEADER_LENGTH){}
	
	/**<<query>>*/
	bool isValid()const 
	{
		const Byte ValidSignature[] = {'N','E','S', 0x1A };
		for (unsigned i=0; i<SIGNATURE_LENGTH; ++i) {
			if (ValidSignature[i] != m_bytes[i])
				return false;
		}
		return true;
	}
	unsigned prg()const { return m_bytes[PRG_BYTE]; }
	unsigned chr()const { return m_bytes[CHR_BYTE]; }
	unsigned map()const {
		Byte low4bit = m_bytes[MIR_BYTE] >> 4;
		Byte high4bit = m_bytes[MIR_BYTE+1] & 0xF0;
		return low4bit+high4bit;
	}
	unsigned mir()const { return m_bytes[MIR_BYTE] & 0x0F; }
	
	const std::vector<Byte>* bytes()const { return &m_bytes; }

	/** <<modifier>> */
	void read(std::istream& is)
	{
		char c;
		for (unsigned i=0; i < HEADER_LENGTH && is.get(c); ++i){
			m_bytes[i]=Byte(c);
		}
	}
};
