#include "front_end.hpp"

#include "instruction.hpp"
#include "state_machine.hpp"
#include "code_data_parser.hpp"

#include <deque>


/** parse PRG-ROM bank.
 * Concept: To run emulator and collect call.
 * Input:   NesMemoryMapper and ReferenceTable
 * Output:  ReferenceTable
 *
 * to collect calls in the bank.
 * to separate code and data.
 * by traversing from initial call.
 */
class CodeDataParserImpl
{
	// working memory for traversing
	/** to width first traverse */
	// result data by parse.
    
    const NesMemoryMapper& m_mapper;
	// parse start point; first in first out. 
	std::deque<Word> m_resume_queue;
    InstructionMap m_instructions;
    
	/** procedure flag.
     * if m_flags[id] = proc_id,
     * the memory(id) is belongs to the procedure.  */
	ByteSequence m_flags;
public:
	CodeDataParserImpl(const NesMemoryMapper& mapper)
            :m_mapper(mapper)
    {
        init();
    }
	/** <<policy>> parse memory space.*/
    boost::shared_ptr<ByteSequence >
    execute()
    {
        StateMachine state(m_resume_queue.front(), m_mapper);
        followControl(state);
        
        boost::shared_ptr<ByteSequence > result
            (new ByteSequence);
        
        result->resize(m_flags.size()-nes::CODE_OFFSET);
        for (unsigned i=0; i<result->size(); ++i) {
            (*result)[i]=m_flags[i+nes::CODE_OFFSET];
        }
        return result;
    }
private:
    void followControl(StateMachine state)
    {
        while (!m_resume_queue.empty()) {
            Word addr = m_resume_queue.front();
            state.setPC(addr);
            m_resume_queue.pop_front();
            call(state);
        }
    }
    void init()
    {
        m_flags.resize(m_mapper.size());
        std::fill(m_flags.begin(), m_flags.end(), 0);
        
        m_resume_queue.push_back(m_mapper.reset());
        m_resume_queue.push_back(m_mapper.nmi());
        m_resume_queue.push_back(m_mapper.irq());
        
    }
	void reserveAddress(Word addr)
    {
        if (NesMemoryMapHelper::isRomAddress(addr)) {
            m_resume_queue.push_back(addr);
        }
    }
    void searchJumpTable(StateMachine& state)
    {
        ReferenceJumpTable parser(m_flags);
        std::vector<Word> table = parser.parse(state);
        if (parser.found()) {
            for (unsigned i=0; i<table.size(); ++i) {
                reserveAddress(table[i]);
            }
        }
    }
    /** call and run.*/
	void call(StateMachine& state)
    {
        run(state);
    }
	/* proc_id is written to detect multi-entry procedures' junction.*/
	void run(StateMachine& state)
    {
        const NesMemoryMapper& mapper = state.mapper();
        Word last_jump_addr = state.pc(); // for jump tablE search.
        Word last_stack_ptr = state.sp();
	
        while (true) {
            Word addr = state.pc();
            if (addr < nes::CODE_OFFSET
                || nes::CODE_END <= addr) {
                break;
            }
            MemoryID id = mapper.id(addr);
            assert((unsigned)id < (unsigned)m_flags.size());
            if (m_flags[id] & cdp::CODE) {
                break;
            }
            PInstruction instr = state.nextInstruction();
            
            for (unsigned i=0;
                 i<instr->bytelength()
                     && id+i < m_flags.size(); ++i) {
                m_flags[id+i] |= cdp::CODE;
            }
            
            MemoryID op_id = mapper.id(instr->operand());
            AddressingMode mode = instr->mode();
            ReferenceType type = ReferenceHelper::map(
                    instr->type());

            m_instructions[id]=instr;
			
            if (type == reference::CALL) {
                StateMachine temp(state);
                state.execute(instr);
                run(state);
                state.copy(temp);
                state.setPC(instr->next());
                //break; // if goto next, the stack will too big.
            } else if (type == reference::JUMP) {
                state.execute(instr);
                
                if (state.pc() >= nes::CODE_OFFSET) {
                    if (mode == opcode::IND) {
                        m_flags[op_id] |= cdp::IND_CODE;
                        StateMachine searcher(state);
                        searcher.restoreMemory();
                        // i don't know why this works.
                        searcher.setPC(last_jump_addr);
                        searcher.setSP(last_stack_ptr);
                        searchJumpTable(searcher);
                    }
                }
                run(state);
                break;
            } else if (type == reference::BREAK) {
                break;
            } else if (type == reference::RETURN) {
                state.execute(instr);
                Word return_addr = state.pc() - 1;
                if (nes::CODE_OFFSET <= return_addr
                    && return_addr < nes::CODE_END) {
                    if (state.isCorrespondReturn()) {
                        state.resetCall();
                        run(state);
                    } else {
                        // stack may be used to indirect jump.
                    }
                }
                break;
            } else if (type == reference::BRANCH) {
                Word next_addr = instr->next();
                if (nes::CODE_OFFSET <= next_addr
                    && next_addr < nes::CODE_END) {
                    state.execute(instr);
                    StateMachine temp(state);
                    state.setupMatchCase(instr);
                    temp.setupOtherCase(instr);
                    run(state);
                    state.copy(temp);
                }
            } else if (type == reference::READ) {
                ReferenceType mode_type
                    = ReferenceHelper::mapWithAddressingMode(
                            instr->type(), instr->mode());
                MemoryID targetID=state.targetID(
                        instr->mode(),instr->operand());
                
                switch (mode_type){
                case reference::READ:
                case reference::IDX_READ:
                    m_flags[op_id] |= cdp::DATA;
                    m_flags[targetID] |= cdp::DATA;
                    break;
                case reference::PRE_IND_READ:
                case reference::POST_IND_READ:
                    m_flags[op_id] |= cdp::IND_DATA;
                    m_flags[targetID] |= cdp::DATA;
                    break;
                default:
                    break;
                }
                state.execute(instr);
            } else if (type == reference::WRITE) {
                ReferenceType mode_type
                    = ReferenceHelper::mapWithAddressingMode(
                            instr->type(), instr->mode());
                MemoryID targetID=state.targetID(
                        instr->mode(),instr->operand());
                
                switch (mode_type){
                case reference::WRITE:
                case reference::IDX_WRITE:
                    m_flags[op_id] |= cdp::DATA;
                    m_flags[targetID] |= cdp::DATA;
                    break;
                case reference::PRE_IND_WRITE:
                case reference::POST_IND_WRITE:
                    m_flags[op_id] |= cdp::IND_DATA;
                    m_flags[targetID] |= cdp::DATA;
                    break;
                default:
                    break;
                }
                state.execute(instr);
            } else {
                state.execute(instr);
            }
        }
    }
};

PByteSequence
CodeDataModule::parse(const NesMemoryMapper& mapper)
{
    CodeDataParserImpl impl(mapper);
    return impl.execute();
}
