// TTAReader.cs
// 2008/02/19

using System;
using System.IO;
using System.Runtime.InteropServices;

using CRC32 = QVorbis.CRC32;

namespace TTA {

// Header
[StructLayout(LayoutKind.Sequential, Pack = 2)]
public struct Header {

	[MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
	public Byte[] Signature;

	public UInt16 Format;
	public UInt16 Channels;
	public UInt16 BitsPerSample;
	public UInt32 SamplingRate;
	public UInt32 Samples;

	public UInt32 CRC32;

}; // Header

// FormatError
public class FormatError : Exception {

	public FormatError(String msg) : base(msg)
	{
	}

} // FormatError

// Reader
public class Reader {

	private const Double FRAME_TIME = 1.04489795918367346939;

	private Stream _s;

	private Header _header;

	private Int32 _samplesPerFrame;
	private Int32 _frames;

	private Int32[] _size;

	private Int32 _current;

	public Reader(Stream s)
	{
		_s = s;

		_header = new Header();

		_samplesPerFrame = 0;

		_size = null;

		_current = 0;
	}

	public UInt32 Samples {
		get { return _header.Samples; }
	}

	public Int32 SamplesPerFrame {
		get { return _samplesPerFrame; }
	}

	public UInt16 Channels {
		get { return _header.Channels; }
	}

	public UInt16 BitsPerSample {
		get { return _header.BitsPerSample; }
	}

	public UInt32 SamplingRate {
		get { return _header.SamplingRate; }
	}

	public Int32 Frames {
		get { return _frames; }
	}

	public Int32[] SizeArray {
		get { return _size; }
	}

	public void ReadHeader()
	{
		Byte[] buf = new Byte[Marshal.SizeOf(_header)];

		_s.Read(buf, 0, buf.Length);

		var gh = GCHandle.Alloc(buf, GCHandleType.Pinned);
		try {
			IntPtr p = gh.AddrOfPinnedObject();

			_header = (Header)Marshal.PtrToStructure(p, typeof(Header));

			Marshal.StructureToPtr(_header, p, false);
		} finally {
			gh.Free();
		}

		if (_header.Signature[0] != 'T' ||
			_header.Signature[1] != 'T' ||
			_header.Signature[2] != 'A' ||
			_header.Signature[3] != '1') {
			throw new FormatError("ReadHeader.Signature");
		}

		if (_header.Format != 1) {
			throw new FormatError("ReadHeader.PCMFormat");
		}

		if (_header.Channels != 1 &&
			_header.Channels != 2) {
			throw new FormatError("ReadHeader.Channels");
		}

		if (_header.BitsPerSample != 16) {
			throw new FormatError("ReadHeader.BitsPerSample");
		}

		{
			var crc = new CRC32();
			if (crc.Generate(buf, 0, buf.Length - 4) != _header.CRC32) {
				throw new FormatError("ReadHeader.Header.CRC");
			}
		}

		Double dSamplesPerFrame = FRAME_TIME * _header.SamplingRate;

		_samplesPerFrame = (Int32)dSamplesPerFrame;

		_frames = (Int32)(((Int64)_header.Samples + _samplesPerFrame - 1) / _samplesPerFrame);

		Byte[] sz = new Byte[_frames * 4 + 4];

		_s.Read(sz, 0, sz.Length);

		_size = new Int32[_frames];

		for (Int32 i = 0; i < _frames; i++) {
			_size[i] = BitConverter.ToInt32(sz, i * 4);
		}

		UInt32 fcrc = BitConverter.ToUInt32(sz, _frames * 4);
		{
			var crc = new CRC32();
			if (crc.Generate(sz, 0, sz.Length - 4) != fcrc) {
				throw new FormatError("ReadHeader.SeekPoint.CRC");
			}
		}

		_current = 0;
	}

	public Byte[] FetchFrameData()
	{
		if (_current >= _frames) {
			return null;
		}

		Byte[] frame = new Byte[_size[_current]];

		_s.Read(frame, 0, frame.Length);

		_current += 1;

		return frame;
	}

} // Reader

} // namespace TTA

