#!/usr/bin/env python

import sqlite3
import sys
import getopt
import os.path

class GACacheDB(object):
    "Google Analytics Cache Database"
    DB_FILE = "database/analytics_dat"

    def __init__(self):
        self.con = None
        self.cur = None

    def create_table(self):
        con = sqlite3.connect(self.DB_FILE)
        cur = con.cursor()

        cur.execute("""
          create table logdata (
            url text,
            date int,
            pageviews int,
            visits int, 
            unique (url, date));""")
        con.commit()
        cur.close()
        con.close()

    def __enter__(self):
        self.con = sqlite3.connect(self.DB_FILE)
        self.cur = self.con.cursor()

    def __exit__(self, exc_type, exc_value, traceback):
        if exc_type:
            self.con = None
            self.cur = None
            return False
        self.con.commit()
        self.cur.close()
        self.con.close()
        self.con = None
        self.cur = None
        return True


    def add(self, url, date, pageviews, visits):
        d = dict(url=url,
                 date=date,
                 pv=pageviews,
                 visits=visits)

        try:
            self.cur.execute("""insert into logdata ( url, date, pageviews, visits )
                           values ( :url, :date, :pv, :visits );""", d)
        except sqlite3.IntegrityError:
            self.cur.execute("""update logdata set pageviews = :pv, visits = :visits
                           where url = :url and date = :date;""", d)


def main():
    usage = "%s [-c] logfile" % sys.argv[0]
    (optlist, args) = getopt.getopt(sys.argv[1:], "c")

    db = GACacheDB()
    if ('-c', '') in optlist:
        # create database
        db.create_table()
        return
    try:
        fname = args[0]
    except IndexError:
        sys.exit(usage)

    f = open(fname, "r")
    with db:
        for l in f:
            term = l.strip().decode("utf-8").rsplit("\t", 4)
            db.add(url=term[0],
                   date=term[1],
                   pageviews=term[2],
                   visits=term[3])

if __name__ == '__main__':
    main()
