using System;
using System.Text;
using System.Text.RegularExpressions;
using System.Diagnostics;

namespace RomRecoard{
	class Recoard{
		protected uint m_address = 0;
		protected byte [] m_data; //data size は動的に設定する
		protected bool m_valid = false, m_error = true;
		
		//---- property ----
		public bool Valid{
			get{return m_valid;}
		}
		public bool Error{
			get{return m_error;}
		}
		public uint Address{
			get{return m_address;}
		}
		public byte [] Data{
			get{return m_data;}
		}
		public string TargetFilename{
			get; set;
		}
		public uint TargetOffset{
			get; set;
		}
	}
/*
Motorola exormacs format/ Motora S-record
S0 Start Record/ ファイル名
S1,S2,S3/ データ
S4/ Symbol(未使用でいいや)
S5/ データカウント
S6/ 未使用
S7/ S3の終了値
S8/ S2の終了値
S9/ S1の終了値

データ構造
http://www.amelek.gda.pl/avr/uisp/srecord.htm
The general format of an S-record follows:
+-------------------//------------------//-----------------------+
| type | count | address  |            data           | checksum |
+-------------------//------------------//-----------------------+

type -- A char[2] field. These characters describe the type of record 
(S0, S1, S2, S3, S5, S7, S8, or S9).

count -- A char[2] field. These characters when paired and interpreted 
as a hexadecimal value, display the count of remaining character pairs 
in the record.

address -- A char[4,6, or 8] field. These characters grouped and 
interpreted as a hexadecimal value, display the address at which the 
data field is to be loaded into memory. The length of the field 
depends on the number of bytes necessary to hold the address. A 2-byte 
address uses 4 characters, a 3-byte address uses 6 characters, and a 
4-byte address uses 8 characters.

data -- A char [0-64] field. These characters when paired and 
interpreted as hexadecimal values represent the memory loadable data 
or descriptive

checksum -- A char[2] field. These characters when paired and interpreted
as a hexadecimal value display the least significant byte of the ones
complement of the sum of the byte values represented by the pairs of
characters making up the count, the address, and the data fields.
*/
	class MotorolaSRecoard : Recoard{
		enum type :int{
			S0 = 0,
			S1_ADDRESS16BIT_START,
			S2_ADDRESS24BIT_START,
			S3_ADDRESS32BIT_START,
			S5 = 5,
			S7_ADDRESS32BIT_END = 7,
			S8_ADDRESS24BIT_END,
			S9_ADDRESS16BIT_END
		};
		
		//---- private function for constractor ----
		private bool address_set(type recoard_type, string r, ref uint address, ref int str_address_length)
		{
			string pattern_address = "";
			
			switch(recoard_type){
			case type.S0:
			case type.S5:
				break;
			case type.S1_ADDRESS16BIT_START:
			case type.S9_ADDRESS16BIT_END:
				str_address_length = 4;
				pattern_address = "^([0-9A-F]{4})";
				break;
			case type.S2_ADDRESS24BIT_START:
			case type.S8_ADDRESS24BIT_END:
				str_address_length = 6;
				pattern_address = "^([0-9A-F]{6})";
				break;
			case type.S3_ADDRESS32BIT_START:
			case type.S7_ADDRESS32BIT_END:
				str_address_length = 8;
				pattern_address = "^([0-9A-F]{8})";
				break;
			default:
				Debug.Assert(false);
				break;
			}
			if(str_address_length != 0){
				GroupCollection g;
				if(Static.Utility.matching(r, pattern_address, out g) == false){
					return false;
				}
				m_address = Convert.ToUInt32(g[1].Value, 16);
			}
			return true;
		}

		private uint uint_add(uint data, int shift){
			uint sum = 0;
			for(int i = 0; i < shift; i+=8){
				sum += (data >> i) & 0xff;
			}
			return sum;
		}
		private bool checksum_compute(byte datacount, uint address, byte [] data, byte checksum)
		{
			uint sum;
			sum = uint_add(address, 32);
			sum += datacount;
			foreach(uint t in data){
				sum += t;
			}
			sum += checksum;
			//carry の分を加算する 
/*			sum = uint_add(sum, 32); //32 -> 16
			sum &= 0xffff;
			sum = uint_add(sum, 16); //16 -> 8
			sum &= 0xff;*/
			return (sum & 0xff) != 0xff;
		}
		//---- constracter ----
		public MotorolaSRecoard(string r){
			//type 取得
			GroupCollection submatch;
			if(Static.Utility.matching(r, "^S([01235789])", out submatch) == false){
				return;
			}
			/*const*/ type recoard_type = (type) Convert.ToInt32(submatch[1].Value, 10);
			r = Static.Utility.string_forward(r, 2);
			
			//count 取得
			if(Static.Utility.matching(r, "^([0-9A-F]{2})", out submatch) == false){
				return;
			}
			/*const*/ int datacount = Convert.ToInt32(submatch[1].Value, 16);
			r = Static.Utility.string_forward(r, 2);
			
			//address 取得
			//address の文字数が異なる, ついでに valid も設定
			int str_address_length = 0;
			if(address_set(recoard_type, r, ref m_address, ref str_address_length) == false){
				return;
			}
			r = Static.Utility.string_forward(r, str_address_length);
			//data取得
			bool havedata;
			//data がない type は行わない
			switch(recoard_type){
			case type.S0:
			case type.S5:
			case type.S7_ADDRESS32BIT_END:
			case type.S8_ADDRESS24BIT_END:
			case type.S9_ADDRESS16BIT_END:
				m_valid = false;
				havedata = true;
				break;
			case type.S1_ADDRESS16BIT_START:
			case type.S2_ADDRESS24BIT_START:
			case type.S3_ADDRESS32BIT_START:
				m_valid = true;
				havedata = true;
				break;
			default:
				m_valid = false;
				havedata = false;
				return; //break;
			}
			if(havedata == true){
				//最後の1は checksum のbyte数
				int length = datacount - (str_address_length/2) - 1;
				string pt = "^";
				for(int i = 0; i < length; i++){
					//pt = string.Concat(pt, "([0-9A-F]{2})");
					pt += "([0-9A-F]{2})";
				}
				if(Static.Utility.matching(r, pt, out submatch) == false){
					return;
				}
				
				m_data = new byte[length];
				//index 0 は全ての match を含むので 1 から、
				//Count は index 0 + match数なので - 1
				for(int i = 0; i < submatch.Count - 1; i++){
					int j = i+1;
					m_data[i] = Convert.ToByte(submatch[j].Value, 16);
				}
				r = Static.Utility.string_forward(r, length * 2);//byte数から文字数に変換するために*2
			}
			//checksum取得
			if(Static.Utility.matching(r, "^([0-9A-F]{2})$", out submatch) == false){
				return;
			}
			byte checksum = Convert.ToByte(submatch[1].Value, 16);
			m_error = checksum_compute((byte) datacount, m_address, m_data, checksum);
		}
	}
}
