/***********************************************************
        io.c -- input/output
***********************************************************/
#include "ar.h"
#include <assert.h>

#define CRCPOLY  0xA001         /* ANSI CRC-16 */
                         /* CCITT: 0x8408 */

uint16_t crctable[UCHAR_MAX + 1];

void
make_crctable(void)
{
    unsigned int i, j, r;

    for (i = 0; i <= UCHAR_MAX; i++) {
        r = i;
        for (j = 0; j < CHAR_BIT; j++)
            if (r & 1)
                r = (r >> 1) ^ CRCPOLY;
            else
                r >>= 1;
        crctable[i] = r;
    }
}

static const char *
bitstring(uint16_t bitbuf, int n, char *ptr, size_t sz)
{
    static char str[256];
    size_t size = sz;

    if (ptr == NULL) {
      ptr = str;
      size = sizeof(str);
    }

    if (n+1 > size)
        return "bit size is too big";

    ptr[n] = '\0';

    while (n > 0) {
      ptr[--n] = (bitbuf & 1) ? '1' : '0';
      bitbuf >>= 1;
    }
    return ptr;
}

/*
  fill bitbuf for reading.

    o shift bitbuf n bits left.
    o read 8 bits from file and put it to subbitbuf.
    o get n bits from subbitbuf and fill into bitbuf.

         7 6 5 4 3 2 1 0 7 6 5 4 3 2 1 0
         +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
bitbuf   |                      a b c d e|
         +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

                                already filled at last time.
                               <--------->
                                7 6 5 4 3 2 1 0
                               +-+-+-+-+-+-+-+-+
                    subbitbuf  |a b c d e 1 1 1|
                               +-+-+-+-+-+-+-+-+
                                         <----->
                                          bitcount (size of subbitbuf data)
*/
void
fillbuf(struct lzh_istream *rp, int n)
{
    assert(n <= BITBUFSIZ);

    rp->bitbuf <<= n;
    while (n > rp->bitcount) {
        /* set bitcount bits from subbitbuf */
        /* note: subbitbuf << n is extended unsigned int */
        n -= rp->bitcount;
        rp->bitbuf |= rp->subbitbuf << n;
        if (rp->compsize != 0) {
            rp->compsize--;
            rp->subbitbuf = getc(rp->fp);
        }
        else
            rp->subbitbuf = 0;
        rp->bitcount = CHAR_BIT;
    }
    /* set n bits from subbitbuf */
    rp->bitcount -= n;
    rp->bitbuf |= rp->subbitbuf >> rp->bitcount;
}

unsigned int
getbits(struct lzh_istream *rp, int n)
{
    unsigned int x;

    x = rp->bitbuf >> (BITBUFSIZ - n);
    fillbuf(rp, n);
    return x;
}

void
putbits(struct lzh_ostream *wp, int n, unsigned int x)
{                               /* Write rightmost n bits of x */
    if (n < wp->bitcount) {
        wp->subbitbuf |= x << (wp->bitcount -= n);
    }
    else {
        if (wp->compsize < wp->origsize) {
            putc(wp->subbitbuf | (x >> (n -= wp->bitcount)), wp->fp);
            wp->compsize++;
        }
        else
            wp->unpackable = 1;
        if (n < CHAR_BIT) {
            wp->subbitbuf = x << (wp->bitcount = CHAR_BIT - n);
        }
        else {
            if (wp->compsize < wp->origsize) {
                putc(x >> (n - CHAR_BIT), wp->fp);
                wp->compsize++;
            }
            else
                wp->unpackable = 1;
            wp->subbitbuf = x << (wp->bitcount = 2 * CHAR_BIT - n);
        }
    }
}

int
fread_crc(void *p, int n, FILE * f, uint16_t *crc)
{
    int i;

    i = n = fread(p, 1, n, f);
    if (crc) {
        while (--i >= 0)
            UPDATE_CRC(*crc, *(unsigned char*)p++);
    }
    return n;
}

void
fwrite_crc(void *p, int n, FILE * f, uint16_t *crc)
{
    if (fwrite(p, 1, n, f) < n)
        error("Unable to write");
    if (crc) {
        while (--n >= 0)
            UPDATE_CRC(*crc, *(unsigned char*)p++);
    }
}

void
init_getbits(struct lzh_istream *rp)
{
    rp->bitbuf = 0;
    rp->subbitbuf = 0;
    rp->bitcount = 0;
    fillbuf(rp, BITBUFSIZ);
}

void
init_putbits(struct lzh_ostream *wp)
{
    wp->bitcount = CHAR_BIT;
    wp->subbitbuf = 0;
}
