//
//  LetterFix.m
//  LetterFix2
//
//  Created by kuri on 10/02/01.
//  Copyright 2010-2021 kuri. All rights reserved.
//
#import <objc/runtime.h>
#import <objc/objc-runtime.h>
#import <WebKit/WebKit.h>
#import <WebKit/DOMHTMLObjectElement.h>
#import "LetterFix.h"
#import "LFApp.h"

static NSMutableArray *khash;
static NSMutableArray *check_at_save;
static LFApp *app;
static NSString *romanNums      = @"ⅠⅡⅢⅣⅤⅥⅦⅧⅨⅩ"; // U+2160 - U+2169
static NSString *dayOfWeek      = @"㈰㈪㈫㈬㈭㈮㈯㉀㈷㉂㉃㈹㈺㈱㈾㈴㈲㈻㈶㈳㈵㈼㈽㈿㈸"; // MacJapanese 0x8740-0x8758
static NSString *circledNumbers = @"①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳ⅠⅡⅢⅣⅤ"; // MS CP932 0x8740-0x8758
static NSString *symbols        = @"㍉㌔㌢㍍㌘㌧㌃㌶㍑㍗㌍㌦㌣㌫㍊㌻㎜㎝㎞㎎㎏㏄㍾㍽㍼㍻№㏍℡㊤㊥㊦㊧㊨㈱㈲㈹∑";
static NSString *forcedReplaces = @"―～∥－￠￡￢"; // U+2015 U+FF5E U+2225 U+FF0D U+FFE0 U+FFE1 U+FFE2

static NSString *zwspaces       = @"​﻿"; // Here are invisible characters U+200B & U+FEFF.

static NSString *subOfRomanNum[] = {
    @"I", @"II", @"III", @"IV", @"V", @"VI", @"VII", @"VIII", @"IX", @"X"
};
static NSString *subOfDayOfWeek[] = {
    @"(日)",@"(月)",@"(火)",@"(水)",@"(木)",@"(金)",@"(土)",@"(祭)",@"(祝)",@"(自)",
    @"(至)",@"(代)",@"(呼)",@"(株)",@"(資)",@"(名)",@"(有)",@"(学)",@"(財)",@"(社)",
    @"(特)",@"(監)",@"(企)",@"(協)",@"(労)"
};
static NSString *subOfCircledNum[] = {
    @"(1)", @"(2)", @"(3)", @"(4)", @"(5)", @"(6)", @"(7)", @"(8)", @"(9)", @"(10)",
    @"(11)", @"(12)", @"(13)", @"(14)", @"(15)", @"(16)", @"(17)", @"(18)", @"(19)",
    @"(20)", @"I", @"II", @"III", @"IV", @"V"
};
static NSString *dependents[] = {
    @"ミリ", @"キロ", @"センチ", @"メートル", @"グラム", @"トン", @"アール", @"ヘクタール",
    @"リットル", @"ワット", @"カロリー", @"ドル", @"セント", @"パーセント", @"ミリバール",
    @"ページ", @"mm", @"cm", @"km", @"mg", @"kg", @"cc", @"明治", @"大正", @"昭和",
    @"平成", @"No.", @"K.K.", @"TEL", @"(上)", @"(中)", @"(下)", @"(左)", @"(右)",
    @"(株)", @"(有)", @"(代)", @"Σ"
};
static NSString *subOfForcedReplace[] = {
    @"—", @"〜", @"‖", @"−", @"¢", @"£", @"¬"
};

#define LF_Encoding_ISO2022JP (NSISO2022JPStringEncoding)
#define LF_Encoding_Auto      (0)

void swizzlingMethod(Class aClass, SEL aSelector, SEL nSelector, IMP nImplement) {
    Method orig_method, alt_method;
    
    if ((orig_method = class_getInstanceMethod(aClass, aSelector)) == NULL) {
	NSLog(@"Swizzling Method: Original Method is not found. [%@ %@]", NSStringFromClass(aClass), NSStringFromSelector(aSelector));
	return; // Original Method is not found.
    }
    // If orig_method was the superclass's instance method, so now we'll add the implement as method of self class.
    if (class_addMethod(aClass, aSelector, method_getImplementation(orig_method), method_getTypeEncoding(orig_method)) == YES) {
	orig_method = class_getInstanceMethod(aClass, aSelector);
    }
    if (class_addMethod(aClass, nSelector, nImplement, method_getTypeEncoding(orig_method)) == NO) {
	NSLog(@"Swizzling Method: Can't add new method.");
	return; // Can't add new method
    }
    alt_method  = class_getInstanceMethod(aClass, nSelector);
    method_exchangeImplementations(orig_method, alt_method);
    //NSLog(@"Swizzling Method: Success.");
}

static void backendSetPreferredEncoding(id backend, unsigned long long encoding)
{
    if (app.version == LF_Mavericks) {
	Ivar ivar = object_getInstanceVariable(backend, "_flags", NULL);
	_LF_flags *_flags = (_LF_flags *)((char *)backend + ivar_getOffset(ivar));
	_flags->encodingHint = encoding;
    } else if (app.version == LF_Yosemite || app.version == LF_ElCapitan ||
               app.version == LF_Sierra || app.version == LF_HighSierra ||
               app.version == LF_Mojave || app.version == LF_Catalina ||
               app.version == LF_BigSur || app.version == LF_Monterey) {
	[backend setEncodingHint: encoding];
    }
}

static unsigned long long backendGetPreferredEncoding(id backend)
{
    if (app.version == LF_Mavericks) {
	Ivar ivar = object_getInstanceVariable(backend, "_flags", NULL);
	_LF_flags *_flags = (_LF_flags *)((char *)backend + ivar_getOffset(ivar));
	return _flags->encodingHint; //FIXME: [backend _encodingHint] return wrong value?
    } else if (app.version == LF_Yosemite || app.version == LF_ElCapitan ||
               app.version == LF_Sierra || app.version == LF_HighSierra ||
               app.version == LF_Mojave || app.version == LF_Catalina ||
               app.version == LF_BigSur || app.version == LF_Monterey) {
	return [backend encodingHint];
    }
    return LF_Encoding_Auto;
}

int fixTextNode(DOMText *node, DOMRange *range, BOOL onlyCheck, BOOL onLoad)
{
    int countOfReplaced = 0;
    int c_start = -1;
    int c_end   = -1;
    
    if (node.nodeType != DOM_TEXT_NODE) return 0;

    // 文字列選択位置を保存
    if ([node isEqualNode:range.startContainer]) {
        c_start = range.startOffset;
    }
    if ([node isEqualNode:range.endContainer]) {
        c_end = range.endOffset;
    }
    
    for (NSInteger i = 0; i < node.data.length; i++) {
        NSRange crange = [node.data rangeOfComposedCharacterSequenceAtIndex:i];
        NSString *aChar = [node.data substringWithRange:crange];
        NSInteger loc;
        NSString *replaced = nil;
        BOOL      dirtyFlag = NO;
        
        if ((loc = [forcedReplaces rangeOfString:aChar].location) != NSNotFound) {
            replaced = subOfForcedReplace[loc];
            dirtyFlag = YES;
        } else if ([aChar canBeConvertedToEncoding:NSISO2022JPStringEncoding] != YES &&
                   [aChar characterAtIndex:0] != 0xa0) {
            countOfReplaced++;
            if (app.shouldFixOsDependents == YES) {
                if ((loc = [circledNumbers rangeOfString:aChar].location) != NSNotFound &&
                    loc < 20) {
                    replaced = subOfCircledNum[loc];
                } else if ((loc = [romanNums rangeOfString:aChar].location) != NSNotFound) {
                    replaced = subOfRomanNum[loc];
                } else if ((loc = [dayOfWeek rangeOfString:aChar].location) != NSNotFound) {
                    if (onLoad && app.shouldFixParenSymbols == YES &&
                        !(app.shouldNotFixSomeParenSymbols == YES &&
                          (loc == 11 || loc == 13 || loc == 16))) {
                            replaced = subOfCircledNum[loc];
                        } else {
                            replaced = subOfDayOfWeek[loc];
                        }
                } else if ((loc = [symbols rangeOfString:aChar].location) != NSNotFound) {
                    replaced = dependents[loc];
                }
            }
            if (replaced == nil && app.shouldFixAllLetters == YES) {
                if (([zwspaces rangeOfString:aChar].location) != NSNotFound &&
                    app.shouldIgnoreZeroWidthSpace == YES) {
                    replaced = @"";
                } else {
                    replaced = @"〓";
                    if (app.shouldAppendCodeInfo == YES) {
                        replaced = [replaced stringByAppendingString:@"("];
                        for (int j = 0; j < [aChar length]; j++) {
                            if (j!=0) {
                                replaced = [replaced stringByAppendingString:@" "];
                            }
                            if ([[aChar substringWithRange:NSMakeRange(j, 1)] canBeConvertedToEncoding:NSISO2022JPStringEncoding] == YES) {
                                replaced = [replaced stringByAppendingString:[aChar substringWithRange:NSMakeRange(j, 1)]];
                            } else {
                                replaced = [replaced stringByAppendingString:
                                            [NSString stringWithFormat:@"U+%04X", [aChar characterAtIndex:j]]];
                            }
                        }
                        replaced = [replaced stringByAppendingString:@")"];
                    }
                }
            }
        }
        
        if (replaced != nil && (onlyCheck == NO || dirtyFlag == YES)) {
            if (c_start != -1 && i < c_start) {
                c_start += replaced.length - aChar.length;
            }
            if (c_end != -1 && i < c_end) {
                c_end += replaced.length - aChar.length;
            }
            [node replaceData:(unsigned int)i length:(unsigned int)crange.length data:replaced];
            i += replaced.length - 1;
        } else {
            i += crange.length - 1;
        }
    }
    
    if (!onlyCheck) {
        if ((c_start != -1) && 0 < countOfReplaced) {
            [range setStart:node offset:c_start];
        }
        if ((c_end != -1) && 0 < countOfReplaced) {
            [range setEnd:node offset:c_end];
        }
    }
    
    return countOfReplaced;
}

int fixReflexively(DOMNode *node, DOMRange *range, BOOL onlyCheck, BOOL onLoad)
{
    DOMNode *n;
    int      countOfReplaced = 0;
    
    if (node.hasChildNodes) {
	for (n = node.firstChild; n != NULL; n = n.nextSibling) {
	    countOfReplaced += fixReflexively(n, range, onlyCheck, onLoad); // 再帰的にノードをたどる
	}
    } else {
	if (node.nodeType == DOM_TEXT_NODE) {
	    countOfReplaced = fixTextNode((DOMText*)node, range, onlyCheck, onLoad);
	}
    }
    return countOfReplaced;
}

int fixLetter(id self, id composeView, BOOL onlyCheck, BOOL onLoad)
{
    int countOfReplaced = 0;
    
    @try {
	WebFrame    *frame     = [(WebView *)composeView mainFrame];
	DOMDocument *dom       = [frame DOMDocument];
	DOMElement  *root      = [dom documentElement];
	DOMNodeList *nodelist  = [root getElementsByTagName:@"body"];
	
	if ([nodelist length] != 1) return -1;
	
	DOMNode     *body      = [nodelist item:0];
	DOMRange    *range     = [composeView selectedDOMRange];
	
	countOfReplaced = fixReflexively(body, range, onlyCheck, onLoad);
	
	if (0 < countOfReplaced && !onlyCheck) {
	    [composeView setSelectedDOMRange:range affinity:NSSelectionAffinityUpstream];
	}
    }
    @catch (NSException * e) {
	NSLog(@"LetterFix: caught %@ %@", [e name], [e reason]);
    }
    
    return countOfReplaced;
}

BOOL fixSubject(id self, BOOL onlyCheck, BOOL onLoad)
{
    BOOL result = NO;
    NSMutableString *subject = [NSMutableString stringWithString:[[self backEnd] subject]];
    
    for (NSInteger i = 0; i < subject.length; i++) {
        NSRange   crange = [subject rangeOfComposedCharacterSequenceAtIndex:i];
        NSString *aChar = [subject substringWithRange:crange];
        NSInteger loc;
        NSString *replaced = nil;

        if ([aChar canBeConvertedToEncoding:NSISO2022JPStringEncoding] != YES) {
            if ((loc = [forcedReplaces rangeOfString:aChar].location) != NSNotFound) {
                replaced = subOfForcedReplace[loc];
            }
            if (onLoad == NO) {
                result = YES;
                if (replaced == nil && app.shouldFixOsDependents == YES) {
                    if ((loc = [circledNumbers rangeOfString:aChar].location) != NSNotFound &&
                        loc < 20) {
                        replaced = subOfCircledNum[loc];
                    } else if ((loc = [romanNums rangeOfString:aChar].location) != NSNotFound) {
                        replaced = subOfRomanNum[loc];
                    } else if ((loc = [symbols rangeOfString:aChar].location) != NSNotFound) {
                        replaced = dependents[loc];
                    } else if ((loc = [dayOfWeek rangeOfString:aChar].location) != NSNotFound) {
                        replaced = subOfDayOfWeek[loc];
                    }
                }
                if (replaced == nil && app.shouldFixAllLetters == YES) {
                    if (([zwspaces rangeOfString:aChar].location) != NSNotFound &&
                        app.shouldIgnoreZeroWidthSpace == YES) {
                        replaced = @"";
                    } else {
                        replaced = @"〓";
                        if (app.shouldAppendCodeInfo == YES) {
                            replaced = [replaced stringByAppendingString:@"("];
                            for (int j = 0; j < [aChar length]; j++) {
                                if (j != 0) {
                                    replaced = [replaced stringByAppendingString:@" "];
                                }
                                if ([[aChar substringWithRange:NSMakeRange(j, 1)] canBeConvertedToEncoding:NSISO2022JPStringEncoding] == YES) {
                                    replaced = [replaced stringByAppendingString:[aChar substringWithRange:NSMakeRange(j, 1)]];
                                } else {
                                    replaced = [replaced stringByAppendingString:
                                                [NSString stringWithFormat:@"U+%04X", [aChar characterAtIndex:j]]];
                                }
                            }
                            replaced = [replaced stringByAppendingString:@")"];
                        }
                    }
                }
            } else { // onLoad == YES
                if (app.shouldFixOsDependents == YES &&
                    app.shouldFixParenSymbols == YES &&
                    (loc = [dayOfWeek rangeOfString:aChar].location) != NSNotFound &&
                    (app.shouldNotFixSomeParenSymbols == NO ||
                     (loc != 11 && loc != 13 && loc != 16))) {
                        replaced = subOfCircledNum[loc];
                        result = YES;
                    }
            }
        }
        if (onlyCheck == YES && result == YES) {
            return result;
        } else if (onlyCheck == NO && replaced != nil) {
            [subject replaceCharactersInRange:crange withString:replaced];
            i += replaced.length - 1;
        } else {
            i += crange.length - 1;
        }
    }
    
    if (onlyCheck == NO) {
    [[(LetterFix *)self backEnd] setSubject:subject];
    [[self _windowLF] setTitle:subject];
    
    id headers = [self headersEditor];
    NSTextField *subjectField = nil;
    object_getInstanceVariable(headers, "_subjectField", (void **)&subjectField);
    if (subjectField != nil)
        [subjectField setStringValue:subject];
    }
    
    return result;
}

BOOL _LF_IMP_isLoaded(id self, SEL _cmd)
{
    BOOL result = [self _LF_isLoaded]; // call swizzled(original) method
    if (app.isActive == NO) {
	return result;
    }
    if (result == NO) {
	[khash removeObject:self];
	return result;
    }
    if ([khash containsObject:self] == YES) return result;
    
    @try {
	id composeView = [self webView];
        if ((composeView==NULL) || ([(WebView *)composeView isLoading]==YES) || ([(WebView *)composeView isEditable]==NO)) {
	    return result;
        }
        
	[khash addObject:self];
        
	switch ([self messageType]) {
	    case 1: // 返信
	    case 2: // 全員に返信
	    case 3: // 転送
	    case 4: // 下書きを開く
	    case 7: // リダイレクト
	    case 8: // 差出人に返信
		break;
	    case 5: // 新規メッセージ
	    case 14:// 添付ファイルとして返信
		return result;
	    default:
		NSLog(@"Unknown MessageType: %d", [self messageType]);
		return result;
	}
        
	if (0 < fixLetter(self, [self webView], YES, YES) || fixSubject(self, YES, YES) == YES) {
            if (app.operationModeOnLoad == 0) {
                NSAlert *alert = [[[NSAlert alloc] init] autorelease];
                [alert addButtonWithTitle:@"変換"];
                NSButton *button = [alert addButtonWithTitle:@"変換しない"];
                button.keyEquivalent = @"\x1b";
                [alert setShowsSuppressionButton:YES];
                [alert setMessageText:@"編集前にメッセージを変換しますか？"];
                [alert setInformativeText:@"このメッセージには ISO 2022-JP でエンコードできない文字が含まれています。"];
                [alert setAlertStyle:NSInformationalAlertStyle];
                dispatch_after(dispatch_time(DISPATCH_TIME_NOW, (int64_t)(0.8 * NSEC_PER_SEC)), dispatch_get_main_queue(), ^{
                    [alert beginSheetModalForWindow:[self _windowLF] completionHandler:^(NSModalResponse returnCode){
                        if (returnCode == NSAlertFirstButtonReturn) {
                            fixSubject(self, NO, YES);
                            if (app.version == LF_Mavericks || app.version == LF_Yosemite) {
                                fixLetter(self, [self webView], NO, YES);
                            } else if (app.version == LF_ElCapitan || app.version == LF_Sierra ||
                                       app.version == LF_HighSierra|| app.version == LF_Mojave ||
                                       app.version == LF_Catalina  || app.version == LF_BigSur ||
                                       app.version == LF_Monterey) {
                                fixLetter(self, [self composeWebView], NO, YES);
                            }
                        } else if (returnCode == NSAlertSecondButtonReturn) {
                        }
                        if ([[alert suppressionButton] state] == NSOnState) {
                            if (returnCode == NSAlertFirstButtonReturn) {
                                app.operationModeOnLoad = 1;
                            } else if (returnCode == NSAlertSecondButtonReturn) {
                                app.operationModeOnLoad = 2;
                            }
                        }
                    }];
                });
            } else if (app.operationModeOnLoad == 1) {
                fixSubject(self, NO, YES);
                fixLetter(self, [self webView], NO, YES);
                [[self backEnd] setHasChanges:NO];
            }
	}
    }
    @catch (NSException * exception) {
	NSLog(@"LetterFix: caught %@ %@", [exception name], [exception reason]);
    }
    
    return result;
}

void _LF_IMP_finishLoadingEditor(id self, SEL _cmd)
{
    [self _LF_finishLoadingEditor];

    if (app.isActive == NO) {
        return;
    }

    @try {
        id composeView = [self composeWebView];
        if ((composeView==NULL) || ([(WebView *)composeView isLoading]==YES) || ([(WebView *)composeView isEditable]==NO)) {
            NSLog(@"composeView NULL!!");
            return;
        }
        
        switch ([self messageType]) {
            case 1: // 返信
            case 2: // 全員に返信
            case 3: // 転送
            case 4: // 下書きを開く
            case 7: // リダイレクト
            case 8: // 差出人に返信
                break;
            case 5: // 新規メッセージ
            case 14:// 添付ファイルとして返信
                return;
            default:
                NSLog(@"Unknown MessageType: %d", [self messageType]);
                return;
        }
        
        if (0 < fixLetter(self, composeView, YES, YES) || fixSubject(self, YES, YES)) {
            if (app.operationModeOnLoad == 0) {
                NSAlert *alert = [[[NSAlert alloc] init] autorelease];
                [alert addButtonWithTitle:@"変換"];
                NSButton *button = [alert addButtonWithTitle:@"変換しない"];
		button.keyEquivalent = @"\x1b";
                [alert setShowsSuppressionButton:YES];
                [alert setMessageText:@"編集前にメッセージを変換しますか？"];
                [alert setInformativeText:@"このメッセージには ISO 2022-JP でエンコードできない文字が含まれています。"];
                [alert setAlertStyle:NSInformationalAlertStyle];
                dispatch_after(dispatch_time(DISPATCH_TIME_NOW, (int64_t)(0.5 * NSEC_PER_SEC)), dispatch_get_main_queue(), ^{
                    [alert beginSheetModalForWindow:[self _windowLF] completionHandler:^(NSModalResponse returnCode){
                        if (returnCode == NSAlertFirstButtonReturn) {
                            fixSubject(self, NO, YES);
                            if (app.version == LF_ElCapitan || app.version == LF_Sierra ||
                                app.version == LF_HighSierra|| app.version == LF_Mojave ||
                                app.version == LF_Catalina  || app.version == LF_BigSur ||
                                app.version == LF_Monterey) {
                                fixLetter(self, [self composeWebView], NO, YES);
                            }
                        } else if (returnCode == NSAlertSecondButtonReturn) {
                        }
                        if ([[alert suppressionButton] state] == NSOnState) {
                            if (returnCode == NSAlertFirstButtonReturn) {
                                app.operationModeOnLoad = 1;
                            } else if (returnCode == NSAlertSecondButtonReturn) {
                                app.operationModeOnLoad = 2;
                            }
                        }
                    }];
                });
            } else if (app.operationModeOnLoad == 1) {
                fixSubject(self, NO, YES);
                fixLetter(self, composeView, NO, YES);
                [[self backEnd] setHasChanges:NO];
            }
        }
    }
    @catch (NSException * exception) {
        NSLog(@"LetterFix: caught %@ %@", [exception name], [exception reason]);
    }
    
    return;
}

void _LF_IMP_send_(id self, SEL _cmd, id arg1)
{
    if (app.isActive == NO) {
	[self _LF_send: arg1];
	return;
    }

    id composeView;
    if (app.version == LF_Mavericks || app.version == LF_Yosemite) {
        composeView = [self webView];
    } else if (app.version == LF_ElCapitan || app.version == LF_Sierra ||
               app.version == LF_HighSierra|| app.version == LF_Mojave ||
               app.version == LF_Catalina  || app.version == LF_BigSur ||
               app.version == LF_Monterey) {
        composeView = [self composeWebView];
    } else {
        [self _LF_send: arg1];
        return;
    }
    
    backendSetPreferredEncoding([self backEnd], LF_Encoding_ISO2022JP);
    if (0 < fixLetter(self, composeView, YES, NO)) {
        NSAlert *alert = [[[NSAlert alloc] init] autorelease];
	[alert addButtonWithTitle:@"変換して送信"];
	NSButton *button = [alert addButtonWithTitle:@"キャンセル"];
	button.keyEquivalent = @"\x1b";
	[alert addButtonWithTitle:@"変換せずに送信"];
	[alert addButtonWithTitle:@"変換のみ"];
	[alert setMessageText:@"メッセージを変換して送信しますか？"];
	[alert setInformativeText:@"送信しようとしているメッセージには ISO 2022-JP でエンコードできない文字が含まれています。"];
	[alert setAlertStyle:NSInformationalAlertStyle];
        [alert beginSheetModalForWindow:[self _windowLF] completionHandler:^(NSModalResponse returnCode) {
            if (returnCode == NSAlertFirstButtonReturn) {
                if (app.version == LF_Mavericks || app.version == LF_Yosemite) {
                    fixLetter(self, [self webView], NO, NO);
                } else if (app.version == LF_ElCapitan || app.version == LF_Sierra ||
                           app.version == LF_HighSierra|| app.version == LF_Mojave ||
                           app.version == LF_Catalina  || app.version == LF_BigSur ||
                           app.version == LF_Monterey) {
                    fixLetter(self, [self composeWebView], NO, NO);
                }
                if (backendGetPreferredEncoding([self backEnd]) != LF_Encoding_ISO2022JP) backendSetPreferredEncoding([self backEnd], LF_Encoding_ISO2022JP);
                [[alert window] orderOut:arg1];
                if ((app.shouldCheckSubject != NO) && fixSubject(self, YES, NO) == YES) {
                    _LF_alertSubject(self, _cmd, arg1);
                }
                else [self _LF_send: arg1];
            } else if (returnCode == NSAlertSecondButtonReturn) {
                // Cancel
            } else if (returnCode == NSAlertThirdButtonReturn) { // 変換せずに送信
                if (backendGetPreferredEncoding([self backEnd]) == LF_Encoding_ISO2022JP) {
                    backendSetPreferredEncoding([self backEnd], LF_Encoding_Auto); // ISO2022JPのままだとISO2022JP-2とかで送ってしまう
                }
                [[alert window] orderOut:arg1];
                [self _LF_send: arg1];
            } else if (returnCode == (NSAlertThirdButtonReturn + 1)) { // 変換のみ
                if (app.version == LF_Mavericks || app.version == LF_Yosemite) {
                    fixLetter(self, [self webView], NO, NO);
                } else if (app.version == LF_ElCapitan || app.version == LF_Sierra ||
                           app.version == LF_HighSierra|| app.version == LF_Mojave ||
                           app.version == LF_Catalina  || app.version == LF_BigSur ||
                           app.version == LF_Monterey) {
                    fixLetter(self, [self composeWebView], NO, NO);
                }
                if (backendGetPreferredEncoding([self backEnd]) != LF_Encoding_ISO2022JP) backendSetPreferredEncoding([self backEnd], LF_Encoding_ISO2022JP);
            }
        }];
    } else if ((app.shouldCheckSubject != NO) && fixSubject(self, YES, NO) == YES) {
	_LF_alertSubject(self, _cmd, arg1);
    } else {
	[self _LF_send: arg1];
    }
}

void _LF_alertSubject(id self, SEL _cmd, id arg1)
{
    NSAlert *alert = [[[NSAlert alloc] init] autorelease];
    [alert addButtonWithTitle:@"変換して送信"];
    NSButton *button = [alert addButtonWithTitle:@"キャンセル"];
    button.keyEquivalent = @"\x1b";
    [alert addButtonWithTitle:@"変換せずに送信"];
    [alert addButtonWithTitle:@"変換のみ"];
    [alert setMessageText:@"件名を変換して送信しますか？"];
    [alert setInformativeText:@"送信するメッセージの件名には ISO 2022-JP でエンコードできない文字が含まれています。"];
    [alert setAlertStyle:NSInformationalAlertStyle];
    [alert beginSheetModalForWindow:[self _windowLF] completionHandler:^(NSModalResponse returnCode) {
        if (returnCode == NSAlertFirstButtonReturn) {
            fixSubject(self, NO, NO);
            [[alert window] orderOut:arg1];
            [self _LF_send: arg1];
        } else if (returnCode == NSAlertSecondButtonReturn) {
            // Cancel
        } else if (returnCode == NSAlertThirdButtonReturn) {
            [[alert window] orderOut:arg1];
            [self _LF_send: arg1];
        } else if (returnCode == (NSAlertThirdButtonReturn + 1)) {
            fixSubject(self, NO, NO);
        }
    }];
}

void _LF_IMP_saveDocument_(id self, SEL _cmd, id arg1)
{
    id composeView;
    if (app.shouldCheckOnSave && (app.version == LF_Mavericks || app.version == LF_Yosemite)) {
        composeView = [self webView];
    } else if (app.shouldCheckOnSave &&
               (app.version == LF_ElCapitan || app.version == LF_Sierra ||
                app.version == LF_HighSierra|| app.version == LF_Mojave ||
                app.version == LF_Catalina  || app.version == LF_BigSur ||
                app.version == LF_Monterey)) {
        composeView = [self composeWebView];
    } else {
        [self _LF_saveDocument: arg1];
        return;
    }
    
    if ((app.isActive != NO) && ([check_at_save containsObject:self]==NO) &&(0 < fixLetter(self, composeView, YES, NO))) {
	NSAlert *alert = [[[NSAlert alloc] init] autorelease];
	[alert addButtonWithTitle:@"変換"];
	NSButton *button = [alert addButtonWithTitle:@"変換しない"];
	button.keyEquivalent = @"\x1b";
	[alert setMessageText:@"保存前にメッセージを変換しますか？"];
	[alert setInformativeText:@"このメッセージには ISO 2022-JP でエンコードできない文字が含まれています。\n"
	 @"変換しないを選択した場合、メールの送信またはウインドウを開き直すまでエンコーディングの確認を行いません。"];
	[alert setAlertStyle:NSWarningAlertStyle];
        [alert beginSheetModalForWindow:[self _windowLF] completionHandler:^(NSModalResponse returnCode) {
            if (returnCode == NSAlertFirstButtonReturn) {
                if (app.version == LF_Mavericks || app.version == LF_Yosemite) {
                    fixLetter(self, [self webView], NO, NO);
                } else if (app.version == LF_ElCapitan || app.version == LF_Sierra ||
                           app.version == LF_HighSierra|| app.version == LF_Mojave ||
                           app.version == LF_Catalina  || app.version == LF_BigSur ||
                           app.version == LF_Monterey) {
                    fixLetter(self, [self composeWebView], NO, NO);
                }
            } else if (returnCode == NSAlertSecondButtonReturn) {
                [check_at_save addObject:self];
            }
            [[alert window] orderOut:arg1];
            [self _LF_saveDocument: (id)arg1];
        }];
    } else {
	[self _LF_saveDocument: arg1];
    }
}

void _LF_IMP_animationCompleted(id self, SEL _cmd)
{
    [self performSelector:@selector(isLoaded) withObject:nil afterDelay:0.5];
    [self _LF_animationCompleted];
}

NSWindow *_LF_IMP_window(id self, SEL _cmd)
{
    if (app.version == LF_Mavericks || app.version == LF_Yosemite || app.version == LF_ElCapitan ) {
        return [self window];
    } else if (app.version == LF_Sierra || app.version == LF_HighSierra||
               app.version == LF_Mojave || app.version == LF_Catalina  ||
               app.version == LF_BigSur || app.version == LF_Monterey) {
        return [(NSView *)[self view] window];
    }
    return nil;
}

#define MVMailBundle        (NSClassFromString(@"MVMailBundle"))

@implementation LetterFix
+ (void) initialize
{
    class_setSuperclass([self class], MVMailBundle); // depricated function 10.5対応のためこのワーニングだけは消せません
    [super initialize];
    [self registerBundle];
    
    khash = [[NSMutableArray alloc] initWithCapacity:1];
    check_at_save = [[NSMutableArray alloc] initWithCapacity:1];
    
    app = [[LFApp alloc] init];
    
    Class editorClass = nil;
    if (app.version == LF_Mavericks || app.version == LF_Yosemite) {
        editorClass = NSClassFromString(@"DocumentEditor");
    } else if (app.version == LF_ElCapitan || app.version == LF_Sierra ||
               app.version == LF_HighSierra|| app.version == LF_Mojave ||
               app.version == LF_Catalina  || app.version == LF_BigSur ||
               app.version == LF_Monterey) {
        editorClass = NSClassFromString(@"ComposeViewController");
    } else {
        return;
    }
    class_addMethod(editorClass, @selector(_windowLF), (IMP)_LF_IMP_window, "@@:");
    
    //
    // swizzling method
    //
    if (app.version == LF_Mavericks || app.version == LF_Yosemite) {
        swizzlingMethod(editorClass, @selector(isLoaded), @selector(_LF_isLoaded), (IMP)_LF_IMP_isLoaded);
        swizzlingMethod(editorClass, @selector(_animationCompleted), @selector(_LF_animationCompleted), (IMP)_LF_IMP_animationCompleted);
    } else if (app.version == LF_ElCapitan) {
        swizzlingMethod(editorClass, @selector(finishLoadingEditor), @selector(_LF_finishLoadingEditor), (IMP)_LF_IMP_finishLoadingEditor);
    } else if (app.version == LF_Sierra || app.version == LF_HighSierra ||
               app.version == LF_Mojave || app.version == LF_Catalina ||
               app.version == LF_BigSur || app.version == LF_Monterey) {
        swizzlingMethod(editorClass, @selector(_finishLoadingEditor), @selector(_LF_finishLoadingEditor), (IMP)_LF_IMP_finishLoadingEditor);
    }
    swizzlingMethod(editorClass, @selector(saveDocument:), @selector(_LF_saveDocument:), (IMP)_LF_IMP_saveDocument_);
    swizzlingMethod(editorClass, @selector(send:), @selector(_LF_send:), (IMP)_LF_IMP_send_);
    //
    // end of swizzling method
    //
    
    NSLog(@"LetterFix Plugin (version %s/%@) is registered.", LETTERFIX_VERSION, app.mailVersion);
}
@end
