(ns test.graph
  (:use clojure.contrib.test-is))

(require 'src.graph)
(alias 'gr 'src.graph)

(require 'src.unify-matcher)
(alias 'um 'src.unify-matcher)

(deftest test-transpose
  (is (= (gr/transpose '[[0 1 2] [3 4 5]])
         '[[0 3] [1 4] [2 5]]
         )))

(deftest test-new-ids
  (let [gr0 (gr/new-graph)
        [gr1 & ids] (gr/new-ids gr0 'a 'bc 'def 'e)]
    (is (= gr0 {:next-id 0 :nodes [] :edges []}))
    (is (= gr1 {:next-id 4 :nodes [] :edges []}))
    (is (= ids '(a-0 bc-1 def-2 e-3)))
    ))


; gr-not1
;    _           |\            _
; b |_|----------| >o---------|_| z
;                |/
;
;    _           |\            _
; a |_|----------| >o---------|_| y
;                |/
;
(def gr-not1
  (let [gr (gr/new-graph)
        [gr a b] (gr/inputs gr 'a 'b)
        [gr y] (gr/not1 gr a)
        [gr z] (gr/not1 gr b)
        gr (gr/outputs gr [y 'y] [z 'z])]
    gr))

(deftest test-not1
  (is (= gr-not1
         {:next-id 10
          :nodes '[[n-0 in a]
                   [n-1 in b]
                   [n-3 not1]
                   [n-5 not1]
                   [n-7 out y]
                   [n-9 out z]
                   ]
          :edges '[[e-2 [n-0 0] [n-3 0]]
                   [e-4 [n-1 0] [n-5 0]]
                   [e-6 [n-3 0] [n-7 0]]
                   [e-8 [n-5 0] [n-9 0]]
                   ]})))

; gr-and2
;    _            __
; b |_|----------|  \          _
;    _           |   |--------|_| z
; a |_|----------|__/
;
(def gr-and2
  (let [gr (gr/new-graph)
        [gr a b] (gr/inputs gr 'a 'b)
        [gr z] (gr/and2 gr a b)
        gr (gr/outputs gr [z 'z])]
    gr))

(deftest test-and2
  (is (= gr-and2
         {:next-id 7
          :nodes '[[n-0 in a]
                   [n-1 in b]
                   [n-4 and2]
                   [n-6 out z]
                   ]
          :edges '[[e-2 [n-0 0] [n-4 0]]
                   [e-3 [n-1 0] [n-4 1]]
                   [e-5 [n-4 0] [n-6 0]]
                   ]})))

; gr-andn
;    _                          __
; c |_|------------------------|  \          _
;    _            __           |   |--------|_| z
; b |_|----------|  \     +----|__/
;    _           |   |----+
; a |_|----------|__/
;
(def gr-andn
  (let [gr (gr/new-graph)
        [gr a b c] (gr/inputs gr 'a 'b 'c)
        [gr z] (gr/andn gr a b c)
        gr (gr/outputs gr [z 'z])]
    gr))

(deftest test-andn
  (is (= gr-andn
        {:next-id 11
         :nodes '[[n-0 in a]
                  [n-1 in b]
                  [n-2 in c]
                  [n-5 and2]
                  [n-8 and2]
                  [n-10 out z]
                  ]
         :edges '[[e-3 [n-0 0] [n-5 0]]
                  [e-4 [n-1 0] [n-5 1]]
                  [e-6 [n-5 0] [n-8 0]]
                  [e-7 [n-2 0] [n-8 1]]
                  [e-9 [n-8 0] [n-10 0]]
                  ]})))

; gr-or2
;    _            __
; b |_|----------\  \          _
;    _            |  |--------|_| z
; a |_|----------/__/
;
(def gr-or2
  (let [gr (gr/new-graph)
        [gr a b] (gr/inputs gr 'a 'b)
        [gr z] (gr/or2 gr a b)
        gr (gr/outputs gr [z 'z])]
    gr))

(deftest test-or2
  (is (= gr-or2
         {:next-id 7
          :nodes '[[n-0 in a]
                   [n-1 in b]
                   [n-4 or2]
                   [n-6 out z]
                   ]
          :edges '[[e-2 [n-0 0] [n-4 0]]
                   [e-3 [n-1 0] [n-4 1]]
                   [e-5 [n-4 0] [n-6 0]]
                   ]})))

; gr-orn
;    _                          __
; c |_|------------------------\  \          _
;    _            __            |  |--------|_| z
; b |_|----------\  \     +----/__/
;    _            |  |----+
; a |_|----------/__/
;
(def gr-orn
  (let [gr (gr/new-graph)
        [gr a b c] (gr/inputs gr 'a 'b 'c)
        [gr z] (gr/orn gr a b c)
        gr (gr/outputs gr [z 'z])]
    gr))

(deftest test-orn
  (is (= gr-orn
        {:next-id 11
         :nodes '[[n-0 in a]
                  [n-1 in b]
                  [n-2 in c]
                  [n-5 or2]
                  [n-8 or2]
                  [n-10 out z]
                  ]
         :edges '[[e-3 [n-0 0] [n-5 0]]
                  [e-4 [n-1 0] [n-5 1]]
                  [e-6 [n-5 0] [n-8 0]]
                  [e-7 [n-2 0] [n-8 1]]
                  [e-9 [n-8 0] [n-10 0]]
                  ]})))

(deftest test-random-graph
  (is (= (gr/random-graph-pre (iterate inc 0) 2 1 2)
         {:next-id 10
          :nodes '[[n-0 in   0]
                   [n-1 in   1]
                   [n-4 and2  ] ; n-node -> 2, random -> 0 -> and2
                   [n-7 and2  ] ; n-node -> 1, random -> 3 -> and2
                   [n-9 out  0]
                   ]
          :edges '[[e-2 [n-1 0] [n-4 0]] ; n-node -> 2, random -> 1 -> n-1
                   [e-3 [n-0 0] [n-4 1]] ; n-node -> 2, random -> 2 -> n-0
                   [e-5 [n-1 0] [n-7 0]] ; n-node -> 1, random -> 4 -> n-1
                   [e-6 [n-4 0] [n-7 1]] ; n-node -> 1, random -> 5 -> n-4
                   [e-8 [n-4 0] [n-9 0]] ; n-node -> 0, random -> 6 -> n-4
                   ]}))
  (is (= (gr/random-graph (iterate inc 0) 2 1 2)
         {:next-id 10
          :nodes '[[n-0 in   0]
                   [n-1 in   1]
                   [n-4 and2  ] ; n-node -> 2, random -> 0 -> and2
                   [n-9 out  0]
                   ]
          :edges '[[e-2 [n-1 0] [n-4 0]] ; n-node -> 2, random -> 1 -> n-1
                   [e-3 [n-0 0] [n-4 1]] ; n-node -> 2, random -> 2 -> n-0
                   [e-8 [n-4 0] [n-9 0]] ; n-node -> 0, random -> 6 -> n-4
                   ]})))


(def gr-unrefered-conns1
  (struct gr/graph 12
    '[[n-0 in a]
      [n-1 in b]
      [n-2 in c]
      [n-5 and2]
      [n-7 not1]
      [n-9 not1]
      [n-11 out z]
      ]
    '[[e-3 [n-0 0] [n-5 0]]
      [e-4 [n-1 0] [n-5 1]]
      [e-6 [n-2 0] [n-7 0]]
      [e-8 [n-7 0] [n-9 0]]
      [e-10 [n-5 0] [n-11 0]]
      ]))

(deftest test-remove-unrefered-conns
  (is (= (gr/remove-unrefered-conns gr-unrefered-conns1)
         {:next-id 12
          :nodes '[[n-0 in a]
                   [n-1 in b]
                   [n-2 in c]
                   [n-5 and2]
                   [n-11 out z]
                   ]
          :edges '[[e-3 [n-0 0] [n-5 0]]
                   [e-4 [n-1 0] [n-5 1]]
                   [e-10 [n-5 0] [n-11 0]]
                   ]})))

(deftest test-dependency
  (is (= (gr/dependency gr-orn 'e-9)
         '[[e-9 [or2 0] e-6 e-7]
           [e-6 [or2 0] e-3 e-4]
           [e-3 [in a]]
           [e-4 [in b]]
           [e-7 [in c]]
           ])))

(deftest test-calc-along-dependency-1
  (is (= (gr/calc-along-dependency
           '[[e-3 [not1 0] e-1]
             [e-1 [in a]]
             ]
           '{a 0})
         '{e-1 0, e-3 1}))
  (let [dep '[[e-5 [and2 0] e-1 e-3]
              [e-1 [in a]]
              [e-3 [in b]]
              ]]
    (is (= (gr/calc-along-dependency dep '{a 1, b 0})
           '{e-1 1, e-3 0, e-5 0}))
    (is (= (gr/calc-along-dependency dep '{a 1, b 1})
           '{e-1 1, e-3 1, e-5 1})))
  (let [dep '[[e-5 [or2 0] e-1 e-3]
              [e-1 [in a]]
              [e-3 [in b]]
              ]]
    (is (= (gr/calc-along-dependency dep '{a 0, b 0})
           '{e-1 0, e-3 0, e-5 0}))
    (is (= (gr/calc-along-dependency dep '{a 1, b 0})
           '{e-1 1, e-3 0, e-5 1}
            ))))
;    _                      __
; b |_|--------------------|  \  f
;                          |   |---+
;            +-------------|__/    |   __
;    _       |   |\         __     +--\  \        _
; c |_|------+---| >o------|  \        |  |------|_| z
;    _           |/    d   |   |------/__/
; a |_|--------------------|__/   e
(def sel-2in1
  (let [gr (gr/new-graph)
        [gr a b c] (gr/inputs gr 'a 'b 'c)
        [gr d] (gr/not1 gr c)
        [gr e] (gr/and2 gr a d)
        [gr f] (gr/and2 gr b c)
        [gr z] (gr/or2 gr e f)
        gr (gr/outputs gr [z 'z])]
    gr))

(deftest test-sel-2in1
  (is (= sel-2in1
         {:next-id 16
          :nodes '[[n-0  in   a]
                   [n-1  in   b]
                   [n-2  in   c]
                   [n-4  not1  ]
                   [n-7  and2  ]
                   [n-10 and2  ]
                   [n-13 or2   ]
                   [n-15 out  z]
                   ]
          :edges '[[e-3  [n-2  0] [n-4  0]]
                   [e-5  [n-0  0] [n-7  0]]
                   [e-6  [n-4  0] [n-7  1]]
                   [e-8  [n-1  0] [n-10 0]]
                   [e-9  [n-2  0] [n-10 1]]
                   [e-11 [n-7  0] [n-13 0]]
                   [e-12 [n-10 0] [n-13 1]]
                   [e-14 [n-13 0] [n-15 0]]
                   ]})))

(deftest test-all-dpt-terms
  (is (= (gr/all-dpt-terms sel-2in1)
         '[[n-0  0]
           [n-1  0]
           [n-2  0]
           [n-4  0]
           [n-7  0]
           [n-10 0]
           [n-13 0]
           ])))

(deftest test-calc-along-dependency-2
  (let [[z-nid _ _] (first (filter (fn [[_ type o-name]]
                                     (and (= type 'out)
                                          (= o-name 'z)))
                                   (:nodes sel-2in1)))
        [z-eid _ _] (first (filter (fn [[_ _ [nid _]]]
                                     (= nid z-nid))
                                   (:edges sel-2in1)))]
    (let [result (gr/calc-along-dependency
                   (gr/dependency sel-2in1 z-eid)
                   '{a 0, b 1, c 0})]
      (is (= (result z-eid) 0)))
    (let [result (gr/calc-along-dependency
                   (gr/dependency sel-2in1 z-eid)
                   '{a 0, b 1, c 1})]
      (is (= (result z-eid) 1))
      )))

(deftest test-calc
  (is (= (gr/calc sel-2in1 '{a 1, b 0, c 0})
         '{z 1}))
  (is (= (gr/calc sel-2in1 '{a 1, b 0, c 1})
         '{z 0})))

(def ha-table '[[0 0] [0 0]
                [1 0] [1 0]
                [0 1] [1 0]
                [1 1] [0 1]])

(def gr-ha
  (let [gr          (gr/new-graph)
        [gr a b]    (gr/inputs gr 'a 'b)
        [gr s cout] (gr/truth-table gr (partition 2 ha-table) [a b])
        gr          (gr/outputs gr [s 's] [cout 'cout])]
    (gr/remove-unrefered-conns gr)))

(deftest test-calc-ha
  (is (= (gr/calc gr-ha '{a 0, b 0}) '{s 0, cout 0}))
  (is (= (gr/calc gr-ha '{a 1, b 0}) '{s 1, cout 0}))
  (is (= (gr/calc gr-ha '{a 0, b 1}) '{s 1, cout 0}))
  (is (= (gr/calc gr-ha '{a 1, b 1}) '{s 0, cout 1}))
  )

(def fa-table '[[0 0 0] [0 0]
                [1 0 0] [1 0]
                [0 1 0] [1 0]
                [1 1 0] [0 1]
                [0 0 1] [1 0]
                [1 0 1] [0 1]
                [0 1 1] [0 1]
                [1 1 1] [1 1]])

(def gr-fa
  (let [gr           (gr/new-graph)
        [gr a b cin] (gr/inputs gr 'a 'b 'cin)
        [gr s cout]  (gr/truth-table gr (partition 2 fa-table) [a b cin])
        gr           (gr/outputs gr [s 's] [cout 'cout])]
    (gr/remove-unrefered-conns gr)))

(deftest test-calc-fa
  (is (= (gr/calc gr-fa '{a 0, b 0, cin 0}) '{s 0, cout 0}))
  (is (= (gr/calc gr-fa '{a 1, b 0, cin 0}) '{s 1, cout 0}))
  (is (= (gr/calc gr-fa '{a 0, b 1, cin 0}) '{s 1, cout 0}))
  (is (= (gr/calc gr-fa '{a 1, b 1, cin 0}) '{s 0, cout 1}))
  (is (= (gr/calc gr-fa '{a 0, b 0, cin 1}) '{s 1, cout 0}))
  (is (= (gr/calc gr-fa '{a 1, b 0, cin 1}) '{s 0, cout 1}))
  (is (= (gr/calc gr-fa '{a 0, b 1, cin 1}) '{s 0, cout 1}))
  (is (= (gr/calc gr-fa '{a 1, b 1, cin 1}) '{s 1, cout 1}))
  )

(deftest test-calc-all
  (is (= (gr/calc-all sel-2in1)
         '[[0] ; [0 0 0]
           [1] ; [1 0 0]
           [0] ; [0 1 0]
           [1] ; [1 1 0]
           [0] ; [0 0 1]
           [0] ; [1 0 1]
           [1] ; [0 1 1]
           [1] ; [1 1 1]
           ]))
  (is (= (gr/calc-all gr-ha)
      ;  cout s
         '[[0 0] ; [0 0]
           [0 1] ; [1 0]
           [0 1] ; [0 1]
           [1 0] ; [1 1]
           ]))
  (is (= (gr/calc-all gr-fa)
      ;  cout s
         '[[0 0] ; [0 0 0]
           [0 1] ; [1 0 0]
           [0 1] ; [0 1 0]
           [1 0] ; [1 1 0]
           [0 1] ; [0 0 1]
           [1 0] ; [1 0 1]
           [1 0] ; [0 1 1]
           [1 1] ; [1 1 1]
           ])))


;                      ?bn10__
; ?bt1 --------------------|  \
;                          |   |----+
;            +-------------|__/     |    __
;            |   |\?bn4     __      +----\  \      _
; ?bt2 ------+---| >o------|  \           |  |----|_| ?bt15
;                |/        |   |---------/__/
; ?bt0 --------------------|__/            ?bn13
;                            ?bn7
;
; :nodes
;- [n-0   in   a]
;- [n-1   in   b]
;- [n-2   in   c]
;- [n-4   not1  ]
;+ [?bn4  not1  ]
;- [n-7   and2  ]
;+ [?bn7  and2  ]
;- [n-10  and2  ]
;+ [?bn10 and2  ]
;- [n-13  or2   ]
;+ [?bn13 or2   ]
;- [n-15  out  z]
;
; :edges
;- [e-3   [n-2   0] [n-4   0]]
;+ [?be3  ?bt2      [?bn4  0]]
;- [e-5   [n-0   0] [n-7   0]]
;+ [?be5  ?bt0      [?bn7  0]]
;- [e-6   [n-4   0] [n-7   1]]
;+ [?be6  [?bn4  0] [?bn7  1]]
;- [e-8   [n-1   0] [n-10  0]]
;+ [?be8  ?bt1      [?bn10 0]]
;- [e-9   [n-2   0] [n-10  1]]
;+ [?be9  ?bt2      [?bn10 1]]
;- [e-11  [n-7   0] [n-13  0]]
;+ [?be11 [?bn7  0] [?bn13 0]]
;- [e-12  [n-10  0] [n-13  1]]
;+ [?be12 [?bn10 0] [?bn13 1]]
;- [e-14  [n-13  0] [n-15  0]]
;+ [?be14 [?bn13 0] ?bt15    ]
(def sel-2in1-ptn
  {:nodes '[[?bn4  not1  ]
            [?bn7  and2  ]
            [?bn10 and2  ]
            [?bn13 or2   ]
            ]
   :edges '[[?be3  ?bt2      [?bn4  0]]
            [?be5  ?bt0      [?bn7  0]]
            [?be6  [?bn4  0] [?bn7  1]]
            [?be8  ?bt1      [?bn10 0]]
            [?be9  ?bt2      [?bn10 1]]
            [?be11 [?bn7  0] [?bn13 0]]
            [?be12 [?bn10 0] [?bn13 1]]
            [?be14 [?bn13 0] ?bt15    ]
            ]})

(deftest test-match1-sel-2in1
  (is (= (gr/match1 sel-2in1
                    (:nodes sel-2in1-ptn)
                    (:edges sel-2in1-ptn))
         '{ ?bn4    n-4
          , ?bn7    n-7
          , ?bn10   n-10
          , ?bn13   n-13
          , ?be3    e-3
          , ?bt0    [n-0 0]
          , ?be5    e-5
          , ?be6    e-6
          , ?be8    e-8
          , ?bt1    [n-1 0]
          , ?be9    e-9
          , ?bt2    [n-2 0]
          , ?be11   e-11
          , ?be12   e-12
          , ?be14   e-14
          , ?bt15   [n-15 0]
          })))

; before                        _
;                        +-----|_|
;  _            |\       |      _
; |_|-----------| >o-----+-----|_|
;               |/
;
; after         |\              _
;         +-----| >o-----------|_|
;         |     |/
;         |
;  _      |     |\              _
; |_|-----+-----| >o-----------|_|
;               |/
(deftest test-replace1
  (let [gr (struct gr/graph 9
                   '[[n-0 in a]
                     [n-2 not1]
                     [n-4 out y]
                     [n-6 out z]
                     ]
                   '[[e-1 [n-0 0] [n-2 0]]
                     [e-3 [n-2 0] [n-4 0]]
                     [e-5 [n-2 0] [n-6 0]]
                     ])
        bfr {:nodes '[[?bn2 not1]]
             :edges '[[?be1 ?bt0     [?bn2 0]]
                      [?be3 [?bn2 0] ?bt4    ]
                      [?be5 [?bn2 0] ?bt6    ]
                      ]}
        cut '[?be3 ?be5]
        aft {:nodes '[[?an7 not1]
                      [?an8 not1]]
             :edges '[[?ae9  ?bt0     [?an7 0]]
                      [?ae10 ?bt0     [?an8 0]]
                      [?ae11 [?an7 0] ?bt4    ]
                      [?ae12 [?an8 0] ?bt6    ]
                      ]}
        aft-gr (gr/replace1 gr {:bfr bfr, :cut cut, :aft aft})]
    (is (not= um/fail
              (um/unify (:edges aft-gr)
                        '[[?e2 [n-0 0] [?n0 0]]
                          [?e3 [n-0 0] [?n1 0]]
                          [?e4 [?n0 0] [n-4 0]]
                          [?e5 [?n1 0] [n-6 0]]
                          ]
                        (um/unify (:nodes aft-gr)
                                  '[[n-0 in a]
                                    [n-4 out y]
                                    [n-6 out z]
                                    [?n0 not1]
                                    [?n1 not1]
                                    ]))))))

; before
;
;  _      |\        |\        _
; |_|-----| >o------| >o-----|_|
;         |/        |/
;
; after
;  _                          _
; |_|------------------------|_|
;
(deftest test-replace
  (let [gr (struct gr/graph 7
                   '[[n-0 in a]
                     [n-2 not1]
                     [n-4 not1]
                     [n-6 out z]
                     ]
                   '[[e-1 [n-0 0] [n-2 0]]
                     [e-3 [n-2 0] [n-4 0]]
                     [e-5 [n-4 0] [n-6 0]]
                     ])
        bfr {:nodes '[[?bn2 not1]
                      [?bn4 not1]
                      ]
             :edges '[[?be1 ?bt0     [?bn2 0]]
                      [?be3 [?bn2 0] [?bn4 0]]
                      [?be5 [?bn4 0] ?bt6    ]
                      ]}
        cut '[?be5]
        aft {:nodes '[]
             :edges '[[?ae1 ?bt0 ?bt6]]}
        aft-gr (gr/replace gr {:bfr bfr, :cut cut, :aft aft})]
    (is (not= um/fail
              (um/unify (:edges aft-gr)
                        '[[?e0 [n-0 0] [n-6 0]]
                          ]
                        (um/unify (:nodes aft-gr)
                                  '[[n-0 in a]
                                    [n-6 out z]
                                    ])))))
  (let [gr (struct gr/graph 7
                   '[[n-0 in a]
                     [n-2 not1]
                     [n-4 out z]
                     ]
                   '[[e-1 [n-0 0] [n-2 0]]
                     [e-3 [n-2 0] [n-4 0]]
                     ])
        bfr {:nodes '[[?bn2 not1]
                      ]
             :edges '[[?be1 ?bt0     [?bn2 0]]
                      [?be3 [?bn2 0] ?bt4    ]
                      ]}
        cut '[?be3]
        aft {:nodes '[[?an2 not1]
                      ]
             :edges '[[?ae1 ?bt0     [?an2 0]]
                      [?ae3 [?an2 0] ?bt4    ]
                      ]}
        aft-gr (gr/replace gr {:bfr bfr, :cut cut, :aft aft} 100)]
    (is (not= um/fail
              (um/unify (:edges aft-gr)
                        '[[?e1 [n-0 0] [?n2 0]]
                          [?e3 [?n2 0] [n-4 0]]
                          ]
                        (um/unify (:nodes aft-gr)
                                  '[[n-0 in a]
                                    [n-4 out z]
                                    [?n2 not1]
                                    ]))))))

;
; gr-dist-bfr
; (will be before-pattern)
;    _           __
; c |_|---------|  \
;               |   |----+
;          +----|__/     |    __
;    _     |     __      +----\  \      _
; a |_|----+----|  \           |  |----|_| z
;    _          |   |---------/__/
; b |_|---------|__/
;
(def gr-dist-bfr
     (struct gr/graph 14
             '[[n-0  in a]
               [n-1  in b]
               [n-2  in c]
               [n-5  and2]
               [n-8  and2]
               [n-11 or2]
               [n-13 out z]
               ]
             '[[e-3  [n-0  0] [n-5  0]]
               [e-4  [n-1  0] [n-5  1]]
               [e-6  [n-0  0] [n-8  0]]
               [e-7  [n-2  0] [n-8  1]]
               [e-9  [n-5  0] [n-11 0]]
               [e-10 [n-8  0] [n-11 1]]
               [e-12 [n-11 0] [n-13 0]]
               ]))

(deftest gr-to-bfr-ptn-nodes
  (is (= (gr/gr-to-bfr-ptn-nodes gr-dist-bfr)
         '[[?cin-a ?*cin-a]
           [?cin-b ?*cin-b]
           [?cin-c ?*cin-c]
           [?bn-5  and2]
           [?bn-8  and2]
           [?bn-11 or2]
           ])))

(deftest gr-to-bfr-ptn-edges
  (is (= (gr/gr-to-bfr-ptn-edges gr-dist-bfr)
         '[[?be-3  [?cin-a ?cin-a-0] [?bn-5   0        ]]
           [?be-4  [?cin-b ?cin-b-0] [?bn-5   1        ]]
           [?be-6  [?cin-a ?cin-a-0] [?bn-8   0        ]]
           [?be-7  [?cin-c ?cin-c-0] [?bn-8   1        ]]
           [?be-9  [?bn-5  0       ] [?bn-11  0        ]]
           [?be-10 [?bn-8  0       ] [?bn-11  1        ]]
           [?be-12 [?bn-11 0       ] [?cout-z ?cout-z-0]]
           ])))

(deftest gr-to-ptn-cut
  (is (= (gr/gr-to-ptn-cut gr-dist-bfr)
         '[?be-12])))

;
; gr-dist-aft
; (will be after-pattern)
;    _      __
; c |_|----\  \
;    _      |  |----+     __
; b |_|----/__/     +----|  \      _
;    _                   |   |----|_| z
; a |_|------------------|__/
;
(def gr-dist-aft
     (struct gr/graph 11
             '[[n-0  in a]
               [n-1  in b]
               [n-2  in c]
               [n-5  or2]
               [n-8  and2]
               [n-10 out z]
               ]
             '[[e-3 [n-1 0] [n-5  0]]
               [e-4 [n-2 0] [n-5  1]]
               [e-6 [n-0 0] [n-8  0]]
               [e-7 [n-5 0] [n-8  1]]
               [e-9 [n-8 0] [n-10 0]]
               ]))

(deftest gr-to-aft-ptn-nodes
  (is (= (gr/gr-to-aft-ptn-nodes gr-dist-aft)
         '[[?an-5 or2]
           [?an-8 and2]
           ])))

(deftest gr-to-aft-ptn-edges
  (is (= (gr/gr-to-aft-ptn-edges gr-dist-aft)
         '[[?ae-3 [?cin-b ?cin-b-0] [?an-5   0        ]]
           [?ae-4 [?cin-c ?cin-c-0] [?an-5   1        ]]
           [?ae-6 [?cin-a ?cin-a-0] [?an-8   0        ]]
           [?ae-7 [?an-5  0       ] [?an-8   1        ]]
           [?ae-9 [?an-8  0       ] [?cout-z ?cout-z-0]]
           ])))

(run-tests)

