/*
 * Copyright (C) 2007 uguu at users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.deployer;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileLock;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

/**
 * <p>
 * ファイルを監視し、アーカイブ・ファイルが配置されたときに作業ディレクトリに展開し、イベントを通知します。
 * </p>
 * <p>
 * {@link #monitor()}メソッドの呼び出しでファイルを一回監視することが出来ます。ファイルを定期的に監視する場合、定期的に{@link #monitor()}メソッドを呼び出してください。
 * </p>
 * 
 * @author $Author$
 * @version $Rev$ $Date$
 */
public final class Deployer {

    /**
     * <p>
     * リスナーのリスト。
     * </p>
     */
    private List<DeployerListener> _listenerList = new ArrayList<DeployerListener>();

    /**
     * <p>
     * 配置ディレクトリ。
     * </p>
     */
    private File                   _deployDirectory;

    /**
     * <p>
     * アーカイブ・ファイルであると認識するファイルのパターン。
     * </p>
     */
    private Pattern                _filePattern;

    /**
     * <p>
     * 作業ディレクトリ。
     * </p>
     */
    private File                   _workDirectory;

    /**
     * <p>
     * ファイル監視。
     * </p>
     */
    private FileMonitor            _fileMonitor;

    /**
     * <p>
     * インスタンスを初期化します。
     * </p>
     * 
     * @param deployDirectory
     *            配置ディレクトリ。ここにアーカイブ・ファイルを配置すると、{@link Deployer}クラスが認識し、作業ディレクトリに展開します。<br>
     *            nullの場合、{@link IllegalArgumentException}例外をスローします。<br>
     *            ディレクトリではない場合、{@link IllegalArgumentException}例外をスローします。
     * @param filePattern
     *            アーカイブ・ファイルであると認識するファイルのパターン。絶対パスと比較されます。<br>
     *            nullの場合、{@link IllegalArgumentException}例外をスローします。
     * @param workDirectory
     *            作業ディレクトリ。アーカイブ・ファイルはここに展開されます。<br>
     *            nullの場合、{@link IllegalArgumentException}例外をスローします。<br>
     *            ディレクトリではない場合、{@link IllegalArgumentException}例外をスローします。
     */
    public Deployer(File deployDirectory, Pattern filePattern, File workDirectory) {
        if (deployDirectory == null) {
            throw new IllegalArgumentException(Message.argumentIsNull("deployDirectory"));
        }
        if (!deployDirectory.isDirectory()) {
            throw new IllegalArgumentException(Message.argumentIsNotDirectory("deployDirectory"));
        }
        if (filePattern == null) {
            throw new IllegalArgumentException(Message.argumentIsNull("filePattern"));
        }
        if (workDirectory == null) {
            throw new IllegalArgumentException(Message.argumentIsNull("workDirectory"));
        }
        if (!workDirectory.isDirectory()) {
            throw new IllegalArgumentException(Message.argumentIsNotDirectory("workDirectory"));
        }

        this._deployDirectory = deployDirectory;
        this._filePattern = filePattern;
        this._workDirectory = workDirectory;

        // ディレクトリの監視を開始する。
        this._fileMonitor = new FileMonitor(this._deployDirectory, this._filePattern);
        this._fileMonitor.addListener(new FileMonitorListenerImpl());
    }

    /**
     * <p>
     * アーカイブ・ファイルの配置、配置解除のイベントが通知されるリスナーを追加します。
     * </p>
     * <p>
     * このメソッドはスレッドセーフです。
     * </p>
     * 
     * @param listener
     *            イベントが通知されるリスナー。<br>
     *            nullの場合、{@link IllegalArgumentException}例外をスローします。
     */
    public void addListener(DeployerListener listener) {
        if (listener == null) {
            throw new IllegalArgumentException(Message.argumentIsNull("listener"));
        }
        synchronized (this._listenerList) {
            this._listenerList.add(listener);
        }
    }

    /**
     * <p>
     * 登録されているリスナーを削除します。
     * </p>
     * <p>
     * このメソッドはスレッドセーフです。
     * </p>
     * 
     * @param listener
     *            削除するリスナー。
     */
    public void removeListener(DeployerListener listener) {
        synchronized (this._listenerList) {
            this._listenerList.remove(listener);
        }
    }

    /**
     * <p>
     * アーカイブ・ファイルを監視し、配置、配置解除を行い、リスナーにイベントを通知します。
     * </p>
     * 
     * @throws FileMonitorFailException
     *             ファイルの監視に失敗した場合。
     */
    public void monitor() throws FileMonitorFailException {
        this._fileMonitor.monitor();
    }

    /**
     * <p>
     * デプロイします。
     * </p>
     * 
     * @param file
     *            アーカイブ・ファイル。
     * @throws IOException
     *             入出力に失敗した場合。
     */
    private void deploy(File file) throws IOException {
        // イベントを通知する。
        synchronized (this._listenerList) {
            for (DeployerListener l : this._listenerList) {
                l.deployStart(this, file);
            }
        }
        // デプロイ先のディレクトリを削除する。
        File destDir = this.getDestDirectory(file, this._deployDirectory, this._workDirectory);
        if (destDir.exists()) {
            this.delete(destDir);
            this.deleteWithParent(destDir.getParentFile());
        }
        // デプロイ先のディレクトリを作成する。
        if (!destDir.mkdirs()) {
            throw new DirectoryCreateFailException(destDir);
        }
        // デプロイ先に展開する。
        this.decompress(file, destDir);
        // イベントを通知する。
        synchronized (this._listenerList) {
            for (DeployerListener l : this._listenerList) {
                l.deployEnd(this, file, destDir);
            }
        }
    }

    /**
     * <p>
     * アンデプロイします。
     * </p>
     * 
     * @param file
     *            アーカイブ・ファイル。
     * @throws IOException
     *             入出力に失敗した場合。
     */
    private void undeploy(File file) throws IOException {
        File destDir = this.getDestDirectory(file, this._deployDirectory, this._workDirectory);
        // イベントを通知する。
        synchronized (this._listenerList) {
            for (DeployerListener l : this._listenerList) {
                l.undeployStart(this, file, destDir);
            }
        }
        // デプロイ先のディレクトリを削除する。
        this.delete(destDir);
        this.deleteWithParent(destDir.getParentFile());
        // イベントを通知する。
        synchronized (this._listenerList) {
            for (DeployerListener l : this._listenerList) {
                l.undeployEnd(this, file);
            }
        }
    }

    /**
     * <p>
     * アーカイブ・ファイルを展開します。
     * </p>
     * 
     * @param file
     *            展開するアーカイブ・ファイル。
     * @param destDir
     *            展開先ディレクトリ。
     * @throws IOException
     *             入出力に失敗した場合。
     */
    private void decompress(File file, File destDir) throws IOException {
        FileInputStream fileIn = null;
        while (fileIn == null) {
            try {
                fileIn = new FileInputStream(file);
            } catch (FileNotFoundException e) {
                fileIn = null;
                try {
                    Thread.sleep(Consts.SLEEP_UNIT_TIME);
                } catch (InterruptedException e1) {
                    e1.printStackTrace();
                }
            }
        }
        try {
            ZipInputStream zipIn = new ZipInputStream(fileIn);
            try {
                FileLock lock = fileIn.getChannel().lock(0, Long.MAX_VALUE, true);
                try {
                    ZipEntry zipEntry;
                    while ((zipEntry = zipIn.getNextEntry()) != null) {
                        if (!zipEntry.isDirectory()) {
                            File f = new File(destDir, zipEntry.getName().replace('/', File.separatorChar));
                            // イベントを通知します。
                            synchronized (this._listenerList) {
                                for (DeployerListener l : this._listenerList) {
                                    l.deployFile(this, file, destDir, f);
                                }
                            }
                            if (!f.createNewFile()) {
                                throw new FileCreateFailException(f);
                            }
                            // ファイルを展開します。
                            FileOutputStream fileOut = new FileOutputStream(f);
                            try {
                                byte[] buf = new byte[Consts.BUFFER_LENGTH];
                                int len;
                                while ((len = zipIn.read(buf)) != -1) {
                                    fileOut.write(buf, 0, len);
                                }
                            } finally {
                                fileOut.close();
                            }
                        } else {
                            File dir = new File(destDir, zipEntry.getName().replace('/', File.separatorChar));
                            if (!dir.mkdirs()) {
                                throw new DirectoryCreateFailException(dir);
                            }
                        }
                    }
                } finally {
                    lock.release();
                }
            } finally {
                zipIn.close();
            }
        } finally {
            fileIn.close();
        }
    }

    /**
     * <p>
     * ディレクトリを完全に削除します。
     * </p>
     * 
     * @param dir
     *            削除するディレクトリ。
     * @throws IOException
     *             削除に失敗した場合。
     */
    private void delete(File dir) throws IOException {
        for (File f : dir.listFiles()) {
            if (!f.isDirectory()) {
                if (!f.delete()) {
                    throw new FileDeleteFailException(f);
                }
            } else {
                this.delete(f);
            }
        }
        if (!dir.delete()) {
            throw new DirectoryDeleteFailException(dir);
        }
    }

    /**
     * <p>
     * 親ディレクトリが空であるならば、作業ディレクトリまで遡って削除します。
     * </p>
     * 
     * @param dir
     *            削除するディレクトリ。
     * @throws IOException
     *             削除に失敗した場合。
     */
    private void deleteWithParent(File dir) throws IOException {
        if (this._workDirectory.getAbsolutePath().equals(dir.getAbsolutePath())) {
            return;
        }
        if (dir.listFiles().length > 0) {
            return;
        }
        File parentDir = dir.getParentFile();
        if (!dir.delete()) {
            throw new IOException(Message.directoryDeleteFail(dir.getAbsolutePath()));
        }
        this.deleteWithParent(parentDir);
    }

    /**
     * <p>
     * アーカイブ・ファイルの展開先ディレクトリを算出します。アーカイブ・ファイルの展開先は配置ディレクトリからの相対パスであるため、少し操作が必要になります。
     * </p>
     * 
     * @param archiveFile
     *            展開するアーカイブ・ファイル。
     * @param deployDirectory
     *            配置ディレクトリ。
     * @param workDirectory
     *            作業ディレクトリ。
     * @return アーカイブ・ファイルの展開先ディレクトリ。
     */
    private File getDestDirectory(File archiveFile, File deployDirectory, File workDirectory) {
        String name = archiveFile.getAbsolutePath().substring(deployDirectory.getAbsolutePath().length());
        File dir = new File(workDirectory, name);
        return dir;
    }

    /**
     * <p>
     * 配置ディレクトリを監視します。
     * </p>
     */
    private class FileMonitorListenerImpl implements FileMonitorListener {

        /**
         * <p>
         * インスタンスを初期化します。
         * </p>
         */
        public FileMonitorListenerImpl() {
        }

        /**
         * {@inheritDoc}
         */
        public void create(FileMonitor monitor, File file) {
            try {
                Deployer.this.deploy(file);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        /**
         * {@inheritDoc}
         */
        public void delete(FileMonitor monitor, File file) {
            try {
                Deployer.this.undeploy(file);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        /**
         * {@inheritDoc}
         */
        public void update(FileMonitor monitor, File file) {
            try {
                Deployer.this.undeploy(file);
                Deployer.this.deploy(file);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

    }

}
