/**
 * \brief C++ wrapper of CoolRain templat engine.
 *
 */
#if !defined(COORAIN__COOLRAIN__HXX__)
#define COOLRAIN__COOLRAIN__HXX__

#include <boost/noncopyable.hxx>
#include <stdarg.h>
#include <CoolRain/Coolrain.h>


namespace CoolRain {

	class RefString : public coolrain_refstring_t {
	public:
		inline RefString() { begin = NULL; end = NULL; }
		inline RefString(char const* from, char const* to) { begin = from; end = to; }
		inline RefString(char const* from, size_t length) { begin = from; end = from + length; }

		inline void fix() { coolrain_refstring_fix(this); }
		inline size_t length() const { return coolrain_refstring_length(this); }
		inline bool equal_(char const* rbegin, char const* rend) { return coolrain_refstring_equal_(this, rbegin, rend); }
		inline bool equal(RefString const& r) { return coolrain_refstring_equal(this, &r); }
		inline unsigned int hash() const { return coolrain_refstring_hash(this); }
	};




	/**
	 *
	 *
	 */

	/**
	 */
	class Writer :
		public coolrain_writer,
		public boost::noncopyable {
	
	protected:
		inline Writer() { }

	public:
		inline Writer(char *buffer, size_t length) {
			coolrain_writer_initialize(this, buffer, length);
		}

		inline ~Writer() { coolrain_writer_destroy(this); }


		inline int chain(Writer *next_writer) { return coolrain_writer_chain(this, next_writer); }
		inline int write(char const *buffer, size_t length) { return coolrain_writer_write(this, buffer, length); }
		inline int flush_nonlock() { return coolrain_writer_flush_nonlock(this); }
		inline int flush() { return coolrain_writer_flush(this); }

		inline size_t used_length() { return coolrain_writer_used_length(this); }
		inline size_t unused_length() { return coolrain_writer_unused_length(this); }

	protected:
		inline int drop(size_t length) { return coolrain_writer_drop(this, length); }
		inline int reset() { return coolrain_writer_reset(this); }
		inline char *reserve(size_t length) { return coolrain_writer_reserve(this, length); }
	};

	/**
	 */
	class FDWriter : public Writer {
	protected:
		inline FDWriter() { }
	
	public:
		inline FDWriter(int fd, bool autoclose = false) {
			coolrain_fd_writer_initialize(this, fd, autoclose);
		}
	};

#  if HAVE_FCGI_H
	/**
	 *
	 */
	class FCGIWriter : public Writer {
	protected:
		inline FCGIWriter() { }

	public:
		FCGIWriter(FCGX_Stream *stream) {
			coolrain_FCGI_writer_initialize(&this, stream);
		}
	};
#endif



	/**
	 *
	 *
	 */

	typedef coolrain_tag_handler_t	TagHandler;
	typedef coolrain_tag_desc	TagDesc;

	typedef coolrain_filter_handler_t FilterHandler;
	typedef coolrain_filter_desc	FilterDesc;

	/**
	 */
	class Tagset :
		public coolrain_tagset,
    		public boost::noncopyable {
	public:
		inline Tagset() { coolrain_tagset_initialize(this); }
		inline ~Tagset() { coolrain_tagset_destroy(this); }

		inline int addTag(char const* name, TagHandler handler, unsigned int flags) {
			return coolrain_tagset_add_tag(this, name, handler, flags);
		}
		inline int registerTags(TagDesc* table) { return coolrain_tagset_register_tags(this, table); }
		inline TagDesc* lookupTag(RefString const& name) { return coolrain_tagset_lookup_tag(this, &name); }


		inline int addFilter(char const* name, FilterHandler handler, unsigned int flags) {
			return coolrain_stash_add_filter(this, name, handler, flags);
		}
		inline int registerFilters(FilterDesc* table) { coolrain_tagset_register_filters(this, table); }
		inline FilterDesc* lookupFilter(RefString const& name) { return coolrain_tagset_lookup_filter(this, &name);
	};



	/**
	 *
	 */
	class Variant :
		public coolrain_variant {
	public:
		enum {
			TYPE_NULL	= COOLRAIN_VARIANT_TYPE_NULL,
			TYPE_INT	= COOLRAIN_VARIANT_TYPE_INT,
			TYPE_DOUBLE	= COOLRAIN_VARIANT_TYPE_DOUBLE,
			TYPE_REFSTRING	= COOLRAIN_VARIANT_TYPE_REFSTRING,
			TYPE_STATIC_STRING = COOLRAIN_VARIANT_TYPE_STATIC_STRING,
			TYPE_DYNAMIC_STRING = COOLRAIN_VARIANT_TYPE_DYNAMIC_STRING,
			TYPE_POINTER	= COOLRAIN_VARIANT_TYPE_POINTER,
		};

		inline Variant() { coolrain_variant_initialize(this); }
		inline ~Variant() { coolrain_variant_destroy(this); }

		inline Variant(Variant const& r) {
			coolrain_variant_initialize(this);
			coolrain_variant_copy(this, &r);
		}

		inline Variant& operator=(Variant const&r) {
			coolrain_variant_copy(&r);
			return *this;
		}


		inline void unset() { coolrain_variant_unset(this); }

		inline int getInt() const { coolrain_variant_get_int(this); }
		inline void setInt(int new_value) { coolrain_variant_set_int(this, new_value); }

		inline double getDouble() const { return coolrain_variant_get_double(this); }
		inline void setDouble(double new_value) { coolrain_variant_set_double(this, new_value); }

		inline RefString const* getRefString() const { return dynamic_cast<RefString const *>(coolrain_variant_get_refstring(this)); }
		inline void setRefString(RefString const& new_value) { coolrain_variant_set_refstring(this, &new_value); }

		inline char const* getString() const { return coolrain_variant_get_string(this); }
		inline void setStaticString(char const* new_value) { return coolrain_variant_set_static_string(this, new_value); }
		inline void setDynamicString(char const* new_value) { return coolrain_variant_set_dynamic_string(this, new_value); }

		inline void* getPointer() const { coolrain_variant_get_pointer(this); }
		inline void  setPointer(void *new_value, void (*destroy_handler)(void*)) { coolrain_variant_set_pointer(this, new_value, destroy_handler); }

		inline int write(Writer* writer) { return coolrain_variant_write(this, writer); }
	};



	typedef coolrain_stash_value_list StashValueList;

	/**
	 */
	class Stash :
		public coolrain_stash,
		public boost::noncopyable {
	public:
		inline Stash() { coolrain_stash_initialize(this); }
		inline ~Stash() { coolrain_stash_destroy(this); }

		inline Variant const* get(RefString const& key) const { return coolrain_stash_get_value(this, &key); }
		inline Variant const* get_(char const *key) const { return coolrain_stash_get_value_(this, key); }

		inline void set(StashValueList *values, size_t count) {
			coolrain_stash_set_values(this, values, count);
		}
		inline void restore(StashValueList *values, size_t count) {
			coolrain_stash_restore(this, values, count);
		}
	};



	/**
	 */
	typedef coolrain_template_error_handler_t TemplateErrorHandler;

	class EvalContext : 
		public coolrain_eval_context
	{
	public:
		inline EvalContext() {}
		inline EvalContext(EvalContext const& r) { *this = r; }

		inline EvalContext& operator=(EvalContext const& r) {
			coolrain_eval_context_copy(this, &r);
		}
	};




	/**
	 */
	class Template :
		public coolrain_template,
		public boost::noncopyable {
	public:
		inline Template(Tagset *tagset) { coolrain_template_initialize(this, tagset); }
		inline ~Template() { coolrain_template_destroy(this); }

		inline int clear() { return coolrain_template_clear(this); }

		inline int setErrorHandler(TemplateErrorHandler handler, void *user_data) {
			return coolrain_template_set_error_handler(this, handler, user_data);
		}
		inline int raiseError(char const* fmt, ...) {
			va_list va;
			va_start(va, fmt);
			return coolrain_template_vraise_error(this, fmt, va);
		}
		inline int vraiseError(char const *fmt, va_list va) {
			return coolrain_template_vraise_error(this, fmt, va);
		}

		inline int compile(char const *path) { return coolrain_template_compile(this, path); }
		inline int save(char const *path) { return coolrain_template_save(this, path); }
		inline int load(char const *path) { return coolrain_template_load(this, path); }

		inline int parse(
			char const*	name,
			char const*	source,
			size_t		length,
			void		(*destroy_callback)(Template*),
			void		*destroy_data)
		{
		    return coolrain_template_parse(this, name, source, length, destroy_callback, destroy_data);
		}

		inline int eval_block(EvalContext const& cntx) { return coolrain_eval_block(this, &cntx); }
		inline int eval_content(EvalContext const& cntx) { return coolrain_eval_content(this, &cntx); }
		inline int run(Stash* stash, Writer* writer) { return coolrain_template_run(this, stash, writer); }
	};

}


#endif

