/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.validator;

import java.util.Properties;

import jp.co.fujitsu.reffi.client.nexaweb.parser.ElementValue;
import jp.co.fujitsu.reffi.client.nexaweb.parser.ElementValues;
import jp.co.fujitsu.reffi.common.util.ResourceUtil;

import com.nexaweb.plugin.validation.exceptions.ValidationException;
import com.nexaweb.xml.Element;


/**
 * <p>[概 要] </p>
 * バイト長範囲バリデータクラスです。
 * 
 * <p>[詳 細] </p>
 * 
 * <p>[備 考] </p>
 *
 * <b>使用例）</b>
 * 
 * BaseAction継承クラス#validatorsメソッドで以下のように使用します。
 * <PRE>
 *	&#064;Override
 *	protected void validators(List<CustomValidator> validators) {
 *		// バリデーション対象エレメントを引数にしてインスタンス生成
 *		ByteLengthValidator validator = 
 *			new ByteLengthValidator(getElementByName("target"));
 *		// 独自のエラーメッセージを使う場合 
 *		validator.setErrorMessage("29bytesまでの制限があります。");
 *		// バイト長範囲（0bytes～29bytes）を設定
 *		validator.setBytesRange(0,29);
 *
 *		// 引数validatorsに追加
 *		validators.add(validator);
 *	}
 * 
 * </PRE>
 *
 * <p>Copyright (c) 2008-2010 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public class ByteLengthValidator extends CustomValidator{

	/** エラーメッセージ取得ソースです。 */
	private final String MESSAGE_RESOURCE = "jp.co.fujitsu.reffi.common.exception.corelogic_message";

	/** 最大バイト長です。 */
	private int max;

	/** 最小バイト長です。 */
	private int min;

	/**
	 * <p>[概 要] </p>
	 * 許容範囲のバイト長を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * minフィールドに引数minを、maxフィールドに引数maxを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param min 最小文字長数
	 * @param max 最大文字長数
	 */
	public void setBytesRange(int min, int max) {
		this.min = min;
		this.max = max;
	}

	/**
	 * <p>[概 要] </p>
	 * コンストラクタです。
	 *   
	 * <p>[詳 細] </p>
	 * 引数element付きでsuper()を呼び出します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param element
	 */
	public ByteLengthValidator(Element element) {
		super(element);
	}

	/**
	 * <p>[概 要] </p>
	 * エラー文言登録メソッドです。
	 *  
	 * <p>[詳 細] </p>
	 * jp.co.fujitsu.reffi.shared.exception.corelogic_message.properties
	 * から対応するエラーメッセージを取得して返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return エラー時メッセージ
	 */
	@Override
	public String registerErrorMessage() {
		Properties messages = ResourceUtil.instance.asProperties(MESSAGE_RESOURCE);
		return messages.getProperty("EFC2004");
	}

	/**
	 * <p>[概 要] </p>
	 * バイト長バリデーションを行います。
	 *  
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param value エレメント値
	 * @return エラー有：false、エラー無：true
	 */
	@Override
	protected boolean validate(Object value) throws ValidationException {
		ElementValues values = (ElementValues) value;

		int max = this.max;
		int min = this.min;
		for (int i = 0; i < values.size(); i++) {
			ElementValue elementValue = values.getElementValue(i);
			String v = elementValue.getValue();
			if (v.getBytes().length < min || v.getBytes().length > max) {
				return false;
			}
		}
		return true;
	}
}
